﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaIntegration.VistA;
using BMS.Utils;
using BMS.VistaIntegration.Data;

namespace BMS.VistaIntegration.ShareSessions
{
    public class VistaSessionFactory : IVistASessionFactory
    {
        public class ClientSession : IVistASession
        {
            private readonly ShareSessionMock shareSession;
            private bool isClosed = false;

            public ClientSession(ShareSessionMock shareSession)
            {
                this.shareSession = shareSession;
                shareSession.IncCountRef();
            }

            public VistASite VistASite
            {
                get { return shareSession.VistASite; }
            }

            public void Open(VistAConnectionInfo connection)
            {
                shareSession.Open(connection);
            }

            public void Close()
            {
                if (isClosed)
                    return;

                isClosed = true;
                shareSession.Close();
            }

            public IVistAQuery MakeQuery()
            {
                return shareSession.MakeQuery();
            }

            public void Dispose()
            {
                Close();
            }


            public bool IsAlive()
            {
                return shareSession.IsAlive();
            }
        }

        public class ShareSessionMock : IVistASession
        {
            private readonly object lockThis = new object();
            private readonly IVistASessionFactory sessionFactory;
            private readonly VistASite site;
            private IVistASession session;
            private bool isLoaded;
            private bool isValid = true;
            private VistAConnectionInfo connection;
            private int countRef = 0;

            public ShareSessionMock(IVistASessionFactory factory, VistASite site)
            {
                this.sessionFactory = factory;
                this.site = site;
            }

            public VistASite VistASite
            {
                get { return site; }
            }

            private static bool SameConnection(VistAConnectionInfo var1, VistAConnectionInfo var2)
            {
                return var1.ConnectionString == var2.ConnectionString &&
                    var1.AccessCode == var2.AccessCode &&
                    var1.VerifyCode == var2.VerifyCode;
            }

            public void Open(VistAConnectionInfo connection)
            {
                lock (lockThis)
                {
                    if (isLoaded)
                        return;

                    this.connection = connection;
                    session = sessionFactory.MakeVistASession(site);
                    session.Open(connection);
                    isLoaded = true;
                    isValid = true;
                }
            }

            public void IncCountRef()
            {
                lock (lockThis)
                {
                    countRef++;
                }
            }

            private void CloseAndReleaseSession()
            {
                session.Close();
                session = null;
                connection = null;
            }

            public void Close()
            {
                bool closeSession;
                lock (lockThis)
                {
                    countRef--;
                    closeSession = !isValid && countRef == 0;
                }
                if (closeSession)
                    CloseAndReleaseSession();
            }

            public IVistAQuery MakeQuery()
            {
                return session.MakeQuery();
            }

            public void Dispose()
            {
                Close();
            }

            public void RemoveFromDictionary()
            {
                bool closeSession;
                lock (lockThis)
                {
                    if (!isValid)
                        return;
                    closeSession = countRef == 0;
                    isValid = false;
                }
                if (closeSession)
                    CloseAndReleaseSession();

            }


            public bool IsAlive()
            {
                return session.IsAlive();
            }
        }

        private readonly Dictionary<string, ShareSessionMock> dictionary = new Dictionary<string, ShareSessionMock>();
        private readonly IVistASessionFactory factory;

        public VistaSessionFactory(IVistASessionFactory factory)
        {
            this.factory = factory;
        }

        public IVistASession MakeVistASession(Data.VistASite site)
        {
            ShareSessionMock sessionMock = null;
            ShareSessionMock oldSessionMock = null;

            lock (dictionary)
            {
                if (!dictionary.TryGetValue(site.Id, out sessionMock))
                {
                    sessionMock = new ShareSessionMock(factory, site);
                    dictionary[site.Id] = sessionMock;
                }
                else
                {
                    bool isAlive = sessionMock.IsAlive();
                    if (!isAlive) 
                    {
                        oldSessionMock = sessionMock;
                        sessionMock = new ShareSessionMock(factory, site);
                        dictionary[site.Id] = sessionMock;
                    }
                    return sessionMock;
                }
            }
            if (oldSessionMock != null)
            {
                try
                {
                    oldSessionMock.Dispose();
                }
                catch
                {
                }
            }
            return new ClientSession(sessionMock);
        }

        public void Clear()
        {
            lock (dictionary)
            {
                dictionary.Values.ForEach(s => s.RemoveFromDictionary());
                dictionary.Clear();
            }
        }

        public DataRetrievalMethod RetrievalMethod
        {
            get { return factory.RetrievalMethod; }
        }
    }
}
