﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using InfoWorld.HL7.ITS.Extension.EIS;

namespace BMS.ServicesWrapper.EIS
{
    /// <summary>
    /// 
    /// </summary>
    public interface IEISWrapper
    {
        #region Bed

        /// <summary>
        /// Get the bed with the specified internal id.
        /// </summary>
        /// <param name="id">the bed Id</param>
        /// <param name="vistaSiteId">the vista site Id</param>
        /// <returns>the bed or null</returns>
        Bed GetBed(II id, II vistaSiteId);

        /// <summary>
        /// Get the bed with the specified ien.
        /// </summary>
        /// <param name="ien">the ien</param>
        /// <param name="ien">the vista site id</param>
        /// <returns>the bed or null</returns>
        Bed GetBed(string ien, II vistaSiteId);

        /// <summary>
        /// Gets all the beds in the specified ward.
        /// </summary>
        /// <param name="wardId">the ward id</param>
        /// <returns>a list of all beds in the ward</returns>
        IList<Bed> GetBedsInWard(II wardId, II vistaSiteId);

        /// <summary>
        /// Gets all the beds in the specified division.
        /// </summary>
        /// <param name="wardId">the division id</param>
        /// <returns>a list of all beds in the division</returns>
        IList<Bed> GetBedsInDivision(II divisionId, II vistaSiteId);

        /// <summary>
        /// Gets the beds by vista.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        List<Bed> GetBedsByVista(II vistaSiteId);

        /// <summary>
        /// Gets the beds by vista trait.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        List<Bed> GetBedsByVistaTrait(II vistaSiteId);

        /// <summary>
        /// Creates a bed.
        /// </summary>
        /// <param name="bed">the bed to create</param>
        void CreateBed(Bed bed);

        /// <summary>
        /// Updates the bed.
        /// </summary>
        /// <param name="bed">the bed to update, identified by internal id or ien</param>
        void UpdateBed(Bed bed);

        /// <summay>
        /// Deletes the bed with the specified id.
        /// </summary>
        /// <param name="id">the internal id of the bed</param>
        void DeleteBed(II id);

        /// <summary>
        /// Deletes the association between bed and division.
        /// </summary>
        /// <param name="bed">The bed.</param>
        /// <param name="divisionId">The division id.</param>
        void DeleteAssociationBetweenBedAndDivision(Bed bed, Division division);

        /// <summary>
        /// Updates the association between bed and division.
        /// </summary>
        /// <param name="bed">The bed.</param>
        /// <param name="oldDivision">The old division.</param>
        /// <param name="newDivision">The new division.</param>
        void UpdateAssociationBetweenBedAndDivision(Bed bed, Division oldDivision, Division newDivision);

        /// <summary>
        /// Gets the beds.
        /// </summary>
        /// <param name="bedIds">The bed ids.</param>
        IList<Bed> GetBeds(List<II> bedIds, II vistaSiteId);

        #endregion

        #region Ward

        /// <summary>
        /// Get all wards for a facility and/or a division.
        /// </summary>
        /// <param name="facilityId">the id of the facility</param>
        /// <param name="divisionId">the id of the division</param>
        /// <returns>a list of wards that satisfy the criteria</returns>
        IList<Ward> GetWards(II divisionId);

        /// <summary>
        /// Gets the wards.
        /// </summary>
        /// <param name="wardIds">The wards ids.</param>
        /// <param name="domain">The domain.</param>
        /// <returns></returns>
        IList<Ward> GetWards(List<II> wardIds);

        /// <summary>
        /// Gets the wards by vista site id.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        IList<Ward> GetWardsByVistaSiteId(II vistaSiteId);

        /// <summary>
        /// Get the ward with the specified Ien.
        /// </summary>
        /// <param name="ien">the ien</param>
        /// <param name="ien">the vista site id</param>
        /// <returns>the ward or null</returns>
        Ward GetWard(string ien, II vistaSiteId);

        /// <summary>
        /// Get the ward with the specified id.
        /// </summary>
        /// <param name="id">the internal id</param>
        /// <returns>the ward or null</returns>
        Ward GetWard(II id);

        /// <summary>
        /// Creates the specified ward.
        /// </summary>
        /// <param name="ward">the ward to save</param> 
        void CreateWard(Ward ward);

        /// <summary>
        /// Updates the specified ward.
        /// </summary>
        /// <param name="ward">the ward to save, identified by internal id or ien</param>
        void UpdateWard(Ward ward);

        /// <summary>
        /// Updates the association beteen ward and division.
        /// </summary>
        /// <param name="ward">The ward.</param>
        /// <param name="oldDiviSion">The old divi sion.</param>
        /// <param name="newDivision">The new division.</param>
        void UpdateAssociationBetweenWardAndDivision(Ward ward, Division oldDivision, Division newDivision);

        /// <summary>
        /// Delete the ward with the specified id.
        /// </summary>
        /// <param name="id">the internal id</param>
        void DeleteWard(II id);

        void DeleteAssociationBetweenWardAndDivision(Ward ward, Division division);

        #endregion

        #region Division

        /// <summary>
        /// Gets all the divisions in the specified facility.
        /// </summary>
        /// <param name="facilityId">the facility id</param>
        /// <param name="withVirtualDivision"></param>
        /// <returns>a list of all divisions in the facility</returns>
        IList<Division> GetDivisions(II facilityId);

        /// <summary>
        /// Get the division with the specified internal id.
        /// </summary>
        /// <param name="id">the division Id</param>
        /// <returns>the division or null</returns>
        Division GetDivision(II id);

        /// <summary>
        /// Finds the division with the specified name in the specified Facility.
        /// </summary>
        /// <param name="facilityId">the facility Id</param>
        /// <param name="name">the name of the division</param>
        /// <returns>the division or null</returns>
        Division FindDivision(II facilityId, string name);

        /// <summary>
        /// Creates a division.
        /// </summary>
        /// <param name="division">the division to create</param>
        void CreateDivision(Division division);

        /// <summary>
        /// Updates the division.
        /// </summary>
        /// <param name="division">the division to update, identified by internal id or ien</param>
        void UpdateDivision(Division division);

        /// <summay>
        /// Deletes the division with the specified id.
        /// </summary>
        /// <param name="id">the internal id of the division</param>
        void DeleteDivision(II id);

        /// <summary>
        /// Gets the divisions for bed.
        /// </summary>
        /// <param name="bedId">The bed id.</param>
        /// <returns></returns>
        IList<Division> GetDivisionsForBed(II bedId, II vistaSiteId);

        #endregion

        #region Facility

        /// <summary>
        /// Gets the facilities.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        IList<Facility> GetFacilities(II vistaSiteId);

        /// <summary>
        /// Gets the facilities.
        /// </summary>
        /// <returns></returns>
        IList<Facility> GetFacilities();

        /// <summary>
        /// Returns the list of facilities. It uses the IIS credential to authenticate in STS/EIS.
        /// </summary>
        /// <returns>All facilities.</returns>
        IList<Facility> GetFacilitiesUsingWindowsAuthentication();
        /// <summary>
        /// Gets the facility with the specified id.
        /// </summary>
        /// <param name="id"> the internal id of the facility</param>
        /// <returns>the facility or null</returns>
        Facility GetFacility(II id);

        /// <summary>
        /// Gets the facility with the specified code.
        /// </summary>
        /// <param name="code">The code.</param>
        /// <returns></returns>
        Facility GetFacilityByCode(string code);

        /// <summary>
        /// Gets the facilities by site number.
        /// </summary>
        /// <param name="siteNumber">The site number.</param>
        /// <returns></returns>
        IList<Facility> GetFacilitiesBySiteNumber(string siteNumber);

        /// <summary>
        /// Gets the facility ids for ward.
        /// </summary>
        /// <param name="wardId">The ward id.</param>
        /// <returns></returns>
        IList<Facility> GetFacilityIdsForWard(II wardId);

        /// <summary>
        /// Updates the facility.
        /// </summary>
        /// <param name="facility">the facility to update, identified by internal id or ien</param>
        void UpdateFacility(Facility facility);

        /// <summary>
        /// Gets the facility for division.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <returns></returns>
        Facility GetFacilityByDivisionId(II divisionId);

        #endregion

        #region Hospital Location

        /// <summary>
        /// Gets the hospital locations.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        IList<HospitalLocation> GetHospitalLocations(II vistaSiteId);

        /// <summary>
        /// Gets the hospital locations by vista DB.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        IList<HospitalLocation> GetHospitalLocationsByVistaDB(II vistaSiteId);

        /// <summary>
        /// Gets the hospital locations by ids.
        /// </summary>
        /// <param name="ids">The ids.</param>
        /// <returns></returns>
        List<HospitalLocation> GetHospitalLocationsByIds(List<II> ids, II vistaSiteId);

        /// <summary>
        /// Gets the hospital locations by ids DB.
        /// </summary>
        /// <param name="ids">The ids.</param>
        /// <returns></returns>
        List<HospitalLocation> GetHospitalLocationsByIdsDB(List<II> ids);

        /// <summary>
        /// Gets the hospital location.
        /// </summary>
        /// <param name="hospitalLocationId">The hospital location id.</param>
        /// <returns></returns>
        HospitalLocation GetHospitalLocation(II id, II vistaSiteId);

        /// <summary>
        /// Gets the hospital location DB.
        /// </summary>
        /// <param name="id">The id.</param>
        /// <returns></returns>
        HospitalLocation GetHospitalLocationDB(II id);

        /// <summary>
        /// Gets the hospital location.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        HospitalLocation GetHospitalLocation(string ien, II vistaSiteId);

        /// <summary>
        /// Creates the hospital location.
        /// </summary>
        /// <param name="hospitalLocation">The hospital location.</param>
        void CreateHospitalLocation(HospitalLocation hospitalLocation);

        /// <summary>
        /// Updates the hospital location.
        /// </summary>
        /// <param name="hospitalLocation">The hospital location.</param>
        void UpdateHospitalLocation(HospitalLocation hospitalLocation);

        /// <summary>
        /// Deletes the hospital location.
        /// </summary>
        /// <param name="id">The id.</param>
        void DeleteHospitalLocation(II id);

        #endregion

        #region Vista Site

        /// <summary>
        /// Gets the vista sites.
        /// </summary>
        /// <param name="visnId">The visn id.</param>
        /// <returns></returns>
        IList<VistaSite> GetVistaSites(II visnId);

        /// <summary>
        /// Gets the vista site.
        /// </summary>
        /// <param name="id">The vista site id.</param>
        /// <returns></returns>
        VistaSite GetVistaSite(II id);

        /// <summary>
        /// Gets the vista site.
        /// </summary>
        /// <param name="id">The vista site code.</param>
        /// <returns></returns>
        VistaSite GetVistaSiteByCode(string code);

        /// <summary>
        /// Gets the vista sites using windows authentication.
        /// </summary>
        /// <returns></returns>
        IList<VistaSite> GetVistaSitesUsingWindowsAuthentication();

        #endregion

        #region Visn

        /// <summary>
        /// Gets all visns with the specified region.
        /// </summary>
        /// <param name="regionId">the internal id of the region</param>
        /// <returns>a list of visns</returns>
        IList<Visn> GetVisns(II regionId);

        /// <summary>
        /// Gets the visn with the specified id.
        /// </summary>
        /// <param name="id"> the internal id of the visn</param>
        /// <returns>the visn or null</returns>
        Visn GetVisn(II id);

        /// <summary>
        /// Returns the list of visns. It uses the IIS credential to authenticate in STS/EIS.
        /// </summary>
        /// <returns>All visns.</returns>
        IList<Visn> GetVisnsUsingWindowsAuthentication();

        /// <summary>
        /// Gets the visn by number.
        /// </summary>
        /// <param name="number">The number.</param>
        /// <returns></returns>
        Visn GetVisnByNumber(string number);

        #endregion

        #region Region

        /// <summary>
        /// Gets the region with the specified id.
        /// </summary>
        /// <param name="id"> the internal id of the region</param>
        /// <returns>the region or null</returns>
        Region GetRegion(II id);

        /// <summary>
        /// Gets the regions.
        /// </summary>
        /// <returns></returns>
        IList<Region> GetRegions();

        #endregion

        #region Patient

        /// <summary>
        /// Retrieves the patient with the specified id
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        Patient GetPatientById(II id, string vistaCode);

        /// <summary>
        /// Retrieves the patients with the specified ids
        /// </summary>
        /// <param name="patientId">The patients' ids.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        IList<Patient> GetPatientsByIds(IList<II> ids, string vistaCode);

        /// <summary>
        /// Retrieves the patient with the specified SSN.
        /// </summary>
        /// <param name="ssn">the SSN</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns>the patient or null</returns>
        Patient GetPatientBySsn(II ssn, string vistaCode);

        /// <summary>
        /// Gets the last name of the patients by.
        /// </summary>
        /// <param name="lastName">The last name.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        IList<Patient> GetPatientsByLastName(string lastName, string vistaCode);

        /// <summary>
        /// Gets the patients by last name initial and last four SSN.
        /// </summary>
        /// <param name="filter">The filter.</param>
        /// <param name="vistaCode">The vista code.</param>
        /// <returns></returns>
        IList<Patient> GetPatientsByLastNameInitialAndLastFourSSN(string filter, string vistaCode);

        /// <summary>
        /// Creates the specified patient.
        /// </summary>
        /// <param name="patient">the patient to save</param> 
        void CreatePatient(Patient patient);

        /// <summary>
        /// Updates the specified patient.
        /// </summary>
        /// <param name="patient">the patient to save, identified by internal id or ien</param> 
        void UpdatePatient(Patient patient);

        /// <summary>
        /// Deletes the patient with the specified id.
        /// </summary>
        /// <param name="id">the id of the patient to delete</param> 
        void DeletePatient(II id);

        #endregion

        #region Person

        /// <summary>
        /// Gets the person.
        /// </summary>
        /// <param name="ien">The ien.</param>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        Person GetMedicalPerson(string ien, II vistaSiteId);

        /// <summary>
        /// Creates the specified person.
        /// </summary>
        /// <param name="patient">the person to save</param> 
        void CreateMedicalPerson(Person person);

        /// <summary>
        /// Updates the person.
        /// </summary>
        /// <param name="person">The person.</param>
        void UpdateMedicalPerson(Person person);

        /// <summary>
        /// Deletes the person.
        /// </summary>
        /// <param name="id">The id.</param>
        void DeleteMedicalPerson(II id);

        /// <summary>
        /// Retrieves the medical persons with the specified ids
        /// </summary>
        /// <param name="patientId">The medical persons' ids.</param>
        /// <returns></returns>
        IList<Person> GetMedicalPersonsByIds(IList<II> ids);


        #endregion

        /// <summary>
        /// Gets all domains.
        /// </summary>
        /// <returns></returns>
        IList<String> GetDomains();

        /// <summary>
        /// Gets the places.
        /// </summary>
        /// <param name="metadata">The metadata.</param>
        /// <param name="parentId">The parent id.</param>
        /// <returns></returns>
        IList<Place> GetPlaces(Place metadata, II parentId);

        /// <summary>
        /// Gets the places.
        /// </summary>
        /// <param name="metadata">The metadata.</param>
        /// <returns></returns>
        IList<Place> GetPlaces(Place metadata);

        /// <summary>
        /// Gets the places.
        /// </summary>
        /// <param name="metadata">The metadata.</param>
        /// <param name="traits">The traits.</param>
        /// <returns></returns>
        IList<Place> GetPlaces(Place metadata, TraitList traits);

        /// <summary>
        /// Gets the entity by id.
        /// </summary>
        /// <param name="place">The place.</param>
        /// <returns></returns>
        Entity GetEntityById(Place place);

        /// <summary>
        /// Gets the entity client.
        /// </summary>
        Proxy.EIS.EntityManagementClient EntityClient { get; }

        #region Cache

        /// <summary>
        /// Clears the cache.
        /// </summary>
        void ClearCache();

        /// <summary>
        /// Fills the cache.
        /// </summary>
        void FillCache();

        /// <summary>
        /// Caches the bed.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheBed(II id);

        /// <summary>
        /// Caches the ward.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheWard(II id);

        /// <summary>
        /// Caches the division.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheDivision(II id);

        /// <summary>
        /// Caches the facility.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheFacility(II id);

        /// <summary>
        /// Caches the vista site.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheVistaSite(II id);

        /// <summary>
        /// Caches the visn.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheVisn(II id);

        /// <summary>
        /// Caches the region.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheRegion(II id);

        /// <summary>
        /// Caches the hospital location.
        /// </summary>
        /// <param name="id">The id.</param>
        void CacheHospitalLocation(II id);

        /// <summary>
        /// Removes the bed.
        /// </summary>
        /// <param name="id">The id.</param>
        void RemoveCacheBed(II id);

        /// <summary>
        /// Removes the ward.
        /// </summary>
        /// <param name="id">The id.</param>
        void RemoveCacheWard(II id);

        /// <summary>
        /// Removes the division.
        /// </summary>
        /// <param name="id">The id.</param>
        void RemoveCacheDivision(II id);

        /// <summary>
        /// Removes the cache hospital location.
        /// </summary>
        /// <param name="id">The id.</param>
        void RemoveCacheHospitalLocation(II id);

        #endregion
    }
}
