﻿using System;
using System.Collections.Generic;
using System.Linq;
using FC = BMS.Facade.Data;
using System.Activities;
using BMS.Facade.Data;
using BMS.Utils;
using BMS.ServicesWrapper.Security;
using System.Configuration;
using BMS.ServicesWrapper.EIS;
using InfoWorld.HL7.ITS;
using BMS.Facade;

namespace BMS.Workflows.CustomActivities
{
    public class SendBedUnavailableNotifications : CodeActivity
    {
        public InArgument<FC.BedUnavailable> BedUnavailable { get; set; }

        protected override void Execute(CodeActivityContext context)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    FC.BedUnavailable bedUnavailable = this.BedUnavailable.Get<FC.BedUnavailable>(context);
                                      
                    //send email notifications only if the bed unavailable is triggered by an out of service event.
                    if (bedUnavailable.Type != null && bedUnavailable.Type.code != Constants.BED_OUT_OF_SERVICE_TYPE_CODE)
                        return;

                    //get TimeZoneInfo of the VistA Site
                    FC.VistASettings vistaSettings = Facade.FacadeManager.ConfigurationInterface.GetVistASettings(bedUnavailable.VistaSite.Id);
                    TimeZoneInfo timeZoneInfo;
                    if (vistaSettings != null && vistaSettings.SiteTimeZone != null)
                        timeZoneInfo = TimeZoneInfo.FindSystemTimeZoneById(vistaSettings.SiteTimeZone.displayName);
                    else
                        timeZoneInfo = TimeZoneInfo.Local;
                    //convert bedUnavailable date time properties to VistA Site TimeZone
                    TimeZoneUtil.ConvertObjectDates(bedUnavailable, true, timeZoneInfo);

                    IList<Division> divisions = EISFactory.InstanceFromWCF.GetDivisionsForBed(bedUnavailable.Bed.Id, bedUnavailable.VistaSite.Id);
                    FC.Bed bed = EISFactory.InstanceFromWCF.GetBed(bedUnavailable.Bed.Id, bedUnavailable.VistaSite.Id);
                    IList<Ward> wards = new List<Ward>();
                    if (bed.WardList != null && bed.WardList.Count > 0)
                        wards = EISFactory.InstanceFromWCF.GetWards(bed.WardList.Select<Ward, II>(wrd => wrd.Id).ToList());

                    if (divisions == null)
                        return;
                    string domainId = SecurityFactory.InstanceWindows.GetCurrentDomain();
                    foreach (Division division in divisions)
                    {
                        FacilitySettings settings = Facade.FacadeManager.ConfigurationInterface.GetFacilitySettings(division.Facility.Id);
                        IList<EventNotification> notifications = Facade.FacadeManager.ConfigurationInterface.GetEventNotification(division.Name, domainId, division.Facility.Id);

                        string from = settings.EventMailSender;
                        string to = string.Empty;

                        foreach (EventNotification notification in notifications)
                        {
                            if (!notification.Name.Equals(Constants.BED_OUT_OF_SERVICE))
                                continue;
                            if (notification.BedControllerNotification.code == Constants.Yes)
                                to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.BedControllerEmailAddress;
                            break;
                        }

                        if (!string.IsNullOrWhiteSpace(to) && !string.IsNullOrWhiteSpace(from))
                        {
                            //compose and send the actual message
                            string subject = "BED OUT OF SERVICE";
                            string bodyFormat = "WARD: {0} " + Environment.NewLine +
                                                "BED: {1} " + Environment.NewLine +
                                                "DATE/TIME: {2} " + Environment.NewLine +
                                                "BY: {3} " + Environment.NewLine +
                                                "REASON: {4} " + Environment.NewLine +
                                                "TYPE: {5}";

                            string body = string.Format(bodyFormat,
                                (wards != null && wards.Count > 0) ? String.Join("; ", wards.Select<Ward, string>(w => w.Name)) : string.Empty,
                                bedUnavailable.Bed.Name,
                                (bedUnavailable.EditedDate ?? bedUnavailable.CreationDate).Value.ToString("MM/dd/yy hh:mm"),
                                string.IsNullOrWhiteSpace(bedUnavailable.EditedBy) ? bedUnavailable.CreatedBy : bedUnavailable.EditedBy,
                                bedUnavailable.Reason != null ? bedUnavailable.Reason.displayName : string.Empty,
                                bedUnavailable.Type != null ? bedUnavailable.Type.displayName : string.Empty
                                );

                            BMS.Utils.SendMail.Execute(from, to, subject, body, ConfigurationManager.AppSettings[Constants.SMTP_HOST], 25, null);
                        }
                    }

                    //convert bedUnavailable date time properties to UTC
                    TimeZoneUtil.ConvertObjectDates(bedUnavailable, false, timeZoneInfo);
                }
                catch (Exception e)
                {
                    Tracer.TraceException(e);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
