﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.EVS;
using InfoWorld.HL7.ITS;
using BMS.Facade;
using System.ServiceModel;
using System.Reflection;
using BMS.Utils;
using BMS.VistaIntegration.Data;
using FC = BMS.Facade.Data;

namespace BMS.VistaWorker2.Writer.Implementation.Concrete
{
    /// <summary>
    /// Abstract base writer for eis entities.
    /// </summary>
    /// <typeparam name="T">The entity</typeparam>
    /// <typeparam name="B">The bms entity.</typeparam>
    public abstract class BaseEisWriter<T, B> : BaseEntityWriter<T>
        where T : class, IEntity
        where B : class , new()
    {

        protected abstract IEntityDal<B> Dal { get; }

        /// <summary>
        /// Gets the BMS entity.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <returns></returns>
        protected virtual B GetBmsEntity(T entity)
        {
            return Dal.Get(entity);
        }

        /// <summary>
        /// Translates the VistA entity to an existing BMS entity.
        /// </summary>
        /// <param name="entity">The VistA entity.</param>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected abstract void TranslateFromVistaToBMS(T entity, B bmsEntity);

        /// <summary>
        /// Inserts the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected virtual void InsertBmsEntity(T vistaEntity, B bmsEntity)
        {
            Dal.Add(vistaEntity, bmsEntity);
        }

        /// <summary>
        /// Updates the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected virtual void UpdateBmsEntity(T vistaEntity, B bmsEntity)
        {
            Dal.Update(vistaEntity, bmsEntity);
        }

        /// <summary>
        /// Deletes the BMS entity.
        /// </summary>
        /// <param name="bmsEntity">The BMS entity.</param>
        protected virtual void DeleteBmsEntity(T vistaEntity, B bmsEntity)
        {
            Dal.Delete(vistaEntity, bmsEntity);
        }


        /// <summary>
        /// Ares the fields equal.
        /// </summary>
        /// <param name="entity">The entity.</param>
        /// <param name="bmsEntity">The BMS entity.</param>
        /// <returns></returns>
        protected abstract bool AreFieldsEqual(T entity, B bmsEntity);

        protected override void OnInsert(T entity)
        {
            OnUpdate(entity);
        }

        public bool UpdateOldEntity(T entity, B bmsEntity)
        {
            LoadData(entity);
            return UpdateEntity(entity, bmsEntity);
        }

        private bool UpdateEntity(T entity, B bmsEntity)
        {
            bool sameFields = AreFieldsEqual(entity, bmsEntity);
            if (sameFields)
                return false;
            TranslateFromVistaToBMS(entity, bmsEntity);
            UpdateBmsEntity(entity, bmsEntity);
            Logger.LogFormat(BmsLogger.Level.Info, "Update {0} with success", entity);
            return true;
        }

        protected virtual void LoadData(T entity)
        {
        }


        protected B MakeNewEntity(T entity)
        {
            B bmsEntity = new B();
            TranslateFromVistaToBMS(entity, bmsEntity);
            InsertBmsEntity(entity, bmsEntity);
            Logger.LogFormat(BmsLogger.Level.Info, "Insert {0} with success", entity);
            return bmsEntity;
        }

        public B InsertOrUpdate(T entity)
        {
            B bmsEntity = GetBmsEntity(entity);
            LoadData(entity);
            bool exists = bmsEntity != null;
            if (exists)
                UpdateEntity(entity, bmsEntity);
            else
                bmsEntity = MakeNewEntity(entity);
            return bmsEntity;
        }

        protected override bool OnUpdate(T entity)
        {
            B bmsEntity = GetBmsEntity(entity);
            LoadData(entity);
            bool exists = bmsEntity != null;
            if (exists)
                return UpdateEntity(entity, bmsEntity);
            else
                MakeNewEntity(entity);
            return true;
        }


        protected override void OnDelete(T entity)
        {

            B bmsEntity = GetBmsEntity(entity);
            if (bmsEntity != null)
            {
                DeleteBmsEntity(entity, bmsEntity);
                Logger.LogFormat(BmsLogger.Level.Info, "Delete {0} with success", entity);
            }
            else
            {
                Logger.LogFormat(BmsLogger.Level.Info, "{0} not found for delete ", entity);
            }
        }



    }
}
