﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.EVS;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.VistaWorker.Exceptions;
using BMS.VistaWorker.Writer.Abstract;
using BMS.VistaWorker.Writer.Concrete.CatchExceptions;
using BMS.Facade;
using System.Configuration;

namespace BMS.VistaWorker.Writer.Concrete
{
    /// <summary>
    /// Abstract base writer.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    abstract class BaseEntityWriter<T> : IEntityWriter<T> where T : IEntity
    {
        private readonly ICatchExceptions _catchException;
        protected VistaSite VistaSite { get; private set; }
        protected TimeZoneInfo VistATimeZone { get; private set; }
        protected BMS.Utils.BmsLogger Logger { get; private set; }

        public BaseEntityWriter(ICatchExceptions catchException)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this._catchException = catchException;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void InitData(VistaSite vistaSite, TimeZoneInfo vistATimeZone, Utils.BmsLogger logger)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this.VistaSite = vistaSite;
                this.VistATimeZone = vistATimeZone;
                this.Logger = logger;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected static IEISWrapper EIS { get { return EISFactory.InstanceWindows; } }

        protected static IEVSWrapper EVS { get { return EVSFactory.InstanceWindows; } }

        protected abstract void OnInsert(T entity);

        protected abstract void OnUpdate(T entity);

        protected abstract void OnDelete(T entity);

        public void Insert(T entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                _catchException.RunAction(() => OnInsert(entity));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void Update(T entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                _catchException.RunAction(() => OnUpdate(entity));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void Delete(T entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                _catchException.RunAction(() => OnDelete(entity));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Converts datetimes properties on entity to UTC.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="entity">The entity.</param>
        protected void ConvertToUtc<E>(E entity) where E : class
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeZoneUtil.ConvertObjectDates(entity, false, VistATimeZone);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


    }
}
