﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Reader.Abstract;
using BMS.VistaWorker.Reader.Concrete;
using BMS.VistaWorker.Reader.Concrete.Entities;
using System.Data.SqlClient;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Exceptions;
using BMS.VistaWorker.Abstract;
using BMS.VistaWorker.Reader.Concrete.Entities.EVS;

namespace BMS.VistaWorker.Reader.LinqToSql
{
    /// <summary>
    /// The repository implementation.
    /// </summary>
    public sealed class Repository : IRepository
    {
        private readonly Type[] _eisEvsTypes = 
        {
            typeof(OrderableItemLog),
            typeof(TreatingSpecialtyLog),
            typeof(TypeOfMovementLog),
            typeof(SpecialtyLog),           
            typeof(HospitalLocationLog),
            typeof(WardLocationLog),
            typeof(RoomBedLog),
            typeof(PatientLog),
            typeof(MedicalCenterDivisionLog)
        };

        private VistaDataContext _dataContext;
        private bool _isOpen = false;
        private readonly String _connectionString;
        private readonly int _bulkSize;
        private DateTime _lastTime;

        /// <summary>
        /// Initializes a new instance of the <see cref="Repository"/> class.
        /// </summary>
        /// <param name="connectionString">The connection string.</param>
        public Repository(String connectionString, int bulkSize)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                this._connectionString = connectionString;
                this._bulkSize = bulkSize;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the specified entity from data source.
        /// </summary>
        /// <typeparam name="T">The entity type</typeparam>
        /// <param name="entity">The entity.</param>
        public void Delete(ILogData entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                CatchExceptionAction(() =>
                {
                    try
                    {
                        _dataContext.GetTable(entity.GetType()).DeleteOnSubmit(entity);
                        _dataContext.SubmitChanges();
                    }
                    catch (InvalidCastException)
                    {
                        if (entity is WorkFlowLog)
                            _dataContext.ExecuteCommand("delete dbo.WORKFLOW_LOG where [ID]={0}", entity.Id);
                    }
                });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates the RetryVistaException.
        /// </summary>
        /// <param name="innerException">The inner exception.</param>
        /// <returns></returns>
        private RetryVistaException CreateRetryVistaException(Exception innerException)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new RetryVistaException("Cannot read from vista db", innerException);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Run an action , catch TimeoutException and SqlException and throw them as RetryVistaException.
        /// </summary>
        /// <param name="action"></param>
        private void CatchExceptionAction(Action action)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    action();
                }
                catch (TimeoutException e)
                {
                    throw CreateRetryVistaException(e);
                }
                catch (SqlException e)
                {
                    throw CreateRetryVistaException(e);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Determines whether argument is not null.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="log">The log.</param>
        /// <returns>
        ///   <c>true</c> if [is not null] [the specified log]; otherwise, <c>false</c>.
        /// </returns>
        private static bool IsNotNull<T>(T log) where T : class
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return log != null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Opens the repositoty.
        /// </summary>
        /// <returns></returns>
        public IRepository Open()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (_isOpen)
                    throw new InvalidOperationException();
                _dataContext = DataContextCache.Instance.Get(_connectionString);
                _isOpen = true;
                return this;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Closes the repository.
        /// </summary>
        public void Close()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!_isOpen)
                    return;
                _dataContext = null;
                _isOpen = false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Disposes the repository.
        /// </summary>
        public void Dispose()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Close();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private IEnumerable<ILogData> GetEntityLogData(Type type)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IEnumerable<ILogData> entities = GetBulkData(() => _dataContext.GetTable(type).Cast<ILogData>().Where(t => t.CreateDateTime < _lastTime).ToList());
                return entities;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the eis and evs data.
        /// </summary>
        /// <returns></returns>
        private IEnumerable<ILogData> GetEisEvsData()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                foreach (Type type in _eisEvsTypes)
                {
                    IEnumerable<ILogData> logDatas = GetEntityLogData(type);
                    foreach (ILogData data in logDatas)
                        yield return data;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Gets data in bulks.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="getDataFunc">The get data func.</param>
        /// <returns></returns>
        private IEnumerable<T> GetBulkData<T>(Func<IList<T>> getDataFunc)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return Util.ProcessingBulkData<T>(_bulkSize, getDataFunc, BulkDataProcessedCallback);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Clear datacontext cache when bulk data was processed.
        /// </summary>
        /// <param name="bulkSize"></param>
        /// <param name="finish"></param>
        private void BulkDataProcessedCallback(int bulkSize, bool finish)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (bulkSize > 0)
                    _dataContext.ClearDbCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Composes the workflow data.
        /// </summary>
        /// <param name="workFlowLog">The work flow log.</param>
        /// <param name="order">The order.</param>
        /// <param name="patientApp">The patient app.</param>
        /// <param name="patientMov">The patient mov.</param>
        /// <param name="schAdm">The SCH adm.</param>
        /// <param name="patient">The patient.</param>
        /// <param name="orderableItem">The orderable item.</param>
        /// <returns></returns>
        private WorkFlowLog ComposeWorkflowData(WorkFlowLog workFlowLog, Order order, PatientAppointment patientApp,
            PatientMovement patientMov, ScheduledAdmission schAdm,
            Patient patient, OrdersOrderableItem orderableItem)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                if (order != null)
                {
                    if (orderableItem != null)
                        order.OrdersOrderableItems.Add(orderableItem);
                    if (workFlowLog.Order != null)
                        return null;
                    workFlowLog.Order = order;
                    order.Patient = patient;
                }
                if (patientApp != null)
                    workFlowLog.PatientAppointment = patientApp;
                if (patientMov != null)
                    workFlowLog.PatientMovement = patientMov;
                if (schAdm != null)
                    workFlowLog.ScheduledAdmission = schAdm;
                return workFlowLog;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the workflow bulk data.
        /// </summary>
        /// <returns></returns>
        private IList<WorkFlowLog> GetBulkWfData()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                var logs = _dataContext.WorkFlowLog.Where(o => o.CreateDateTime <= _lastTime).Take(_bulkSize).OrderBy(o => o.CreateDateTime);

                var orders = from l in logs
                             from order in _dataContext.Order.Where(o => o.IEN == (l.OrderIdInt.HasValue ? l.OrderIdInt.ToString() : null)).DefaultIfEmpty()
                             from patient in _dataContext.Patient.Where(p => p.Name.Trim().ToLower() == order.ObjectOfOrder.Trim().ToLower()).DefaultIfEmpty()
                             from oi in _dataContext.OrdersOrderableItem.Where(oi => oi.OrderId == order.IEN).DefaultIfEmpty()
                             from pa in _dataContext.PatientAppointment.Where(pa => pa.IEN == (l.PatientAppointmentIdInt.HasValue ? l.PatientAppointmentIdInt.ToString() : null)).DefaultIfEmpty()
                             from pm in _dataContext.PatientMovement.Where(pm => pm.IEN == (l.PatientMovementIdInt.HasValue ? l.PatientMovementIdInt.ToString() : null)).DefaultIfEmpty()
                             from sa in _dataContext.ScheduledAdmission.Where(sa => sa.IEN == (l.ScheduledAdmissionIdInt.HasValue ? l.ScheduledAdmissionIdInt.ToString() : null)).DefaultIfEmpty()
                             select ComposeWorkflowData(l, order, pa, pm, sa, patient, oi);

                var result = orders.Where(IsNotNull).ToList();
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the workflow data.
        /// </summary>
        /// <returns></returns>
        private IEnumerable<ILogData> GetWfData()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IEnumerable<WorkFlowLog> entitiesResult = GetBulkData(GetBulkWfData);
                return entitiesResult;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IEnumerable<ILogData> GetLogEntities()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _lastTime = _dataContext.GetSystemDate();

                IEnumerable<ILogData> eisAndEvsData = GetEisEvsData();
                IEnumerable<ILogData> wfData = GetWfData();

                return Enumerable.Concat(eisAndEvsData, wfData);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
