﻿using System;
using System.ServiceModel;
using BMS.DataContracts;
using System.Collections.Generic;
using BMS.FaultContracts;
using BMS.Utils;
using InfoWorld.HL7.ITS;

namespace BMS.ServiceContracts
{
    /// <summary>
    /// BedManagerQuery service contract used for filtering Admissions, Transfers, Bed occupancies.
    /// </summary>
    [ServiceContract()]
    [ServiceKnownType(typeof(AdmissionOrder))]
    [ServiceKnownType(typeof(DischargeOrder))]
    [ServiceKnownType(typeof(MovementOrder))]
    public interface IBedManagerQuery
    {
        /// <summary>
        /// Queries the BMS system for admissions created later than the creation time sent as a parameter.
        /// </summary>
        /// <param name="facilityUid">The facility's uid.</param>
        /// <param name="admissionCreationTime">Date/time parameter used to search for later created admissions.</param>
        /// <param name="isOnlyCurrentWaitingAdmission">if set to true get all the admission where remove date is null else get all admission where admission creation date is GTE the filter admission date.</param>        
        /// <returns>
        /// List of located admissions.
        /// </returns>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Naming", "CA1704:IdentifiersShouldBeSpelledCorrectly", MessageId = "Uid"), OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<DataContracts.AdmissionInfo> FilterAdmissions(Guid facilityUid, DateTime admissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission);

        /// <summary>
        /// Queries BMS for transfers to the facility passed through the parameter.
        /// </summary>
        /// <param name="facilityUid">Facility ID (Guid) where the transfers occur to.</param>
        /// <param name="facilityVISN">The facility VISN.</param>
        /// <returns>
        /// List of located tranfers.
        /// </returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<DataContracts.TransferInfo> FilterTransfers(Guid? facilityUid, string facilityVISN);

        /// <summary>
        /// Queries BMS for bed occupancy info (admissions events) that were registered at the specific division.
        /// </summary>
        /// <param name="divisionUid">Division id (Guid) were the admissions took place.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="pagingContext">Object containing the paging parameter(page number, number of records per page, order by column).</param>
        /// <returns>
        /// List of located bed occupacies.
        /// </returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<BedOccupancyInfo> FilterBedOccupancy(string wardList, II facilityId);

        /// <summary>
        /// Filters the beds DND and OOS.
        /// </summary>
        /// <param name="wardList">The ward list.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<DataContracts.BedDNDAndOOS> FilterBedsDNDAndOOS(string wardList);

        /// <summary>
        /// Queries BMS for the national patient waiting.
        /// </summary>
        /// <param name="facilityRegion">The facility region.</param>
        /// <param name="facilityVISN">The facility VISN.</param>
        /// <param name="specialtyCode">The specialty code.</param>
        /// <param name="specialtyCodeSystem">The specialty code system.</param>
        /// <param name="isCurrentWaitingList">if set to true  return all the patients where transfer event is null else filter by event date.</param>
        /// <param name="transferDateTime">The transfer date time.</param>
        /// <returns></returns>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Naming", "CA1709:IdentifiersShouldBeCasedCorrectly", MessageId = "VISN"), OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<DataContracts.PatientWaitingStandardView> FilterNationalPatientWaitingStandardView(bool isDisplayOnlyNationalPatients, string facilityRegion, string facilityVISN, string specialtyCode, string specialtyCodeSystem, DateTime transferDateTime, bool isCurrentWaitingList);

        /// <summary>
        /// Queries BMS for the national patient waiting date view.
        /// </summary>
        /// <param name="facilityRegion">The facility region.</param>
        /// <param name="facilityVISN">The facility VISN.</param>        
        /// <param name="specialtyCode">The specialty code.</param>
        /// <param name="specialtyCodeSystem">The specialty code system.</param>
        /// <param name="isCurrentWaitingList">if set to true  return all the patients where transfer event is null else filter by event date.</param>
        /// <param name="transferDateTime">The transfer date time.</param>
        /// <returns></returns>
        [System.Diagnostics.CodeAnalysis.SuppressMessage("Microsoft.Naming", "CA1709:IdentifiersShouldBeCasedCorrectly", MessageId = "VISN"), OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<DataContracts.PatientWaitingDateView> FilterNationalPatientWaitingDateView(bool isDisplayOnlyNationalPatients, string facilityRegion, string facilityVISN, string specialtyCode, string specialtyCodeSystem, DateTime transferDateTime, bool isCurrentWaitingList);


        /// <summary>
        /// Gets the new events.
        /// </summary>
        /// <param name="wardUidList">The ward uid list.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="timeZoneMinutesOffset">The facility offset.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<DataContracts.NewEvent> GetNewEvents(string wardUidList, Guid facilityUid, int timeZoneMinutesOffset);

        /// <summary>
        /// Queries BMS for beds that needs cleaning.
        /// </summary>
        /// <param name="divisionUid">The division uid.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <returns>
        /// List of bed cleaning.
        /// </returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<DataContracts.BedCleanInfo> FilterBedClean(string wardList, DateTime startDate, DateTime endDate);

        /// <summary>
        /// Gets the log entries that check the specified filter conditions.
        /// </summary>
        /// <param name="vistaId">The vista id.</param>
        /// <param name="retrievalMethod">The retrieval method.</param>
        /// <param name="dataType">Type of the data.</param>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="status">The status.</param>
        /// <param name="pagingContext">The paging context.</param>
        /// <returns>A list of log entries.</returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<JobLogInfo> FilterLogData(IList<string> vistaIds, IList<BMS.Utils.DataRetrievalMethod> retrievalMethods, IList<VistaDataType> dataTypes, DateTime startDate, DateTime? endDate, IList<BMS.Utils.JobStatus> status, PagingContext pagingContext, bool is_view_only_last_information);

        /// <summary>
        /// Gets the last recorded act date for the patient specified by its IEN in the requested VistA site.
        /// </summary>
        /// <param name="patientId">The patient id.</param>
        /// <param name="actType">Type of the act.</param>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns>
        /// The last recorded act date or null if no medical act is found.
        /// </returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        Act GetLastAct(II patientId, ActType actType, II vistaSiteId);

        /// <summary>
        /// Gets the percentage of occupancy.
        /// </summary>
        /// <param name="vistaSitesList">The vista sites list.</param>
        /// <param name="wardList">The ward list.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<BedOccupancyCount> GetPercentageOfOccupancy(string vistaSitesList, string wardsList);

        /// <summary>
        /// Gets the patients waiting count.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="vistaSiteUid">The vista site uid.</param>
        /// <param name="vistaDivisionCode">The vista division code.</param>
        /// <param name="timeZoneOffset">The time zone offset.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        PatientWaitingCount GetPatientsWaitingCount(Guid facilityUid, Guid vistaSiteUid, string vistaDivisionCode, int timeZoneOffset);

        /// <summary>
        /// Filters the icon info.
        /// </summary>
        /// <param name="listType">Type of the list(All or Whiteboard or Search).</param>
        /// <param name="iconFlagIds">Icon Flag Ids for search.</param>
        /// <param name="facility">The facility.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<IconInfo> FilterIconInfo(string listType, string iconFlagIds, II facilityId);

        /// <summary>
        /// Filters the bed patient icon association.
        /// </summary>
        /// <param name="bedUid">The bed uid.</param>
        /// <param name="patientUid">The patient uid.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<IconAssociationInfo> FilterBedPatientIconAssociation(Guid bedUid, Guid? patientUid, Guid facilityUid);

        /// <summary>
        /// Gets the occupied beds.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<OccupiedBedInfo> GetOccupiedBeds(II vistaSiteId);

        /// <summary>
        /// Filters the bed icon association.
        /// </summary>
        /// <param name="bedUids">The bed uids.</param>
        /// <param name="facilityUid">The facility uid.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<IconAssociation> FilterBedIconAssociation(string bedUids, Guid facilityUid);

        /// <summary>
        /// Filters the waiting list report.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="filterBy">The filter by.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<WaitingListReport> FilterWaitingListReport(Guid facilityUid, string filterBy);

        /// <summary>
        /// Get all patients must be evacuated.
        /// </summary>
        /// <param name="wardList">The ward list.</param>
        /// <param name="facility">The facility.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<EvacuationPatient> GetEvacuationPatients(string wardList, II facilityId);

        /// <summary>
        /// Filters the bed clean assigned and pending.
        /// </summary>
        /// <param name="wardList">The ward list.</param>
        /// <param name="acceptedBy">The accepted by.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<DataContracts.BedCleanInfo> FilterBedCleanAssignedAndPending(string wardList, string acceptedBy);

        /// <summary>
        /// Filters the ems staff.
        /// </summary>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<EmsStaff> FilterEmsStaff(II facilityId);

        /// <summary>
        /// Gets the bed staff history.
        /// </summary>
        /// <param name="bedId">The bed id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<BedStaffHistory> GetBedStaffHistory(II bedId);

        /// <summary>
        /// Gets the cancelable orders.
        /// </summary>
        /// <param name="vistaSiteId">The vista site id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]        
        IList<Order> GetCancelableOrders(II vistaSiteId);

        /// <summary>
        /// Gets the already exists vista integration logs.
        /// </summary>
        /// <param name="data">The data.</param>
        /// <param name="vista">The vista.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<VistaIntegrationLog> GetAlreadyExistsVistaIntegrationLogs(IEnumerable<VistaIntegrationLog> data, string vista);

        /// <summary>
        /// Filters the bed unavailable info.
        /// </summary>
        /// <param name="vistaId">The vista id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<BedUnavailableInfo> FilterBedUnavailableInfo(II vistaId);

        /// <summary>
        /// Filters the updating patients.
        /// </summary>
        /// <param name="vistaId">The vista id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        IList<II> FilterUpdatingPatients(II vistaId);

        /// <summary>
        /// Gets the movement iens.
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="vistaId">The vista id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<MovementIen> GetMovementIens(DateTime startDate, DateTime endDate, II vistaId);

        /// <summary>
        /// Filters the movements without bed.
        /// </summary>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="vistaId">The vista id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<MovementWithoutBed> FilterMovementsWithoutBed(DateTime startDate, DateTime endDate, II vistaId);

        /// <summary>
        /// Gets the missing admissions.
        /// </summary>
        /// <param name="admissionIenList">The admission ien list.</param>
        /// <param name="vistaId">The vista id.</param>
        /// <returns></returns>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        List<string> GetMissingAdmissions(string admissionIenList, II vistaId);

        /// <summary>
        /// Gets the BP last run
        /// </summary>
        /// <param name="vistaId">The vista id.</param>
        /// <param name="retrievalMethod">The retrieval method.</param>
        /// <param name="dataType">Type of the data.</param>
        /// <param name="status">The status.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        DateTime? FilterBPLastRun(string vistaId, IList<BMS.Utils.DataRetrievalMethod> retrievalMethods, IList<VistaDataType> dataTypes, IList<BMS.Utils.JobStatus> status);

        /// <summary>
        /// Gets the home page BMS data.
        /// </summary>
        /// <param name="facilityUid">The facility uid.</param>
        /// <param name="vistaSitesList">The vista sites list.</param>
        /// <param name="wardUidList">The ward uid list.</param>
        /// <param name="timeZoneMinutesOffset">The time zone minutes offset.</param>
        /// <param name="admissionCreationTime">The admission creation time.</param>
        /// <param name="isOnlyCurrentWaitingAdmission">if set to <c>true</c> [is only current waiting admission].</param>
        /// <param name="isOnlyInHouseWaitingAdmission">if set to <c>true</c> [is only in house waiting admission].</param>
        /// <param name="bedOccupancylist">The bed occupancylist.</param>
        /// <param name="newEvents">The new events.</param>
        /// <param name="facilityWaitingList">The facility waiting list.</param>
        [OperationContract]
        [FaultContract(typeof(CustomSQLException))]
        [FaultContract(typeof(OtherException))]
        void GetHomePageBMSData(Guid facilityUid, string vistaId, string wardUidList, int timeZoneMinutesOffset, DateTime admissionCreationTime, bool isOnlyCurrentWaitingAdmission, bool isOnlyInHouseWaitingAdmission,
            IList<DataRetrievalMethod> retrievalMethods, IList<VistaDataType> dataTypes, IList<JobStatus> status,
            out List<DataContracts.BedOccupancyCount> bedOccupancylist, out List<DataContracts.NewEvent> newEvents, out List<DataContracts.AdmissionInfo> facilityWaitingList, out DateTime? bpLastRunDate);
    }
}
