﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using InfoWorld.HL7.ITS;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using System.ServiceModel;
using System.Linq.Expressions;


namespace BMS.Facade
{
    /// <summary>
    /// Holds TimeZone implementation.
    /// </summary>
    public static class TimeZoneUtil
    {
        private static Dictionary<Type, List<string>> _dateTimePropertyPathsDictionary = new Dictionary<Type, List<string>>();
        private static Dictionary<Type, string> _setsTimezoneAttributePathDictionary = new Dictionary<Type, string>();        
        private static Dictionary<string, TimeZoneInfo> _timeZoneInfoCache = new Dictionary<string, TimeZoneInfo>(StringComparer.InvariantCultureIgnoreCase);

        public static TimeZoneInfo GetCachedTimeZone(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string rootExtension = facilityId.root + facilityId.extension;

                if (_timeZoneInfoCache.ContainsKey(rootExtension))
                    return _timeZoneInfoCache[rootExtension];
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void CacheTimeZoneForFacility(II facilityId, TimeZoneInfo tzi)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (facilityId != null && tzi != null)
                {
                    string root = facilityId.root ?? string.Empty;
                    string extension = facilityId.extension ?? string.Empty;
                    string rootExtension = root + extension;

                    _timeZoneInfoCache[rootExtension] = tzi;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void RemoveTimeZoneForFacility(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (facilityId != null)
                {
                    string root = facilityId.root ?? string.Empty;
                    string extension = facilityId.extension ?? string.Empty;
                    string rootExtension = root + extension;
                    _timeZoneInfoCache.Remove(rootExtension);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        #region Public Methods

        /// <summary>
        /// Converts the object dates.
        /// </summary>
        /// <param name="obj">The source object.</param>
        /// <param name="timeZoneInfo">The time zone info.</param>
        /// <param name="fromUTC">fromUTC = true otherwise false.</param>
        /// <returns></returns>
        public static void ConvertObjectDates(Object obj, bool fromUTC, TimeZoneInfo timeZoneInfo)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (obj == null) return;
                Type type = obj.GetType();
                if (type == null) return;
                List<string> dateTimePropertyPaths = new List<string>();

                if (!_dateTimePropertyPathsDictionary.ContainsKey(type)) //list of properties does not exist
                {
                    //gets relative path of DateTime properties
                    foreach (PropertyInfo propertyInfo in GetPropertiesToSearch(type))
                        GetDateTimeAndSetsTimeZoneProperties(propertyInfo, string.Empty, dateTimePropertyPaths, true, type);

                    //add list to dictionary
                    _dateTimePropertyPathsDictionary[type] = dateTimePropertyPaths;
                }
                else
                    dateTimePropertyPaths = _dateTimePropertyPathsDictionary[type];

                if (timeZoneInfo == null)
                {
                    //if timezoneinfo comes through the context, it mustn't be taken from the object itself
                    timeZoneInfo = TimeZoneInfo.Local;
                    if (_setsTimezoneAttributePathDictionary.ContainsKey(type))
                        timeZoneInfo = GetTimeZoneInfo(GetIdentifierPropertyValue(obj, _setsTimezoneAttributePathDictionary[type]));
                }
                //set datetime values
                foreach (string dateTimePropertyPath in dateTimePropertyPaths)
                {
                    DateTime? dateTime = GetDateTimePropertyValue(obj, dateTimePropertyPath);

                    if (dateTime.HasValue)
                    {
                        DateTime dateTimeValue = dateTime.Value;

                        if (fromUTC == true)
                        {
                            dateTimeValue = DateTime.SpecifyKind(dateTimeValue, DateTimeKind.Utc);
                            SetPropertyObject(obj, TimeZoneInfo.ConvertTime(dateTimeValue, timeZoneInfo), dateTimePropertyPath);
                        }
                        else
                        {
                            if (dateTimeValue.Kind != DateTimeKind.Utc)
                            {
                                if (dateTimeValue.Kind == DateTimeKind.Local)
                                    dateTimeValue = DateTime.SpecifyKind(dateTimeValue, DateTimeKind.Unspecified);
                                try
                                {
                                    SetPropertyObject(obj, TimeZoneInfo.ConvertTimeToUtc(dateTimeValue, timeZoneInfo), dateTimePropertyPath);
                                }
                                catch (ArgumentException)
                                {
                                    if (timeZoneInfo.IsInvalidTime(dateTimeValue))
                                        dateTimeValue = dateTimeValue.AddHours(1);
                                    if (timeZoneInfo.IsAmbiguousTime(dateTimeValue))
                                        dateTimeValue = dateTimeValue.AddHours(-1);
                                    SetPropertyObject(obj, TimeZoneInfo.ConvertTimeToUtc(dateTimeValue, timeZoneInfo), dateTimePropertyPath);
                                }
                            }
                        }
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal static TimeZoneInfo GetTimeZoneInfo(II identifier)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (identifier == null)
                    return null;

                string root = identifier.root ?? string.Empty;
                string extension = identifier.extension ?? string.Empty;
                string rootExtension = root + extension;

                if (_timeZoneInfoCache.ContainsKey(rootExtension))
                    return _timeZoneInfoCache[rootExtension];
                else
                {
                    //actual call to get the timezone info for the facility
                    TimeZoneInfo timeZoneInfo = GetTimeZoneInfoFromService(identifier);
                    if (timeZoneInfo != null)
                        _timeZoneInfoCache[rootExtension] = timeZoneInfo;
                    return timeZoneInfo;
                }             
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static TimeZoneInfo GetTimeZoneInfoFromService(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(facilityId);

                try
                {
                    TimeZoneInfo timeZone = TimeZoneInfo.FindSystemTimeZoneById(settings.FacilitySiteTimeZone.displayName);
                    if (settings.LocalTimeAdjust == 0)
                        return timeZone;
                    else
                    {
                        TimeSpan utcOffset = timeZone.GetUtcOffset(TimeZoneInfo.ConvertTimeFromUtc(DateTime.UtcNow, timeZone));
                        return TimeZoneInfo.CreateCustomTimeZone("CTZ", utcOffset + TimeSpan.FromHours(settings.LocalTimeAdjust), "Custom Time Zone", "CustomTimeZone");
                    }
                }
                catch
                {
                    Facility facility = FacadeManager.EntityInterface.GetFacility(facilityId);
                    StringBuilder adminUrl = new StringBuilder();
                    adminUrl.Append(@"<a href=""../");
                    adminUrl.Append(BMS.Utils.Properties.Resources.ADMIN_FACILITY_EDIT_RELATIVE_PATH);
                    adminUrl.Append(@""">Admin</a>");
                    StringBuilder facilityName = new StringBuilder();
                    facilityName.Append(facility.Code);
                    facilityName.Append(", ");
                    facilityName.Append((facility.Address1 != null) ? (!string.IsNullOrEmpty(facility.Address1.city) ? facility.Address1.city : string.Empty) : string.Empty);
                    facilityName.Append(", ");
                    facilityName.Append(facility.SiteNumber);                     
                    throw new FaultException<GenericException>(new GenericException() { FriendlyMessage = String.Format(BMS.Utils.Properties.Resources.ERR_CONFIGURATION_FACILITY_TIMEZONE_NOT_EXISTS, facilityName.ToString(), adminUrl.ToString()) });
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void ClearCache()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                _timeZoneInfoCache.Clear();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Private Methods

        private static IEnumerable<PropertyInfo> GetPropertiesToSearch(Type type)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                var result = type.GetProperties(BindingFlags.Instance | BindingFlags.Public);
                return result.Where(p => !p.GetCustomAttributes(typeof(NotConvertDateTimeAttribute), false).Any());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the relative path of the DateTime?/DateTime properties of the object.
        /// </summary>
        /// <param name="propertyInfo">The property info.</param>
        /// <param name="path">The path.</param>
        /// <param name="lista">List of Datetime properties.</param>
        private static void GetDateTimeAndSetsTimeZoneProperties(PropertyInfo propertyInfo, string path, IList<string> dateTimePropertyPaths, bool searchForSetsTimeZoneProperty, Type rootType)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                if (!string.IsNullOrEmpty(path))
                    path += ".";

                path += propertyInfo.Name;

                if (propertyInfo.PropertyType == typeof(DateTime?) || propertyInfo.PropertyType == typeof(DateTime))
                {
                    dateTimePropertyPaths.Add(path);
                    return;
                }

                if (propertyInfo.GetCustomAttributes(typeof(ConvertDateTimeAttribute), false).Any())
                {
                    foreach (PropertyInfo pi in GetPropertiesToSearch(propertyInfo.PropertyType))
                        GetDateTimeAndSetsTimeZoneProperties(pi, path, dateTimePropertyPaths, searchForSetsTimeZoneProperty, rootType);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Sets the value of an in-depth property of an object
        /// </summary>
        /// <param name="obj">The source object</param>
        /// <param name="value">Value to set</param>
        /// <param name="propertyPath">The relative path to the desired property (format:  containedproperty.containedproperty. ... .desiredproperty)</param>
        private static void SetPropertyObject(Object obj, object value, String propertyPath)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (obj == null)
                    return;
                Type type = obj.GetType();
                if (String.IsNullOrEmpty(propertyPath))
                    return;
                String currentProperty;
                int index = propertyPath.IndexOf('.');
                if (index != -1)
                    currentProperty = propertyPath.Substring(0, propertyPath.IndexOf('.'));
                else
                    currentProperty = propertyPath;

                PropertyInfo pi = type.GetProperty(currentProperty);
                if (pi != null)
                {
                    if (index == -1)
                        pi.SetValue(obj, value, null);
                    else
                        SetPropertyObject(pi.GetValue(obj, null), value, propertyPath.Substring(index + 1));
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Returns the date time value of an in-depth property of an object
        /// </summary>
        private static DateTime? GetDateTimePropertyValue(object o, string prop)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                object obj = GetPropertyObject(o, prop);
                return obj == null ? (DateTime?)null : (DateTime?)obj;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static II GetIdentifierPropertyValue(object o, string prop)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return (II)GetPropertyObject(GetPropertyObject(o, prop), "Id");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Returns the value of an in-depth property of an object
        /// </summary>
        /// <param name="sourceObject">The source object</param>
        /// <param name="relativePath">The relative path to the desired property (format:  containedproperty.containedproperty. ... .desiredproperty)</param>
        /// <returns>Object if all properties are instantiated and relative path is correct, null otherwise</returns>
        private static object GetPropertyObject(object sourceObject, string relativePath)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                if (sourceObject == null || string.IsNullOrEmpty(relativePath))
                    return null;
                string[] properties = relativePath.Split('.');
                PropertyInfo pi = sourceObject.GetType().GetProperty(properties[0]);
                if (pi == null)
                    return null;
                object obj = pi.GetValue(sourceObject, null);
                for (int i = 1; i < properties.Length && pi != null && obj != null; i++)
                {
                    pi = obj.GetType().GetProperty(properties[i]);
                    obj = pi.GetValue(obj, null);
                }
                return obj == null ? null : obj;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        #endregion
    }
}
