﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Facade.Service;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using BMS.ServicesWrapper.EIS;
using System.Collections;
using BMS.ServicesWrapper;
using BMS.ServicesWrapper.BMService;
using BMS.Facade.Translators;

namespace BMS.Facade.Implementation
{
    public class EntityOperationsImplementation : IEntityOperations
    {
        #region Bed

        public IList<Bed> GetBedsInDivision(II divisionId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetBedsInDivision(divisionId, vistaSiteId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        #endregion

        #region Ward

        public IList<Ward> GetWardsByDivisionId(II divisionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Ward> list = EISFactory.InstanceFromWCF.GetWards(divisionId).OrderBy(w => w.Name).ToList();
                // fill in the division details
                // to do: fill in the facility on each division, if necessary
                // to do: retrieve all divisions in one call
                Division division = GetDivisionById(divisionId);
                list = list.Select(x =>
                {
                    x.Division = division;
                    return x;
                }).ToList();

                return list;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Ward GetWardById(II wardId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Ward ward = EISFactory.Instance.GetWard(wardId);

                //get current facility
                if (ward.DivisionList != null)
                {
                    //set current facility
                    II facilityId = FacadeManager.UserInterface.GetProfile().Facility.Id;
                    foreach (Division div in ward.DivisionList)
                    {
                        Division division = EISFactory.Instance.GetDivision(div.Id);
                        if (division != null)
                        {
                            if (division.Facility != null && division.Facility.Id != null && division.Facility.Id.root.Equals(facilityId.root, StringComparison.InvariantCultureIgnoreCase) && division.Facility.Id.extension.Equals(facilityId.extension, StringComparison.InvariantCultureIgnoreCase))
                                ward.Division = division;
                        }
                    }
                }
                return ward;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Ward> GetWardsByFacility(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Ward> result = new List<Ward>();
                IList<Division> divisionList = GetDivisions(facilityId);
                foreach (Division division in divisionList)
                    if (division.Facility != null && division.Facility.Id != null && facilityId != null && string.Equals(division.Facility.Id.extension, facilityId.extension, StringComparison.InvariantCultureIgnoreCase))
                        result.AddRange(GetWardsByDivisionId(division.Id));
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Ward> GetWardsByVista(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string prefix = string.Empty;
                string suffix = string.Empty;
                int prefixLength = 0;
                int suffixLength = 0;
                //verify if is Integrated facility
                II facilityId = FacadeManager.UserInterface.GetProfile().Facility.Id;
                if (facilityId != null)
                {
                    FacilitySettings facilitySettings = FacadeManager.ConfigurationInterface.GetFacilitySettings(facilityId);
                    if (facilitySettings != null && facilitySettings.IntegratedFacility != null && facilitySettings.IntegratedFacility.code == Constants.Yes)
                    {
                        //check if facility has ward prefix and ward suffix.
                        if (!string.IsNullOrWhiteSpace(facilitySettings.WardPrefix) && facilitySettings.WardPrefix != Constants.NONE)
                        {
                            prefix = facilitySettings.WardPrefix;
                            prefixLength = prefix.Length;
                        }
                        if (!string.IsNullOrWhiteSpace(facilitySettings.WardSuffix) && facilitySettings.WardSuffix != Constants.NONE)
                        {
                            suffix = facilitySettings.WardSuffix;
                            suffixLength = suffix.Length;
                        }
                    }
                }

                IList<Ward> list = EISFactory.InstanceFromWCF.GetWardsByVistaSiteId(vistaSiteId);
                // fill in the current division details
                IList<Division> divisions = GetDivisions(facilityId);
                list = list.Select(x =>
                {
                    if (x.DivisionList != null)
                    {
                        if (x.DivisionList.Count == 1)
                            x.Division = divisions.Where(a => a.Id.root.Equals(x.DivisionList[0].Id.root, StringComparison.InvariantCultureIgnoreCase) && a.Id.extension.Equals(x.DivisionList[0].Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        else
                        {
                            //get currentDivisionId
                            if (divisions != null)
                                x.Division = (from p in divisions
                                              where (from division in x.DivisionList
                                                     select division.Id.extension).Contains(p.Id.extension)
                                              select p).FirstOrDefault();
                        }
                    }
                    return x;
                }).ToList();

                //filter by ward preffix
                if (!string.IsNullOrWhiteSpace(prefix) && prefixLength != 0)
                {
                    list = list.Where(a => a.Name.Length >= prefixLength).ToList();
                    list = list.Where(a => a.Name.Substring(0, prefixLength).ToUpper() == prefix).ToList();
                }

                //filter by ward suffix
                if (!string.IsNullOrWhiteSpace(suffix) && suffixLength != 0)
                {
                    list = list.Where(a => (a.Name.Length - suffixLength >= 0)).ToList();
                    list = list.Where(a => a.Name.Substring(a.Name.Length - suffixLength, suffixLength).ToUpper() == suffix).ToList();
                }



                return (from a in list orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<ValidationAlert> UpdateWard(Ward ward)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<ValidationAlert> alerts = new List<ValidationAlert>();

                // handle the division: 
                // - check if Word Group Text is changed (get the previous Ward group text)
                // - if it is changed, search the new division by name
                // - if it is not found, create a new division with this name
                bool divisionChanged = false;
                Division oldDivision = null;
                Division newDivision = null;
                if (ward.Division != null)
                {
                    if (ward.Division.Id != null)
                    {
                        oldDivision = EISFactory.Instance.GetDivision(ward.Division.Id);
                        if (!string.Equals(oldDivision.Name, ward.Division.Name, StringComparison.InvariantCultureIgnoreCase))
                            divisionChanged = true;
                    }
                    else if (!string.IsNullOrEmpty(ward.Division.Name))
                    {
                        divisionChanged = true;
                    }
                }

                if (divisionChanged)
                {
                    User user = FacadeManager.UserInterface.GetProfile();
                    newDivision = GetDivisionByName(ward.Division.Name);
                    if (newDivision == null)
                    {
                        //create new division entity  
                        newDivision = new Division() { Name = ward.Division.Name, Facility = user.Facility, Ien = "0" };
                        EISFactory.Instance.CreateDivision(newDivision);
                    }
                    if (oldDivision == null)
                        oldDivision = GetWardById(ward.Id).Division;
                    ward.Division = newDivision;
                    // associate the bed to the division
                    IList<Bed> beds = EISFactory.Instance.GetBedsInWard(ward.Id, ward.VistaSite.Id);
                    foreach (Bed bed in beds)
                    {
                        EISFactory.Instance.UpdateAssociationBetweenBedAndDivision(bed, oldDivision, newDivision);
                    }
                }

                // the ward is activated 
                ward.IsActive = true;
                EISFactory.Instance.UpdateAssociationBetweenWardAndDivision(ward, oldDivision, newDivision);
                if (divisionChanged)
                {
                    if (oldDivision != null && oldDivision.Id != null && !string.IsNullOrEmpty(oldDivision.Id.extension) && !oldDivision.Id.extension.Equals(Guid.Empty.ToString()))
                    {
                        oldDivision = GetDivisionById(oldDivision.Id);
                        IList<Ward> oldDivisionWards = GetWardsByDivisionId(oldDivision.Id);

                        if (oldDivisionWards == null || oldDivisionWards.Count == 0)
                        {
                            EISFactory.Instance.DeleteDivision(oldDivision.Id);
                        }
                        else
                        {
                            Ward updateWard = oldDivisionWards.Where(a => a.Id.extension.Equals(ward.Id.extension, StringComparison.InvariantCultureIgnoreCase) && a.Id.root.Equals(ward.Id.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                            if (updateWard != null)
                            {
                                if (oldDivisionWards.Remove(updateWard))
                                {
                                    if (oldDivisionWards.Where(a => a.Division != null && a.Division.Id.extension.Equals(oldDivision.Id.extension, StringComparison.InvariantCultureIgnoreCase) && a.Division.Id.root.Equals(oldDivision.Id.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                                    {
                                        EISFactory.Instance.DeleteDivision(oldDivision.Id);
                                    }
                                }
                            }
                        }

                    }
                }
                return alerts;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<ValidationAlert> DeleteWard(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Ward deletedWard = GetWardById(id);
                Division division = GetDivisionById(deletedWard.Division.Id);

                IList<ValidationAlert> alerts = new List<ValidationAlert>();
                IList<Bed> beds = EISFactory.Instance.GetBedsInWard(id, deletedWard.VistaSite.Id);
                List<II> divisionsForBed = null;
                Ward ward1 = null;
                foreach (Bed bed in beds)
                {
                    divisionsForBed = new List<II>();
                    if (bed.WardList != null && bed.WardList.Count > 1)
                    {
                        foreach (Ward w in bed.WardList)
                        {
                            ward1 = EISFactory.InstanceFromWCF.GetWard(w.Id);
                            if (ward1 != null && ward1.DivisionList != null)
                                divisionsForBed.AddRange(ward1.DivisionList.Select(a => a.Id));
                        }
                        if (divisionsForBed.Where(a => a.extension.Equals(division.Id.extension, StringComparison.InvariantCultureIgnoreCase)).Count() == 1)
                            //delete association between Bed and Division
                            EISFactory.Instance.DeleteAssociationBetweenBedAndDivision(bed, division);
                    }
                    else
                        //delete association between Bed and Division
                        EISFactory.Instance.DeleteAssociationBetweenBedAndDivision(bed, division);
                }

                //delete association between Ward and Division
                EISFactory.Instance.DeleteAssociationBetweenWardAndDivision(deletedWard, division);

                if (division != null)
                {
                    List<Ward> oldDivisionWards = GetWardsByDivisionId(division.Id).ToList();
                    if (oldDivisionWards == null || oldDivisionWards.Count == 0)
                    {
                        EISFactory.Instance.DeleteDivision(division.Id);
                    }
                    else
                    {
                        Ward ward = oldDivisionWards.Where(a => a.Id.extension.Equals(deletedWard.Id.extension, StringComparison.InvariantCultureIgnoreCase) && a.Id.root.Equals(deletedWard.Id.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (ward != null && oldDivisionWards.Remove(ward))
                        {
                            if (oldDivisionWards.Where(a => a.Division != null && a.Division.Id.extension.Equals(division.Id.extension, StringComparison.InvariantCultureIgnoreCase) && a.Division.Id.root.Equals(division.Id.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                            {
                                EISFactory.Instance.DeleteDivision(division.Id);
                            }
                        }
                    }
                }
                return alerts;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Division

        public IList<Division> GetDivisions(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                //divisions are cached at the wrapper level.
                return EISFactory.InstanceFromWCF.GetDivisions(facilityId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Division GetDivisionByName(string name)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                II facilityId = FacadeManager.UserInterface.GetProfile().Facility.Id;

                //divisions are cached at the wrapper level.
                return BMS.ServicesWrapper.EIS.EISFactory.Instance.FindDivision(facilityId, name);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Division GetDivisionById(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                //divisions are cached at the wrapper level.
                return BMS.ServicesWrapper.EIS.EISFactory.InstanceFromWCF.GetDivision(id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Division> GetDivisionsByBedAndFacility(II bedId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                IList<Division> list = EISFactory.Instance.GetDivisionsForBed(bedId, f.VistaSite.Id);
                return list.Where(a => a.Facility.Id.extension.Equals(facilityId.extension, StringComparison.InvariantCultureIgnoreCase)).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateDivision(Division division)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EISFactory.Instance.UpdateDivision(division);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Facility

        public Facility GetFacility(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = EISFactory.InstanceFromWCF.GetFacility(id);
                if (facility == null)
                    return null;

                // fill in the vista site details 
                if (facility.VistaSite != null)
                    facility.VistaSite = GetVistaSite(facility.VistaSite.Id);

                return facility;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Facility GetFacilityByCode(string code)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = EISFactory.Instance.GetFacilityByCode(code);
                if (facility == null)
                    return null;

                // fill in the vista sites details 
                if (facility.VistaSite != null)
                {
                    IList<VistaSite> vistaSites = GetVistaSites();
                    facility.VistaSite = vistaSites.FirstOrDefault(x => IIComparer.Instance.Equals(x.Id, facility.VistaSite.Id));
                }

                return facility;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilitiesByVisn(II visnId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<VistaSite> vistaSites = GetVistaSites(visnId);
                List<Facility> list = new List<Facility>();
                foreach (VistaSite vistaSite in vistaSites)
                    list.AddRange(GetFacilities(vistaSite.Id));
                return (from a in list orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilities(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetFacilities(vistaSiteId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilities()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<Facility> result = new List<Facility>();
                List<VistaSite> vistaSites = GetVistaSites().ToList();
                foreach (VistaSite vistaSite in vistaSites)
                    result.AddRange(GetFacilities(vistaSite.Id));
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Facility> GetFacilitiesUsingWindowsAuthentication()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceWindows.GetFacilitiesUsingWindowsAuthentication();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<ValidationAlert> SaveFacility(Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<ValidationAlert> alerts = new List<ValidationAlert>();

                // the facility is activated 
                facility.IsActive = true;
                EISFactory.Instance.UpdateFacility(facility);

                return alerts;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Facility GetFacilityByDivisionId(II divisionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = EISFactory.Instance.GetFacilityByDivisionId(divisionId);

                // fill in the vista site details 
                if (facility.VistaSite != null)
                    facility.VistaSite = GetVistaSite(facility.VistaSite.Id);

                return facility;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Hospital Location

        public IList<HospitalLocation> GetHospitalLocations(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceFromWCF.GetHospitalLocationsByVistaDB(vistaSiteId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public HospitalLocation GetHospitalLocation(II id, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceFromWCF.GetHospitalLocationDB(id);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region VistaSite

        public IList<VistaSite> GetVistaSites(II visnId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<VistaSite> list = EISFactory.Instance.GetVistaSites(visnId);
                Visn visn = GetVisn(visnId);
                list = list.Select(x =>
                {
                    if (x.Visn != null)
                        x.Visn = visn;
                    return x;
                }).ToList();
                return (from a in list orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<VistaSite> GetVistaSites()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Visn> visns = GetVisns();
                List<VistaSite> result = new List<VistaSite>();
                foreach (Visn visn in visns)
                    result.AddRange(GetVistaSites(visn.Id).ToList());
                return (from a in result orderby a.Name select a).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public VistaSite GetVistaSite(II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                VistaSite vistaSite = EISFactory.InstanceFromWCF.GetVistaSite(vistaSiteId);
                if (vistaSite.Visn != null)
                    vistaSite.Visn = GetVisn(vistaSite.Visn.Id);
                return vistaSite;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<VistaSite> GetVistaSitesUsingWindowsAuthentication()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceWindows.GetVistaSitesUsingWindowsAuthentication();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Visn

        public IList<Visn> GetVisns(II regionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Visn> list = EISFactory.Instance.GetVisns(regionId);
                // fill in the region details 
                Region region = GetRegion(regionId);
                list = list.Select(x =>
                {
                    if (x.Region != null)
                        x.Region = region;
                    return x;
                }).ToList();

                return list;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Visn> GetVisns()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Region> regions = GetRegions();
                List<Visn> result = new List<Visn>();
                foreach (Region reg in regions)
                    result.AddRange(GetVisns(reg.Id));
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Visn> GetVisnsUsingWindowsAuthentication()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceWindows.GetVisnsUsingWindowsAuthentication();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Visn GetVisn(II visnId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Visn visn = EISFactory.InstanceFromWCF.GetVisn(visnId);
                if (visn.Region != null)
                    visn.Region = GetRegion(visn.Region.Id);
                return visn;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Region

        public IList<Region> GetRegions()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetRegions();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Region GetRegion(II regionId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceFromWCF.GetRegion(regionId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Patient

        public Patient GetPatientBySsn(II ssn, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetPatientBySsn(ssn, vistaCode);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Patient GetPatientById(II id, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetPatientById(id, vistaCode);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Patient> GetPatientsByLastName(string lastName, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetPatientsByLastName(lastName, vistaCode);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Patient> GetPatientsByLastNameInitialAndLastFourSSN(string filter, string vistaCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.Instance.GetPatientsByLastNameInitialAndLastFourSSN(filter, vistaCode);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CreatePatient(Patient patient)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EISFactory.Instance.CreatePatient(patient);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdatePatient(Patient patient)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EISFactory.Instance.UpdatePatient(patient);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeletePatient(II id)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EISFactory.Instance.DeletePatient(id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region DischargeClinic

        public void SaveDischargeClinic(DischargeClinic dischargeClinic)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClientFromWCF.CreateHospitalLocationAppointment(dischargeClinic.ToDataContract());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<DischargeClinic> GetDischargeClinics(Facility facility, VistaSite vistaSite)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Guid? facilityUid = null;
                Guid? vistaSiteUid = null;
                if (facility != null && facility.Id != null && !string.IsNullOrEmpty(facility.Id.extension) && !facility.Id.extension.Equals(Guid.Empty.ToString()))
                    facilityUid = Guid.Parse(facility.Id.extension);
                if (vistaSite != null && vistaSite.Id != null && !string.IsNullOrEmpty(vistaSite.Id.extension) && !vistaSite.Id.extension.Equals(Guid.Empty.ToString()))
                    vistaSiteUid = Guid.Parse(vistaSite.Id.extension);
                List<DischargeClinic> result = BMSFactory.BedManagerOperationsClientFromWCF.GetHospitalLocationsAppointment(facilityUid, vistaSiteUid).Select<DataContracts.HospitalLocationAppointment, DischargeClinic>(a => a.ToFacadeContract()).ToList();
                if (vistaSiteUid == null)
                    vistaSite = EISFactory.InstanceFromWCF.GetVistaSite(facility.VistaSite.Id);
                List<Facility> facilities = EISFactory.InstanceFromWCF.GetFacilities(vistaSite.Id).ToList();
                List<II> hospitalLocationIds = new List<II>();
                foreach (DischargeClinic dc in result)
                {
                    if (dc.HospitalLocation != null && dc.HospitalLocation.Id != null && 
                            hospitalLocationIds.Where(a => a.extension.Equals(dc.HospitalLocation.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                        hospitalLocationIds.Add(dc.HospitalLocation.Id);                     
                    if (dc.Facility != null)
                        dc.Facility = facilities.Where(a => a.Id.extension.Equals(dc.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    dc.VistaSite = vistaSite;
                }
                if (hospitalLocationIds != null && hospitalLocationIds.Count > 0)
                {
                    List<HospitalLocation> locations = EISFactory.InstanceFromWCF.GetHospitalLocationsByIdsDB(hospitalLocationIds);
                    if (locations != null)
                    {
                        foreach (DischargeClinic dc in result)
                        {
                            if (dc.HospitalLocation != null && dc.HospitalLocation.Id != null)
                                dc.HospitalLocation = locations.Where(a => a.Id.extension.Equals(dc.HospitalLocation.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        }
                    }
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteDischargeClinic(II dischargeClinicId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClientFromWCF.DeleteHospitalLocationAppointment(dischargeClinicId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        public IList<string> GetDomains()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceFromWCF.GetDomains();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
