﻿using System;
using System.Collections.Generic;
using System.Linq;
using BMS.Facade.Service;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.BMService;
using DC = BMS.DataContracts;
using BMS.Facade.Translators;
using BMS.Utils;
using BMS.ServicesWrapper.EIS;
using System.Collections;
using Microsoft.Reporting.WebForms;
using System.IO;
using System.Web;
using System.Text;
using System.ServiceModel;
using System.Configuration;
using System.Security;
using System.Security.Permissions;

namespace BMS.Facade.Implementation
{
    public class BedOperationsImplementation : IBedOperations
    {
        public Bed GetBed(II bedId, II vistaSiteId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EISFactory.InstanceFromWCF.GetBed(bedId, vistaSiteId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<Bed> GetBedsDNDAndOOS(string wardList, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<II> list = BMSFactory.BedManagerQueryClientFromWCF.FilterBedsDNDAndOOS(wardList).Select<DC.BedDNDAndOOS, II>(a => a.BedId).ToList();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                return EISFactory.InstanceFromWCF.GetBeds(list.Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id).OrderBy(a => a.Name).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<II> GetBedsDNDOrOOS(string wardList, string typeCode)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return BMSFactory.BedManagerQueryClientFromWCF.FilterBedsDNDAndOOS(wardList).
                        Where(a => !a.IsUnavailableFromVista && a.UnavailableTypeCode.Equals(typeCode, StringComparison.InvariantCultureIgnoreCase)).
                        Select<DC.BedDNDAndOOS, II>(b => b.BedId).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the bed occupancy report for the specified division.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <param name="bedStatus">The bed status.</param>
        /// <returns></returns>
        public IList<BedInfo> GetBedOccupancy(List<Ward> wardList, II divisionId, Facility facility, CD bedStatus, bool isDoNotDisplay, string sortBy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder wards = new StringBuilder();
                foreach (Ward ward in wardList)
                {
                    wards.Append(ward.Id.extension);
                    wards.Append(",");
                }
                List<Module> flags = FacadeManager.ConfigurationInterface.GetModules(facility.Id.root, facility.Id).ToList();
                bool discOrdInactive = flags.Where(a => a.Code.Equals(Constants.DISCH_ORD, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? false : true;
                bool anticDiscInactive = flags.Where(a => a.Code.Equals(Constants.ANTIC_DISCH, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? false : true;
                bool discAppInactive = flags.Where(a => a.Code.Equals(Constants.DISCHARGE_APPOINTMENT, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? false : true;
                bool trsfOrdInactive = flags.Where(a => a.Code.Equals(Constants.TRSF_ORD, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? false : true;
                bool emsInactive = flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.Yes, StringComparison.InvariantCultureIgnoreCase) ? false : true;
                TimeZoneInfo facilityTimeZone = TimeZoneUtil.GetTimeZoneInfo(facility.Id);
                IList<BedInfo> allBedsInfo = BMSFactory.BedManagerQueryClientFromWCF.FilterBedOccupancy(wards.ToString(), facility.Id).Select<DC.BedOccupancyInfo, BedInfo>(dcBI => dcBI.ToFacadeContract(facilityTimeZone, emsInactive, discOrdInactive, anticDiscInactive, discAppInactive, trsfOrdInactive)).ToList();

                //patient treating specialty is found in EIS
                IList<Patient> patients = EISFactory.InstanceFromWCF.GetPatientsByIds(allBedsInfo.Select(bi => bi.PatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);
                IList<Patient> waitingPatientList = EISFactory.InstanceFromWCF.GetPatientsByIds(allBedsInfo.Select(bi => bi.WaitingListPatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);

                if (patients != null && patients.Count > 0)
                {
                    //attending physician is found in EIS
                    IList<Person> attendingPhysicianList = EISFactory.InstanceFromWCF.GetMedicalPersonsByIds(patients.Select(bi => bi.AttendingPhysicianId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList());

                    foreach (Patient patient in patients)
                        foreach (BedInfo bedInfo in allBedsInfo)
                            if (bedInfo.PatientId != null &&
                               !string.IsNullOrEmpty(bedInfo.PatientId.root) &&
                               !string.IsNullOrEmpty(bedInfo.PatientId.extension) &&
                               patient.Id != null &&
                               string.Equals(bedInfo.PatientId.root, patient.Id.root, StringComparison.InvariantCultureIgnoreCase) &&
                               string.Equals(bedInfo.PatientId.extension, patient.Id.extension, StringComparison.InvariantCultureIgnoreCase))
                            {
                                bedInfo.PatientTreatingSpecialty = patient.TreatingSpecialty;
                                bedInfo.PatientFirstName = patient.FirstName;
                                bedInfo.PatientGender = (patient.Gender != null) ? patient.Gender.GetDisplayName() : string.Empty;
                                bedInfo.PatientLastName = patient.LastName;
                                bedInfo.PT = patient.LastName.Substring(0, 1) + patient.SSN.extension.Substring(patient.SSN.extension.Length - 4, 4);
                                bedInfo.SSN = patient.SSN.extension;
                                if (patient.AttendingPhysicianId != null && attendingPhysicianList != null)
                                {
                                    Person attendingPhysician = attendingPhysicianList.Where(a => a.Id.root.Equals(patient.AttendingPhysicianId.root, StringComparison.InvariantCultureIgnoreCase) && a.Id.extension.Equals(patient.AttendingPhysicianId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                    if (attendingPhysician != null)
                                    {
                                        if (string.IsNullOrEmpty(attendingPhysician.FirstName))
                                            bedInfo.Provider = attendingPhysician.LastName;
                                        else
                                            bedInfo.Provider = attendingPhysician.LastName + ", " + attendingPhysician.FirstName;
                                    }
                                }
                                break;
                            }
                }

                IList<BedInfo> removeDuplicateBeds = new List<BedInfo>();
                IList<BedInfo> bedInfoList = new List<BedInfo>();
                foreach (BedInfo bi in allBedsInfo)
                {
                    if (bi.WardId != null && !string.IsNullOrEmpty(bi.WardId.extension) && !bi.WardId.extension.Equals(Guid.Empty.ToString()))
                        bi.WardName = wardList.Where(a => a.Id.extension.Equals(bi.WardId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Name;
                    if (bi.WaitingListPatientId != null && waitingPatientList != null)
                    {
                        Patient patient = waitingPatientList.Where(a => a.Id.extension.Equals(bi.WaitingListPatientId.extension, StringComparison.InvariantCultureIgnoreCase) && a.Id.root.Equals(bi.WaitingListPatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (patient != null)
                            bi.WaitingListInfo = string.Format(bi.WaitingListInfo, patient.LastName);
                        else
                            bi.WaitingListInfo = string.Empty;
                    }
                    else
                    {
                        bi.WaitingListInfo = string.Empty;
                    }

                    if (bedInfoList.Where(a => a.BedId.extension.Equals(bi.BedId.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                    {
                        if (bi.PatientId != null && !string.IsNullOrEmpty(bi.PatientId.root) && !string.IsNullOrEmpty(bi.PatientId.extension) && string.IsNullOrEmpty(bi.PT))
                            removeDuplicateBeds.Add(bi);
                        else
                            bedInfoList.Add(bi);
                    }
                    else
                    {
                        bedInfoList.Where(a => a.BedId.extension.Equals(bi.BedId.extension, StringComparison.InvariantCultureIgnoreCase)).First().WaitingListInfo += "; " + bi.WaitingListInfo;
                    }
                }

                //remove duplicate beds
                if (removeDuplicateBeds != null && removeDuplicateBeds.Count > 0)
                {
                    foreach (BedInfo bedInfo in removeDuplicateBeds)
                    {
                        bedInfoList.Where(a => a.BedId.extension.Equals(bedInfo.BedId.extension, StringComparison.InvariantCultureIgnoreCase)).ForEach(a => a.WaitingListInfo += "; " + bedInfo.WaitingListInfo);
                    }
                }

                allBedsInfo = bedInfoList;


                IList<Division> divisions = new List<Division>();
                //the patient risks are found in EIS.
                List<Bed> bedsInDivision = new List<Bed>();
                if (divisionId != null && divisionId.extension != (default(Guid)).ToString())
                {
                    divisions.Add(EISFactory.InstanceFromWCF.GetDivision(divisionId));
                    bedsInDivision = EISFactory.InstanceFromWCF.GetBedsInDivision(divisionId, facility.VistaSite.Id).Distinct().ToList();
                }
                else
                {
                    divisions = EISFactory.InstanceFromWCF.GetDivisions(facility.Id).ToList();
                    foreach (Division division in divisions)
                        bedsInDivision.AddRange(EISFactory.InstanceFromWCF.GetBedsInDivision(division.Id, facility.VistaSite.Id));

                    bedsInDivision = bedsInDivision.Distinct().ToList();
                }

                if (isDoNotDisplay)
                {
                    List<BedInfo> dndBeds = allBedsInfo.Where(a => !string.IsNullOrEmpty(a.BedAvailabilityStatusCode) && (a.BedAvailabilityStatusCode.Equals("DND") || a.BedAvailabilityStatusCode.Equals(Constants.BED_OOS_FROM_VISTA))).ToList();
                    foreach (BedInfo bedInfo in dndBeds)
                    {
                        allBedsInfo.Remove(bedInfo);
                        List<Bed> beds = bedsInDivision.Where(b => b.Id.extension.Equals(bedInfo.BedId.extension, StringComparison.InvariantCultureIgnoreCase)).ToList();
                        if (beds != null && beds.Count > 0)
                        {
                            foreach (Bed bed in beds)
                            {
                                bedsInDivision.Remove(bed);
                            }
                        }
                    }
                }

                //temporary list for beds not foound in EIS, but found in BMS
                List<BedInfo> bedsToRemove = new List<BedInfo>();
                foreach (BedInfo bedInfo in allBedsInfo)
                {
                    Bed bed = bedsInDivision.Find(b => b.Id.extension.Equals(bedInfo.BedId.extension, StringComparison.InvariantCultureIgnoreCase));
                    if (bed == null)
                    {
                        bedsToRemove.Add(bedInfo);
                    }

                    else
                        UpdateBedInfo(bedInfo, bed, divisions);
                }
                foreach (BedInfo bi in bedsToRemove)
                    allBedsInfo.Remove(bi);

                List<string> availabilityBeds = new List<string>();
                availabilityBeds = bedsInDivision.Select(a => a.Id.extension).Except(allBedsInfo.Select(a => a.BedId.extension), StringComparer.InvariantCultureIgnoreCase).ToList();

                //build parameter bed_list
                string bedListParameter = string.Empty;
                bedListParameter = string.Join(",", availabilityBeds);
                List<IconAssociation> iconAssociationList = new List<IconAssociation>();

                if (!string.IsNullOrEmpty(bedListParameter))
                    iconAssociationList = BMSFactory.BedManagerQueryClientFromWCF.FilterBedIconAssociation(bedListParameter, Guid.Parse(facility.Id.extension)).Select<DC.IconAssociation, IconAssociation>(dcIA => dcIA.ToFacadeContract()).ToList();
                if (iconAssociationList != null)
                    waitingPatientList = EISFactory.InstanceFromWCF.GetPatientsByIds(iconAssociationList.Select(bi => bi.WaitingListPatientId).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), null);


                foreach (string bedExtension in availabilityBeds)
                {
                    Bed bed = bedsInDivision.Find(b => b.Id.extension.Equals(bedExtension, StringComparison.InvariantCultureIgnoreCase));
                    if (bed == null)
                        continue;
                    BedInfo bedInfo = new BedInfo();
                    if (iconAssociationList != null && iconAssociationList.Count > 0)
                    {
                        //set icons flags
                        IList<IconAssociation> iconBedAssociationList = iconAssociationList.Where(ic => ic.Bed.Id.extension.Equals(bedExtension, StringComparison.InvariantCultureIgnoreCase)).ToList();
                        if (iconBedAssociationList != null && iconBedAssociationList.Count > 0)
                        {
                            bedInfo.IconFlagFacilityIdsOfBedList = iconBedAssociationList[0].ListIconFlagFacilityIdsOfBed;
                            bedInfo.IconFlagIdsOfBedList = iconBedAssociationList[0].ListIconFlagIdsOfBed;
                            bedInfo.IconFlagFacilityIdsOfPatientList = iconBedAssociationList[0].ListIconFlagFacilityIdsOfPatient;
                            bedInfo.IconFlagIdsOfPatientList = iconBedAssociationList[0].ListIconFlagIdsOfPatient;

                            foreach (IconAssociation iconAssociation in iconBedAssociationList)
                            {
                                if (iconAssociation.WaitingListPatientId != null && waitingPatientList != null)
                                {
                                    Patient patient = waitingPatientList.Where(a => a.Id.extension.Equals(iconAssociation.WaitingListPatientId.extension, StringComparison.InvariantCultureIgnoreCase) && a.Id.root.Equals(iconAssociation.WaitingListPatientId.root, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                    if (patient != null)
                                    {
                                        if (string.IsNullOrEmpty(bedInfo.WaitingListInfo))
                                            bedInfo.WaitingListInfo = string.Format(iconAssociation.WaitingListInfo, patient.LastName);
                                        else
                                            bedInfo.WaitingListInfo += "; " + string.Format(iconAssociation.WaitingListInfo, patient.LastName);

                                    }
                                    else
                                        bedInfo.WaitingListInfo = string.Empty;
                                }
                            }
                        }
                        else
                        {
                            bedInfo.WaitingListInfo = string.Empty;
                        }
                    }
                    UpdateBedInfo(bedInfo, bed, divisions);
                    allBedsInfo.Add(bedInfo);
                }

                if (bedStatus == null || IsStringsEqual(bedStatus.code, "All"))
                    return GetAllBeds(allBedsInfo, sortBy);
                if (IsStringsEqual(bedStatus.code, "Occupied"))
                    return GetOccupiedBeds(allBedsInfo, sortBy);
                if (IsStringsEqual(bedStatus.code, "Available"))
                    return GetAvailableBeds(allBedsInfo, sortBy);
                if (IsStringsEqual(bedStatus.code, "Females"))
                    return GetFemalesBeds(allBedsInfo, sortBy);

                throw new InvalidOperationException();
                //       else return allBedsInfo.Where(a => a.BedStatus.Equals(bedStatus.displayName, StringComparison.InvariantCultureIgnoreCase)).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        private void UpdateBedInfo(BedInfo bedInfo)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (bedInfo.PatientFirstName == null)
                    bedInfo.PatientFirstName = string.Empty;
                if (bedInfo.PatientGender == null)
                    bedInfo.PatientGender = string.Empty;
                if (bedInfo.PatientLastName == null)
                    bedInfo.PatientLastName = string.Empty;
                if (bedInfo.BedAvailabilityStatusCode == null)
                    bedInfo.BedAvailabilityStatusCode = string.Empty;
                if (bedInfo.BedCleaningStatusCode == null)
                    bedInfo.BedCleaningStatusCode = string.Empty;
                if (bedInfo.DischargeStatusCode == null)
                    bedInfo.DischargeStatusCode = string.Empty;
                if (bedInfo.LevelOfCareCode == null)
                    bedInfo.LevelOfCareCode = string.Empty;
                if (bedInfo.AdmissionId == null)
                    bedInfo.AdmissionId = new II();
                if (bedInfo.SSN == null)
                    bedInfo.SSN = string.Empty;
                if (bedInfo.PatientId == null)
                    bedInfo.PatientId = new II(string.Empty, string.Empty);
                if (bedInfo.Reason == null)
                    bedInfo.Reason = string.Empty;
                if (bedInfo.DivisionList == null)
                    bedInfo.DivisionList = new List<Division>();
                if (bedInfo.IconFlagFacilityIdsOfBedList == null)
                    bedInfo.IconFlagFacilityIdsOfBedList = new List<int>();
                if (bedInfo.IconFlagFacilityIdsOfPatientList == null)
                    bedInfo.IconFlagFacilityIdsOfPatientList = new List<int>();
                if (bedInfo.IconFlagIdsOfBedList == null)
                    bedInfo.IconFlagIdsOfBedList = new List<int>();
                if (bedInfo.IconFlagIdsOfPatientList == null)
                    bedInfo.IconFlagIdsOfPatientList = new List<int>();
                if (bedInfo.BedName == null)
                    bedInfo.BedName = string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void UpdateBedInfo(BedInfo bedInfo, Bed bed, IList<Division> divisions)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UpdateBedInfo(bedInfo);
                if (bedInfo.BedId == null)
                    bedInfo.BedId = bed.Id;
                if (!string.IsNullOrEmpty(bed.Name))
                    bedInfo.BedName = bed.Name;
                if (bedInfo.StaffName == null || string.IsNullOrEmpty(bedInfo.StaffName))
                    bedInfo.StaffName = bed.NurseAssignment;
                if (bed.CommentList != null && bed.CommentList.Count > 0)
                {
                    // select only the comments asociated with divisions from the selected facility
                    List<BedComment> comments = new List<BedComment>();
                    foreach (BedComment bc in bed.CommentList)
                    {
                        if (bc.Division != null && bc.Division.Id != null
                                && divisions.Where(a => a.Id.extension.Equals(bc.Division.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                            comments.Add(bc);
                    }
                    //check if the bed is assigned to more than one division
                    if (comments.Count > 1)
                        bedInfo.HasMultipleDivisionsAssigned = true;
                    else
                        bedInfo.HasMultipleDivisionsAssigned = false;
                    //check if the bed has more than one comment                    
                    if (comments.Where(a => !string.IsNullOrEmpty(a.Comment)).ToList().Count > 1)
                        bedInfo.HasMultipleComments = true;
                    else
                        bedInfo.HasMultipleComments = false;
                    //set comment associated with the selected division
                    if (divisions.Count == 1)
                    {
                        BedComment bc = comments.Where(a => a.Division.Id.extension.Equals(divisions[0].Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        bedInfo.Comment = (bc != null) ? bc.Comment : string.Empty;
                    }
                    //set first comment found(the divisions was not selected -> option All)
                    else
                        bedInfo.Comment = comments[0].Comment;
                    if (string.IsNullOrEmpty(bedInfo.DivisionName))
                        bedInfo.DivisionName = EISFactory.InstanceFromWCF.GetDivision(bed.CommentList[0].Division.Id).Name;

                    bedInfo.DivisionList = comments.Select(a => a.Division).ToList();
                }
                else
                {
                    bedInfo.Comment = string.Empty;
                    bedInfo.HasMultipleComments = false;
                    bedInfo.HasMultipleDivisionsAssigned = false;
                    bedInfo.DivisionList = new List<Division>();
                }
                if (divisions.Count == 1)
                {
                    bedInfo.HasMultipleComments = false;
                    bedInfo.HasMultipleDivisionsAssigned = false;
                }
                else
                {
                    if (bedInfo.HasMultipleComments)
                        bedInfo.Comment = string.Empty;
                }
                bedInfo.IsBedHold = bed.IsBedHold;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static bool IsStringsEqual(string s1, string s2)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return string.Compare(s1, s2, StringComparison.InvariantCultureIgnoreCase) == 0;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<BedInfo> GetAllBeds(IList<BedInfo> result, string sortBy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(sortBy))
                {
                    return (from p in result
                            orderby p.DivisionName, p.BedName
                            select p).ToList();
                }
                else
                    return GetBedsSorted(result, sortBy);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static IList<BedInfo> GetAvailableBeds(IList<BedInfo> result, string sortBy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                var beds = from p in result
                           where !IsOccupiedBed(p)
                           select p;
                if (string.IsNullOrEmpty(sortBy))
                {
                    return (from p in beds
                            orderby p.DivisionName, p.BedName
                            select p).ToList();
                }
                else
                    return GetBedsSorted(beds.ToList(), sortBy);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<BedInfo> GetFemalesBeds(IList<BedInfo> result, string sortBy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                var beds = from p in result
                           where IsFemaleBed(p)
                           select p;
                if (string.IsNullOrEmpty(sortBy))
                {
                    return (from p in beds
                            orderby p.DivisionName, p.BedName
                            select p).ToList();
                }
                else
                    return GetBedsSorted(beds.ToList(), sortBy);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<BedInfo> GetOccupiedBeds(IList<BedInfo> result, string sortBy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                var beds = from p in result
                           where IsOccupiedBed(p)
                           select p;
                if (string.IsNullOrEmpty(sortBy))
                {
                    return (from p in beds
                            orderby p.DivisionName, p.BedName
                            select p).ToList();
                }
                else
                    return GetBedsSorted(beds.ToList(), sortBy);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static IList<BedInfo> GetBedsSorted(IList<BedInfo> result, string sortBy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IEnumerable<BedInfo> beds = null;
                switch (sortBy)
                {
                    case Constants.SORT_BY_STAFF_ASCENDING:
                        beds = from p in result
                               orderby p.StaffName
                               select p;
                        break;
                    case Constants.SORT_BY_STAFF_DESCENDING:
                        beds = from p in result
                               orderby p.StaffName descending
                               select p;
                        break;
                    case Constants.SORT_BY_ATTENDING_ASCENDING:
                        beds = from p in result
                               orderby p.Provider
                               select p;
                        break;
                    case Constants.SORT_BY_ATTENDING_DESCENDING:
                        beds = from p in result
                               orderby p.Provider descending
                               select p;
                        break;
                    case Constants.SORT_BY_WARD_ASCENDING:
                        beds = from p in result
                               orderby p.WardName
                               select p;
                        break;
                    case Constants.SORT_BY_WARD_DESCENDING:
                        beds = from p in result
                               orderby p.WardName descending
                               select p;
                        break;
                }
                return beds.ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static bool IsFemaleBed(BedInfo bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return IsStringsEqual(bed.PatientGender, "Female");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static bool IsOccupiedBed(BedInfo bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return IsStringsEqual(bed.PatientGender, "Female") || IsStringsEqual(bed.PatientGender, "Male");
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the beds clean.
        /// </summary>
        /// <param name="divisionId">The division id.</param>
        /// <param name="startDate">The start date.</param>
        /// <param name="endDate">The end date.</param>
        /// <param name="pagingContext">The paging context.</param>
        /// <returns></returns>
        public IList<BedCleanInfo> GetBedsClean(string wardList, II divisionId, II facilityId, DateTime startDate, DateTime endDate, TimeZoneInfo facilityTimeZone)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Guid? divisionUid = null;
                if (divisionId != null && !string.IsNullOrEmpty(divisionId.extension))
                    divisionUid = Guid.Parse(divisionId.extension);
                if (facilityTimeZone == null)
                    facilityTimeZone = TimeZoneUtil.GetTimeZoneInfo(facilityId);
                DateTime startDateUTC = TimeZoneInfo.ConvertTimeToUtc(startDate, facilityTimeZone);
                DateTime endDateUTC = TimeZoneInfo.ConvertTimeToUtc(endDate, facilityTimeZone);

                List<string> iconCodes = new List<string>();
                iconCodes.Add(Constants.BED_AVAILABILITY_STATUS_ISOLATION_CODE);
                iconCodes.Add(Constants.EMS_NOTIFIED);
                Dictionary<string, IconInfo> icons = FacadeManager.IconInterface.GetIconsByCodes(iconCodes);
                IconInfo isolationIcon = icons[Constants.BED_AVAILABILITY_STATUS_ISOLATION_CODE];
                IconInfo emsNotifyIcon = icons[Constants.EMS_NOTIFIED];

                IList<BedCleanInfo> bedCleanInfo = BMSFactory.BedManagerQueryClient.FilterBedClean(wardList, startDateUTC, endDateUTC)
                                        .Select<DC.BedCleanInfo, BedCleanInfo>(dcBC => DateTimeConverter.ConvertDateFromUTC(dcBC.ToFacadeContract(isolationIcon, emsNotifyIcon), facilityId)).ToList();

                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                List<BedCleanInfo> bedsToRemove = new List<BedCleanInfo>();
                if (divisionUid != null)
                {
                    string divisionName = FacadeManager.EntityInterface.GetDivisionById(divisionId).Name;
                    List<Bed> bedsInDivision = EISFactory.InstanceFromWCF.GetBedsInDivision(divisionId, f.VistaSite.Id).ToList();
                    foreach (BedCleanInfo bci in bedCleanInfo)
                    {
                        Bed bed = bedsInDivision.Where(b => b.Id.extension.Equals(bci.Bed.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (bed != null)
                        {
                            bci.Bed.Name = bed.Name;
                            bci.WardName = divisionName;
                        }
                        else
                        {
                            bedsToRemove.Add(bci);
                        }
                    }
                }
                else
                {
                    Division division = null;
                    foreach (BedCleanInfo bci in bedCleanInfo)
                    {
                        try
                        {
                            Bed bed = GetBed(bci.Bed.Id, f.VistaSite.Id);
                            if (bed != null && bed.WardList != null && bed.WardList.Count > 0 && bci.Ward != null && bci.Ward.Id != null)
                            {
                                bci.Bed.Name = bed.Name;
                                Ward ward = bed.WardList.Where(a => a.Id.extension.Equals(bci.Ward.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                                if (ward != null)
                                {
                                    division = FacadeManager.EntityInterface.GetWardById(ward.Id).Division;
                                    if (division != null)
                                        bci.WardName = FacadeManager.EntityInterface.GetDivisionById(division.Id).Name;
                                    else
                                        bci.WardName = string.Empty;
                                }
                                else
                                    bci.WardName = string.Empty;
                            }
                            else
                                bedsToRemove.Add(bci);
                        }
                        catch
                        {
                            bci.WardName = string.Empty;
                        }
                    }
                }

                return bedCleanInfo.Except(bedsToRemove).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the bed clean by id.
        /// </summary>
        /// <param name="bedCleanId">The bed clean id.</param>
        /// <returns></returns>
        public BedClean GetBedCleanById(II bedCleanId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedClean bedClean = new BedClean();
                DC.BedCleaningOperation bedCleaning = BMSFactory.BedManagerOperationsClient.GetBedCleaningOperation(bedCleanId, null);
                bedClean = DateTimeConverter.ConvertDateFromUTC(bedCleaning.ToFacadeContract(bedClean), facilityId);
                Bed bed = GetBed(bedClean.Bed.Id, bedClean.VistaSite.Id);
                if (bed != null)
                    bedClean.Bed.Name = bed.Name;
                if (bedClean.Ward != null)
                    bedClean.Ward = FacadeManager.EntityInterface.GetWardById(bedClean.Ward.Id);
                if (bedClean.Division != null)
                    bedClean.Division = FacadeManager.EntityInterface.GetDivisionById(bedClean.Division.Id);
                return bedClean;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public BedClean GetBedCleanByBedId(II bedId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    BedClean bedClean = new BedClean();
                    DC.BedCleaningOperation bedCleaning = BMSFactory.BedManagerOperationsClient.GetBedCleaningOperation(null, bedId);
                    bedClean = DateTimeConverter.ConvertDateFromUTC(bedCleaning.ToFacadeContract(bedClean), facilityId);
                    Bed bed = GetBed(bedClean.Bed.Id, bedClean.VistaSite.Id);
                    if (bed != null)
                        bedClean.Bed.Name = bed.Name;
                    if (bedClean.Ward != null)
                        bedClean.Ward = FacadeManager.EntityInterface.GetWardById(bedClean.Ward.Id);
                    if (bedClean.Division != null)
                        bedClean.Division = FacadeManager.EntityInterface.GetDivisionById(bedClean.Division.Id);
                    return bedClean;
                }
                catch (FaultException<FaultContracts.EntityNotFoundException>) { return null; }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the bed cleaning operation.
        /// </summary>
        /// <param name="bedclean">Bedclean object.</param>
        public void UpdateBedCleaning(BedClean bedclean, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.UpdateBedCleaningOperation(BMOperationsTranslator.ToDataContract(DateTimeConverter.ConvertDateToUTC(bedclean, facilityId)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the bed unavailable.
        /// </summary>
        /// <param name="bedId">The bed id.</param>
        /// <returns></returns>
        public BedUnavailable GetBedUnavailable(II bedId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedUnavailable bu = null;
                try
                {
                    bu = BMSFactory.BedManagerOperationsClient.GetBedUnavailable(null, bedId).ToFacadeContract();
                    Bed bed = GetBed(bedId, bu.VistaSite.Id);
                    if (bed != null)
                        bu.Bed.Name = bed.Name;
                    DateTimeConverter.ConvertDateFromUTC(bu, facilityId);
                }
                catch { }
                return bu;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Updates the bed.
        /// </summary>
        /// <param name="bed">The bed.</param>
        public void UpdateBed(Bed bed)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EISFactory.Instance.UpdateBed(bed);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void InsertBedUnavailableHistory(BedUnavailableHistory bedUnavailableHistory, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.InsertBedUnavailableHistory(BMOperationsTranslator.ToDataContract(DateTimeConverter.ConvertDateToUTC(bedUnavailableHistory, facilityId)));
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<BedOccupancyCount> GetPercentageOfOccupancy(string vistaSitesList, string wardsList, IList<II> facilityIdList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<BedOccupancyCount> occupancyPercentList = new List<BedOccupancyCount>();
                List<DC.BedOccupancyCount> bedOccupancyInfo = BMSFactory.BedManagerQueryClient.GetPercentageOfOccupancy(vistaSitesList, wardsList).ToList();
                int totalBeds = 0;
                int totalOccupaiedBeds = 0;
                int totalUnavailableBeds = 0;
                Facility f = null;
                foreach (II facilityId in facilityIdList)
                {
                    f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                    List<Division> divisions = EISFactory.InstanceFromWCF.GetDivisions(facilityId).ToList();
                    List<Bed> beds = new List<Bed>();
                    foreach (Division division in divisions)
                    {
                        beds.AddRange(EISFactory.InstanceFromWCF.GetBedsInDivision(division.Id, f.VistaSite.Id));
                    }
                    beds = beds.Distinct().ToList();
                    totalBeds = beds.Count();
                    totalOccupaiedBeds = (from bed in beds
                                          where (from bedOccupacyInfo in bedOccupancyInfo where bedOccupacyInfo.IsOccupiedBed == true select bedOccupacyInfo.BedId.extension).Contains(bed.Id.extension, StringComparer.InvariantCultureIgnoreCase)
                                          select bed).Count();
                    totalUnavailableBeds = (from bed in beds
                                            where (from bedOccupacyInfo in bedOccupancyInfo where bedOccupacyInfo.IsUnavailableBed == true select bedOccupacyInfo.BedId.extension).Contains(bed.Id.extension, StringComparer.InvariantCultureIgnoreCase)
                                            select bed).Count();

                    occupancyPercentList.Add(new BedOccupancyCount() { FacilityId = facilityId, OccupancyPercent = (int)((double)(totalOccupaiedBeds) / (totalBeds - totalUnavailableBeds) * 100) });
                }

                return occupancyPercentList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        public IList<OccupiedBedInfo> GetOccupiedBeds(II vistaSiteId)
        {
            IList<Ward> wards = EISFactory.InstanceWindows.GetWardsByVistaSiteId(vistaSiteId);
            List<Bed> beds = EISFactory.InstanceWindows.GetBedsByVista(vistaSiteId);

            List<DC.OccupiedBedInfo> occupiedBeds = BMSFactory.BedManagerQueryClientWindows.GetOccupiedBeds(vistaSiteId).ToList();
            List<OccupiedBedInfo> result = new List<OccupiedBedInfo>();
            Bed bed = null;
            Ward ward = null;
            foreach (DC.OccupiedBedInfo obi in occupiedBeds)
            {
                if (obi.BedUID != null && !string.IsNullOrEmpty(obi.BedUID.ToString()) && obi.BedUID != Guid.Empty)
                {
                    bed = beds.Where(b => b.Id.extension.Equals(obi.BedUID.ToString(), StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (bed == null)
                        bed = EISFactory.InstanceWindows.GetBed(new II(vistaSiteId.root, obi.BedUID.ToString()), vistaSiteId);
                }
                else
                    bed = null;
                ward = wards.Where(b => b.Id.extension.Equals(obi.WardUID.ToString(), StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                if (ward == null)
                    ward = EISFactory.InstanceWindows.GetWard(new II(vistaSiteId.root, obi.WardUID.ToString()));
                result.Add(new OccupiedBedInfo()
                {
                    ActIEN = obi.ActIEN,
                    PatientUID = obi.PatientUID.ToString(),
                    BedIEN = (bed != null) ? bed.Ien : null,
                    BedName = (bed != null) ? bed.Name : null,
                    WardIEN = (ward != null) ? ward.Ien : null,
                    WardName = (ward != null) ? ward.Name : null,
                    EnteredDate = obi.EnteredDate,
                    EventDate = obi.EventDate,
                    Id = obi.Id,
                    IsAdmission = obi.IsAdmission
                });
            }
            return result;
        }

        public void DeleteIconAssociation(Guid patientUID)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.DeleteIconAssociation(patientUID);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void ExportWhiteboardReportData(List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Warning[] warnings;
                string[] streamIds;
                string mimeType = string.Empty;
                string encoding = string.Empty;
                string extension = string.Empty;
                ReportDataSource rds = null;
                byte[] bytes;
                List<WhiteboardBedInfo> dataSource = null;
                List<Division> divisionList = null;

                // Setup the report viewer object
                ReportViewer viewer = new ReportViewer();
                viewer.ProcessingMode = ProcessingMode.Local;
                viewer.LocalReport.SetBasePermissionsForSandboxAppDomain(new PermissionSet(PermissionState.Unrestricted));
                viewer.LocalReport.ReportPath = HttpUtility.UrlPathEncode("LocalReportWhiteboard/WardWhiteboard.rdlc");

                foreach (Facility facility in facilities)
                {
                    divisionList = FacadeManager.EntityInterface.GetDivisions(facility.Id).ToList();
                    foreach (Division div in divisionList)
                    {
                        if (!string.IsNullOrEmpty(div.WhiteboardReportPath))
                        {
                            dataSource = GetWhiteboardDataSource(facility, div);
                            if (dataSource != null && dataSource.Count > 0)
                            {
                                rds = new ReportDataSource("DataSet1", dataSource);
                                viewer.LocalReport.DataSources.Clear();
                                viewer.LocalReport.DataSources.Add(rds); // Add datasource here
                                bytes = viewer.LocalReport.Render("WORD", null, out mimeType, out encoding, out extension, out streamIds, out warnings);
                                SaveWhiteboardReportFile(facility, div, bytes);
                            }
                        }
                    }
                }
            }
            catch (Exception) { }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<WhiteboardBedInfo> GetWhiteboardDataSource(Facility facility, Division division)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<IconInfo> iconFlagsList = FacadeManager.IconInterface.GetAllIconsByFacility(Guid.Parse(facility.Id.extension)).Where(a => a.Active == true).ToList();
                List<Ward> wards = FacadeManager.EntityInterface.GetWardsByDivisionId(division.Id).ToList();

                var beds = GetBedOccupancy(wards, division.Id, facility, null, true, null);
                return WhiteboardBedInfo.GetWhiteboardData(iconFlagsList, Constants.GenderColorBlueAndPink, Constants.DISPLAY_PT_FIRST_AND_LAST4_CODE, Constants.STAFF_ATTENDING, beds, true);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SaveWhiteboardReportFile(Facility facility, Division division, byte[] fileBytes)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DateTime date = DateTime.UtcNow;
                TimeZoneInfo tzi = FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(facility.Id);
                if (tzi != null)
                    date = TimeZoneInfo.ConvertTime(date, tzi);

                StringBuilder fileName = new StringBuilder();
                fileName.Append(facility.Code);
                fileName.Append(", ");
                fileName.Append((facility.Address1 != null) ? (!string.IsNullOrEmpty(facility.Address1.city) ? facility.Address1.city : string.Empty) : string.Empty);
                fileName.Append(", ");
                fileName.Append(facility.SiteNumber);
                fileName.Append(" - ");
                fileName.Append(division.Name);
                string key = fileName.ToString();
                fileName.Append(" (");
                fileName.Append(String.Format("{0:F}", date).Replace(':', '.'));
                fileName.Append(").doc");

                string folderPath = HttpUtility.UrlPathEncode(division.WhiteboardReportPath);
                try
                {
                    if (Directory.Exists(folderPath))
                    {
                        try
                        {
                            foreach (string file in Directory.GetFiles(folderPath))
                            {
                                if (Path.GetFileName(file).StartsWith(key))
                                    File.Delete(file);
                            }
                        }
                        catch (Exception ex) { Tracer.TraceException(ex); }

                        FileStream fs = File.Create(Path.Combine(folderPath, fileName.ToString()));
                        fs.Write(fileBytes, 0, fileBytes.Length);
                        fs.Close();
                    }
                    else
                        throw new DirectoryNotFoundException("Whiteboard Report: Directory not found exception - " + folderPath + " for Ward Group " + division.Name);
                }
                catch (Exception ex)
                {
                    Tracer.TraceException(ex);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<BedCleanInfo> FilterBedCleanAssignedAndPending(List<Ward> wardList, string acceptedBy, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder wards = new StringBuilder();
                foreach (Ward obj in wardList)
                {
                    wards.Append(obj.Id.extension);
                    wards.Append(",");
                }
                List<string> iconCodes = new List<string>();
                iconCodes.Add(Constants.BED_AVAILABILITY_STATUS_ISOLATION_CODE);
                iconCodes.Add(Constants.EMS_NOTIFIED);
                Dictionary<string, IconInfo> icons = FacadeManager.IconInterface.GetIconsByCodes(iconCodes);
                IconInfo isolationIcon = icons[Constants.BED_AVAILABILITY_STATUS_ISOLATION_CODE];
                IconInfo emsNotifyIcon = icons[Constants.EMS_NOTIFIED];

                List<BedCleanInfo> bedCleanInfo = BMSFactory.BedManagerQueryClient.FilterBedCleanAssignedAndPending(wards.ToString(), acceptedBy)
                                                        .Select<DC.BedCleanInfo, BedCleanInfo>(dcBC => DateTimeConverter.ConvertDateFromUTC(dcBC.ToFacadeContract(isolationIcon, emsNotifyIcon), facilityId)).ToList();
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                List<Division> divisions = FacadeManager.EntityInterface.GetDivisions(facilityId).ToList();
                IList<Bed> beds = EISFactory.InstanceFromWCF.GetBeds(bedCleanInfo.Select(bi => bi.Bed.Id).Where(ii => ii != null && !string.IsNullOrEmpty(ii.extension) && ii.extension != Guid.Empty.ToString()).ToList(), f.VistaSite.Id).OrderBy(a => a.Name).ToList();
                Division division = null;
                Ward ward = null;
                List<BedCleanInfo> bedsCleanToRemove = new List<BedCleanInfo>();
                foreach (BedCleanInfo bci in bedCleanInfo)
                {
                    Bed bed = beds.Where(b => b.Id.extension.Equals(bci.Bed.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    if (bed != null)
                    {
                        bci.Bed = bed;
                        ward = wardList.Where(a => a.Id.extension.Equals(bci.Ward.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                        if (ward != null && ward.DivisionList != null && ward.DivisionList.Count > 0)
                        {
                            division = divisions.Where(a => a.Id.extension.Equals(ward.DivisionList[0].Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                            if (division != null)
                                bci.WardName = division.Name;
                            else
                                bci.WardName = string.Empty;
                        }
                        else
                            bci.WardName = string.Empty;
                    }
                    else
                        bedsCleanToRemove.Add(bci);

                }
                return bedCleanInfo.Except(bedsCleanToRemove).ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void ManageBedStaff(Bed bed, Division division, string staff, Patient patient, Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrEmpty(staff))
                {
                    try
                    {
                        BMSFactory.BedManagerOperationsClient.DeleteBedStaff(bed.Id);
                    }
                    catch { };
                }
                else
                    BMSFactory.BedManagerOperationsClient.CreateBedStaff(
                        new DataContracts.BedStaff()
                        {
                            BedId = bed.Id,
                            DivisionId = division == null ? null : division.Id,
                            Staff = staff,
                            PatientId = patient == null ? null : patient.Id,
                            FacilityId = facility == null ? null : facility.Id
                        });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<BedStaffHistory> GetBedStaffHistory(II bedId, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility f = EISFactory.InstanceFromWCF.GetFacility(facilityId);
                Bed bed = GetBed(bedId, f.VistaSite.Id);
                List<BedStaffHistory> result = BMSFactory.BedManagerQueryClient.GetBedStaffHistory(bedId).Select<DC.BedStaffHistory, BedStaffHistory>(a => DateTimeConverter.ConvertDateFromUTC(a.ToFacadeContract(), facilityId)).ToList();
                result.ForEach(b => b.Bed.Name = bed.Name);
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SendCleanDirtyBedNotifications(BedClean bedClean)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            string to = string.Empty;
            try
            {
                int mailsSent = 0;
                List<Facility> facilities = null;
                IList<Division> divisions = EISFactory.InstanceFromWCF.GetDivisionsForBed(bedClean.Bed.Id, bedClean.VistaSite.Id);
                Bed bed = EISFactory.InstanceFromWCF.GetBed(bedClean.Bed.Id, bedClean.VistaSite.Id);
                IList<Ward> wards = null;
                if (bed.WardList != null && bed.WardList.Count > 0)
                    wards = EISFactory.InstanceFromWCF.GetWards(bed.WardList.Select<Ward, II>(wrd => wrd.Id).ToList());

                if (divisions != null)
                {
                    facilities = divisions.Select<Division, Facility>(division => new Facility() { Id = division.Facility.Id }).ToList();
                    List<Module> flags = null;
                    List<Facility> tempFacility = new List<Facility>();
                    foreach (Facility f in facilities)
                    {
                        flags = Facade.FacadeManager.ConfigurationInterface.GetModules(f.Id.root, f.Id).ToList();
                        if (flags.Where(a => a.Code.Equals(Constants.EMS_MODULE, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().CurrentlyInUse.code.Equals(Constants.No, StringComparison.InvariantCultureIgnoreCase))
                            tempFacility.Add(f);
                    }
                    if (tempFacility != null || tempFacility.Count > 0)
                        foreach (Facility f in tempFacility)
                            facilities.Remove(f);
                    if (facilities == null || facilities.Count == 0)
                        return;
                    string domainId = bed.Id.root;
                    //get TimeZoneInfo of the VistA Site
                    VistASettings vistaSettings = Facade.FacadeManager.ConfigurationInterface.GetVistASettings(bedClean.VistaSite.Id);
                    TimeZoneInfo timeZoneInfo;
                    if (vistaSettings != null && vistaSettings.SiteTimeZone != null)
                        timeZoneInfo = TimeZoneInfo.FindSystemTimeZoneById(vistaSettings.SiteTimeZone.displayName);
                    else
                        timeZoneInfo = TimeZoneInfo.Local;
                    foreach (Facility f in facilities)
                    {
                        FacilitySettings settings = Facade.FacadeManager.ConfigurationInterface.GetFacilitySettings(f.Id);
                        List<EMSNotification> notifications = Facade.FacadeManager.ConfigurationInterface.GetEMSNotifications(domainId, f.Id);

                        string from = settings.EMSMailSender;

                        foreach (EMSNotification notification in notifications)
                        {
                            if (divisions.Where(a => a.Name.Equals(notification.Location, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                            {
                                if (!bedClean.CompletedDate.HasValue)
                                {
                                    if (notification.BedControllerDirty.code == Constants.Yes)
                                        to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.BedControllerEmail;
                                    if (notification.EMSDirty.code == Constants.Yes)
                                        to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.EMSEmail;
                                    if (notification.VistaGroupDirty.code == Constants.Yes)
                                        to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.VistaGroupMail;
                                }
                                else
                                {
                                    if (notification.BedControllerCleaned.code == Constants.Yes)
                                        to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.BedControllerEmail;
                                    if (notification.EMSCleaned.code == Constants.Yes)
                                        to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.EMSEmail;
                                    if (notification.VistaGroupCleaned.code == Constants.Yes)
                                        to += (string.IsNullOrEmpty(to) ? "" : ",") + notification.VistaGroupMail;
                                }
                            }
                        }

                        if (!string.IsNullOrWhiteSpace(to) && !string.IsNullOrWhiteSpace(from))
                        {
                            //compose and send the actual message
                            string subject = bedClean.CompletedDate.HasValue ? "Bed cleaned" : "Dirty bed";
                            string bodyFormat = "RoomBed: {0}" + Environment.NewLine +
                                                "Vista Ward: {1}" + Environment.NewLine +
                                                (bedClean.CompletedDate.HasValue ? "Cleaned by: {4}" + Environment.NewLine : string.Empty) +
                                                "Date/time: {2}" + Environment.NewLine +
                                                "Isolation: {3}";

                            string isolation = Constants.No;
                            DataContracts.BedUnavailable bu = null;
                            try
                            {
                                bu = BMSFactory.BedManagerOperationsClientFromWCF.GetBedUnavailable(null, bedClean.Bed.Id);
                            }
                            catch { }
                            if (bu != null && bu.Reason.code.Contains(Constants.ISOLATION))
                                isolation = Constants.Yes;

                            DateTime dt = bedClean.RequestedDate.HasValue ? DateTime.SpecifyKind(bedClean.RequestedDate.Value, DateTimeKind.Unspecified) : DateTime.UtcNow;
                            string body = string.Format(bodyFormat,
                                bed.Name,
                                (wards != null && wards.Count > 0) ? String.Join("; ", wards.Select<Ward, string>(w => w.Name)) : string.Empty,
                                TimeZoneInfo.ConvertTimeFromUtc(dt, timeZoneInfo).ToString("MM/dd/yy hh:mm"),
                                isolation,
                                bedClean.CompletedBy);

                            BMS.Utils.SendMail.Execute(from, to, subject, body, ConfigurationManager.AppSettings[Constants.SMTP_HOST], 25, null);
                            mailsSent++;
                        }
                    }
                }

                if (mailsSent > 0)
                {
                    bedClean.EMSNotify = bedClean.CompletedDate.HasValue ? null : Constants.Yes;
                    //update bedclean to include this flag
                    BMS.DataContracts.BedCleaningOperation scBedClean = bedClean.ToDataContract();
                    BMSFactory.BedManagerOperationsClientFromWCF.UpdateBedCleaningOperation(scBedClean);
                }
            }
            catch (Exception e)
            {
                Tracer.TraceMessage("SendMail To Field: " + to);
                Tracer.TraceException(e);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
