﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using VI = BMS.VistaIntegration.FacadeContracts;

namespace BMS.Facade.Service
{
    /// <summary>
    /// Interface used for configuration operations.
    /// </summary>
    public interface IConfigurationOperations
    {
        #region Admin

        IList<VI.VistASite> GetVistaSites(string domainId);
        IList<VI.Admin.JobCalendar> GetDefinedCalendars(string domainId);
        void SaveCalendar(VI.Admin.JobCalendar calendar, string domainId);
        void DeleteCalendar(string schedulerId, string domainId);
        void SaveVistASite(VI.VistASite vistaSite, string domainId);
        void DeleteVistaASite(string domainId, string vistSiteId);
        IList<NumiEntry> GetNumiEntries(string domainId);
        void SaveNumiEntry(NumiEntry numiEntry, string domainId);
        void DeleteNumiEntry(string schedulerId, string domainId);

        List<WhiteboardReportConfigEntry> GetWhiteboardReportEntries(string domainId);
        VI.Admin.JobCalendar GetFacilityWhiteboardReportScheduler(string domainId, II facilityId);
        void SaveWhiteboardReportEntry(WhiteboardReportConfigEntry whiteboardReportEntry, string domainId);
        void DeleteWhiteboardReportEntry(string schedulerId, string domainId);
        void SaveFacilityWhiteboardReportScheduler(string domainId, string newSchedulerId, string oldSchedulerId, II facilityId);

        #endregion

        #region EMSNotification

        /// <summary>
        /// Gets the EMS notifications list.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// EMS Notification list.
        /// </returns>
        List<EMSNotification> GetEMSNotifications(string domainId, II facilityId);

        /// <summary>
        /// Gets an EMS notification defined for a location.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// EMS notification object.
        /// </returns>
        EMSNotification GetEMSNotification(string location, string domainId, II facilityId);

        /// <summary>
        /// Saves and EMS notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="isNew">The is new flag.</param>
        /// <param name="facilityId">The facility id.</param>
        void SaveEMSNotification(EMSNotification notification, string domainId, bool isNew, II facilityId);

        /// <summary>
        /// Deletes an EMS notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        void DeleteEMSNotification(EMSNotification notification, string domainId, II facilityId);

        /// <summary>
        /// Checks whether an EMS Notification defined for the provided location exists.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// True or False
        /// </returns>
        bool EMSNotificationExists(string location, string domainId, II facilityId);

        #endregion

        #region EventNotification

        /// <summary>
        /// Gets the event notifications list.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// Event notification list.
        /// </returns>
        List<EventNotification> GetEventNotifications(string domainId, II facilityId);

        /// <summary>
        /// Gets an event notification defined for an event.
        /// </summary>
        /// <param name="eventName">The event.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// Event notification object.
        /// </returns>
        IList<EventNotification> GetEventNotification(string wardGroupName, string domainId, II facilityId);

        /// <summary>
        /// Saves an event notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="isNew">The is new flag.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <param name="wardGroupName">The ward group.</param>
        void SaveEventNotification(IList<EventNotification> eventsNotifications, string domainId, bool isNew, II facilityId, string wardGroupName);

        /// <summary>
        /// Deletes an event notification configuration.
        /// </summary>
        /// <param name="notification">The notification.</param>
        /// <param name="domainId">The domain id.</param>
        void DeleteEventNotification(string domainId, II facilityId, string wardGroupName);

        /// <summary>
        /// Checks whether an Event Notification defined for the provided event name exists.
        /// </summary>
        /// <param name="eventName">The name of the event.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// True or False
        /// </returns>
        bool EventNotificationExists(string wardGroupName, string domainId, II facilityId);

        #endregion

        #region Settings

        /// <summary>
        /// Gets the list of sister sites groups.
        /// </summary> 
        /// <returns>A list of sister sites groups.</returns>
        IList<SisterSitesGroup> GetSisterSites();

        /// <summary>
        /// Saves a group of sister sites.
        /// </summary> 
        /// <param name="group">A group of sister sites.</param> 
        void SaveSisterSites(int recordNumber, SisterSitesGroup group);

        /// <summary>
        /// Retrieves the profile for a specific user. 
        /// </summary>
        /// <returns>An user.</returns>
        User GetUserProfile(string username, string domainId);

        /// <summary>
        /// Saves the user profile.
        /// </summary>
        /// <param name="user">The user profile to be saved.</param>
        void SaveUserProfile(User user);

        /// <summary>
        /// Retrieves the settings defined for a facility.
        /// </summary> 
        /// <param name="facilityId">The internal identifier of the facility.</param>
        /// <returns>Facility Settings object.</returns>
        FacilitySettings GetFacilitySettings(II facilityId);

        /// <summary>
        /// Saves into configuration section the settings for a facility.
        /// </summary>
        /// <param name="facilityId">The internal id of the facility.</param> 
        /// <param name="facilitySettings">The facility settings.</param>
        void SaveFacilitySettings(II facilityId, FacilitySettings facilitySettings);

        #endregion

        #region Bed Board Module

        /// <summary>
        /// Gets the modules of BMS.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns>
        /// List of application modules.
        /// </returns>
        IList<Module> GetModules(string domainId, II facilityId);

        /// <summary>
        /// Adds or edits the module.
        /// </summary>
        /// <param name="module">The module object.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        void AddEditModule(Module module, string domainId, II facilityId);

        /// <summary>
        /// Deletes the module.
        /// </summary>
        /// <param name="module">The module object.</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityId">The facility id.</param>
        void DeleteModule(Module module, string domainId, II facilityId);

        #endregion

        /// <summary>
        /// Gets the records number per page.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Records number per page</returns>
        int GetRecordsNumberPerPage(string domainId);
        
        #region Marquee
        /// <summary>
        /// Gets the marquee.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <returns>Marquee text or empty string.</returns>
        string GetMarquee(string domainId);

        /// <summary>
        /// Saves the marquee.
        /// </summary>
        /// <param name="marquee">The marquee text.</param>
        /// <param name="domainId">The domain id.</param>
        void SaveMarquee(string marquee, string domainId);
        #endregion

        #region Evacuation


        /// <summary>
        /// Gets the is evacuation.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityExtension">The facility extension.</param>
        /// <returns></returns>
        bool GetIsEvacuation(string domainId, string facilityExtension);

        /// <summary>
        /// Saves the is evacuation.
        /// </summary>
        /// <param name="isEvacuation">if set to <c>true</c> [is evacuation].</param>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityExtension">The facility extension.</param>
        void SaveIsEvacuation(bool isEvacuation, string domainId, string facilityExtension);

        #endregion

        #region VistASettings
        VistASettings GetVistASettings(II vistAId);
        void SaveVistASettings(VistASettings settings, II vistAId);
        #endregion

        #region VISN Bed Summary Report

        /// <summary>
        /// Gets the medical center ids. Return a string with format: facilityExetension1,medicalCenterId1;.....;facilityExetensionN,medicalCenterIdN.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityExtension">The facility extension.</param>
        /// <returns></returns>
        //string GetMedicalCenterIds(string domainId, IList<string> facilityExtension);

        /// <summary>
        /// Gets the medical center ids. Return a string with format: facilityExetension1,offset1,medicalCenterId1;.....;facilityExetensionN,offsetN,medicalCenterIdN.
        /// </summary>
        /// <param name="domainId">The domain id.</param>
        /// <param name="facilityExtension">The facility extension.</param>
        /// <returns></returns>
        string GetFacMedicalCenterIds(string domainId, IList<II> facilityIIList);

        #endregion

        /// <summary>
        /// Gets the configurations for home page.
        /// </summary>
        /// <param name="facilityId">The facility id.</param>
        /// <param name="isEvacuation">if set to <c>true</c> [is evacuation].</param>
        /// <param name="marqueeText">The marquee text.</param>
        /// <param name="facilitySettings">The facility settings.</param>
        /// <param name="flags">The flags.</param>
        void GetConfigurationsForHomePage(II facilityId, out bool isEvacuation, out string marqueeText, out FacilitySettings facilitySettings, out List<Module> flags);
    }
}
