﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using System.Linq.Expressions;
using System.Web.Mvc.Html;
using Microsoft.Web.Mvc;
using VistATool.Utils;
using System.Reflection;

namespace VistATool.Views.Shared
{
    public static class HtmlHelpers
    {
        public static string MemberName<T, V>(this Expression<Func<T, V>> expression)
        {
            var memberExpression = expression.Body as MemberExpression;
            if (memberExpression == null)
                throw new InvalidOperationException("Expression must be a member expression");

            return memberExpression.Member.Name;
        }

        /// <summary>
        /// a html helper for rendering a control for data entering
        /// </summary>
        /// <typeparam name="TModel">the type of model to bind to</typeparam>
        /// <typeparam name="TProperty">the type of the property of the model to bind to</typeparam>
        /// <param name="htmlHelper">the html helper used to render the control</param>
        /// <param name="expression">the expression representing the model and property to bind to</param>
        /// <returns>a html representing a text box and a button to open a calendar</returns>
        public static MvcHtmlString CustomDatePickerFor<TModel, TProperty>(this HtmlHelper<TModel> htmlHelper,
                Expression<Func<TModel, TProperty>> expression)
        {
            return HtmlHelpers.CustomDatePickerFor(htmlHelper, expression, null);
        }

        /// <summary>
        /// a html helper for rendering a control for data entering
        /// </summary>
        /// <typeparam name="TModel">the type of model to bind to</typeparam>
        /// <typeparam name="TProperty">the type of the property of the model to bind to</typeparam>
        /// <param name="htmlHelper">the html helper used to render the control</param>
        /// <param name="expression">the expression representing the model and property to bind to</param>
        /// <param name="htmlAttributes">other html atribubes</param>
        /// <returns>a html representing a text box and a button to open a calendar</returns>
        public static MvcHtmlString CustomDatePickerFor<TModel, TProperty>(this HtmlHelper<TModel> htmlHelper,
            Expression<Func<TModel, TProperty>> expression,
            HtmlAttributes htmlAttributes)
        {
            if (htmlAttributes == null)
                htmlAttributes = new HtmlAttributes();

            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributesForDateTime();

            if (!htmlAttributes.IsEnabled)
                return InputExtensions.TextBoxFor<TModel, TProperty>(htmlHelper, expression, newAttributes);

            var res = InputExtensions.TextBoxFor<TModel, TProperty>(htmlHelper, expression, newAttributes).ToHtmlString()
                + htmlAttributes.CalendarButton;
            return MvcHtmlString.Create(res);
        }

        /// <summary>
        /// a html helper for rendering a combo for hour and anothe for minutes
        /// </summary>
        /// <typeparam name="TModel">the type of model to bind to</typeparam>
        /// <typeparam name="TProperty">the type of the property of the model to bind to</typeparam>
        /// <param name="htmlHelper">the html helper used to render the control</param>
        /// <param name="hExpression">the expression representing the model and property of the hour part to bind to</param>
        /// <param name="mExpression">the expression representing the model and property of the minute part to bind to</param>
        /// <returns>a html representing a combo for hour and a combo fot minutes</returns>
        public static MvcHtmlString CustomTimePickerFor<TModel, TProperty>(this HtmlHelper<TModel> htmlHelper,
                Expression<Func<TModel, TProperty>> hExpression, Expression<Func<TModel, TProperty>> mExpression, HtmlAttributes htmlAttributes)
        {
            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributes();

            var memberH = (MemberExpression)hExpression.Body;
            var memberM = (MemberExpression)mExpression.Body;
            var param = Expression.Parameter(typeof(TProperty), "value");
            var setH = Expression.Lambda<Action<TModel, TProperty>>(
                Expression.Assign(memberH, param), hExpression.Parameters[0], param);
            var setM = Expression.Lambda<Action<TModel, TProperty>>(
                Expression.Assign(memberM, param), mExpression.Parameters[0], param);

            // compile it
            var actionH = setH.Compile();
            var actionM = setM.Compile();
            var model = htmlHelper.ViewData.Model;
            TProperty propertyH = hExpression.Compile()(model);
            TProperty propertyM = mExpression.Compile()(model);
            
            //should be string, but double check
            //set the hour value to the D2 format if not set already
            if (propertyH is string)
            {
                string value = (string)Convert.ChangeType(propertyH, typeof(string));
                if (value.Length == 1)
                {
                    value = "0" + value;
                    actionH(model, (TProperty)Convert.ChangeType(value, typeof(TProperty)));
                }
            }
            //set the minute value to the D2 format if not set already
            if (propertyM is string)
            {
                string value = (string)Convert.ChangeType(propertyM, typeof(string));
                if (value.Length == 1)
                {
                    value = "0" + value;
                    actionM(model, (TProperty)Convert.ChangeType(value, typeof(TProperty)));
                }
            }

            string res = SelectExtensions.DropDownListFor(htmlHelper, hExpression, new SelectList(TimeLists.HourList), newAttributes).ToString() +
             ":" + SelectExtensions.DropDownListFor(htmlHelper, mExpression, new SelectList(TimeLists.MinuteList), newAttributes).ToString();

            return MvcHtmlString.Create(res);
        }

        /// <summary>
        /// a html helper for rendering a combo box
        /// </summary>
        /// <typeparam name="TModel">the type of model to bind to</typeparam>
        /// <typeparam name="TProperty">the type of the property of the model to bind to</typeparam>
        /// <param name="htmlHelper">the html helper used to render the control</param>
        /// <param name="expression">the expression representing the model and property to bind to</param>
        /// /// <param name="htmlAttributes">other html atribubes</param>
        /// <returns>a html representing a combo box </returns>
        public static MvcHtmlString CustomDropDownListFor<TModel, TProperty>(this HtmlHelper<TModel> htmlHelper,
            Expression<Func<TModel, TProperty>> expression,
            IEnumerable<SelectListItem> selectList, HtmlAttributes htmlAttributes)
        {
            if (htmlAttributes == null)
                return SelectExtensions.DropDownListFor<TModel, TProperty>(htmlHelper, expression, selectList);

            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributes();
            return SelectExtensions.DropDownListFor<TModel, TProperty>(htmlHelper, expression, selectList, newAttributes);
        }

        /// <summary>
        /// a html helper for rendering a combo box
        /// </summary>
        /// <typeparam name="TModel">the type of model to bind to</typeparam>
        /// <typeparam name="TProperty">the type of the property of the model to bind to</typeparam>
        /// <param name="htmlHelper">the html helper used to render the control</param>
        /// <param name="expression">the expression representing the model and property to bind to</param>
        /// <param name="optionLabel">the optional label</param>
        /// <param name="htmlAttributes">other html atribubes</param>
        /// <returns>a html representing a combo box </returns>
        public static MvcHtmlString CustomDropDownListFor<TModel, TProperty>(this HtmlHelper<TModel> htmlHelper, Expression<Func<TModel, TProperty>> expression, IEnumerable<SelectListItem> selectList, string optionLabel, HtmlAttributes htmlAttributes)
        {
            if (htmlAttributes == null)
                return SelectExtensions.DropDownListFor<TModel, TProperty>(htmlHelper, expression, selectList, optionLabel);

            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributes();
            return SelectExtensions.DropDownListFor<TModel, TProperty>(htmlHelper, expression, selectList, optionLabel, newAttributes);

        }

        public static MvcHtmlString CustomTextBoxFor<TModel, TProperty>(this HtmlHelper<TModel> htmlHelper, Expression<Func<TModel, TProperty>> expression, HtmlAttributes htmlAttributes)
        {
            if (htmlAttributes == null)
                return InputExtensions.TextBoxFor<TModel, TProperty>(htmlHelper, expression, htmlAttributes);

            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributes();
            return InputExtensions.TextBoxFor<TModel, TProperty>(htmlHelper, expression, newAttributes);
        }

        public static MvcHtmlString CustomSubmitButton(this HtmlHelper helper, string name, string buttonText, HtmlAttributes htmlAttributes)
        {
            if (htmlAttributes == null)
                return ButtonsAndLinkExtensions.SubmitButton(helper, name, buttonText);

            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributes();
            return ButtonsAndLinkExtensions.SubmitButton(helper, name, buttonText, newAttributes);
        }

        public static MvcHtmlString CustomCheckBoxFor<TModel>(this HtmlHelper<TModel> htmlHelper, Expression<Func<TModel, bool>> expression, HtmlAttributes htmlAttributes)
        {
            if (htmlAttributes == null)
                return InputExtensions.CheckBoxFor<TModel>(htmlHelper, expression);

            if (!htmlAttributes.IsVisible)
                return MvcHtmlString.Create(null);

            var newAttributes = htmlAttributes.BuildAttributes();
            return InputExtensions.CheckBoxFor<TModel>(htmlHelper, expression, newAttributes);

        }

        /// <summary>
        /// a class for specifying the html atributes for controls
        /// </summary>
        public class HtmlAttributes
        {
            public HtmlAttributes(string CssClass)
                : this()
            {
                this.CSSClass = CssClass;
            }

            public HtmlAttributes()
            {
                IsVisible = true;
                IsEnabled = true;
            }

            /// <summary>
            /// the html css attribute
            /// </summary>
            public string CSSClass { get; set; }
            /// <summary>
            /// the html style attribute
            /// </summary>
            public string Style { get; set; }
            /// <summary>
            /// the html disabled attribute
            /// </summary>
            public bool IsEnabled { get; set; }
            /// <summary>
            /// if true the control is not rendered
            /// </summary>
            public bool IsVisible { get; set; }

            internal IDictionary<string, object> BuildAttributes()
            {
                var newAttributes = new Dictionary<string, object>();
                if (!this.IsEnabled)
                    newAttributes.Add("disabled", "disabled");
                if (!string.IsNullOrWhiteSpace(this.CSSClass))
                    newAttributes.Add("class", this.CSSClass);
                if (!string.IsNullOrWhiteSpace(this.Style))
                    newAttributes.Add("style", this.Style);

                return newAttributes;
            }
            internal IDictionary<string, object> BuildAttributesForDateTime()
            {
                var newAttributes = new Dictionary<string, object>();

                if (!string.IsNullOrWhiteSpace(this.Style))
                    newAttributes.Add("style", this.Style);

                if (!this.IsEnabled)
                    newAttributes.Add("disabled", "disabled");
                else
                    newAttributes.Add("class", ControlName);

                return newAttributes;
            }

            string _ControlName = null;
            private string ControlName
            {
                get
                {
                    if (_ControlName == null)
                        _ControlName = "datep" + nameSufix++;
                    return _ControlName;
                }
            }

            internal string CalendarButton
            {
                get
                {
                    return "<script language=\"text/javascript\" type=\"text/javascript\">$(document).ready(function () {$('."
                       + ControlName + "').datepicker({showOn: 'button', dateFormat: 'mm/dd/yy' }); });</script>";
                }
            }
            private static int nameSufix = 1;
        }
    }
}