﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using Microsoft.Web.Mvc;
using InfoWorld.HL7.ITS;
using BMS.Web.Models;
using BMS.Facade.Data;
using BMS.Facade;
using BMS.Web.App_GlobalResource;
using BMS.Utils;
using BMS.ServicesWrapper.WF;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.Utils.Properties;
using System.Text.RegularExpressions;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Controller class for the entering/tracking/transfering patients screen.
    /// </summary>
    [ValidateInput(false)]
    public class TransferController : BaseController
    {
        #region Get Methods

        /// <summary>
        /// Default action for the TransferController.
        /// </summary>
        /// <returns>Returns the appropriate view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TransferSelectPatientViewModel model = new TransferSelectPatientViewModel();
                model.VisnExtension = QueryStrings["visn"];
                model.VisnRoot = loggedUser.Domain;
                model.FacilityIdParam = QueryStrings["facility"];
                model.IsGoBackHomePage = bool.Parse(QueryStrings["isGoBackHomePage"]);
                model.LoggedUser = loggedUser;
                FillFacilityList(model);
                model.SelectedFacilityId = Strings.ResponsibleFacility;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Add the transfer
        /// </summary>
        /// <param name="visnNumber">The visn number.</param>
        /// <param name="message">The message.</param>
        /// <returns>Returns the appropriate view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult AddTranfer(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TransferViewModel model = new TransferViewModel();
                FillBedBoardParameter(model, QueryStrings["visn"], loggedUser.Domain, QueryStrings["facility"], bool.Parse(QueryStrings["isGoBackHomePage"]));
                model.Message = QueryStrings["message"];

                //check to see if user has entered anything
                bool isFacilityError;
                if (!string.IsNullOrEmpty(QueryStrings["isFacilityError"]) && bool.TryParse(QueryStrings["isFacilityError"], out isFacilityError))
                {
                    if (isFacilityError)
                    {
                        // facility not selected
                        model.DisplayFacilityMessage = Strings.DisplayBlock;
                        model.DisplaySSNMessage = Strings.DisplayNone;
                        model.StyleDisabled = Strings.Disabled;
                        model.IsAddTransfer = false;
                    }
                    else
                    {
                        if (QueryStrings["message"].Equals(Strings.YouMustEnterAValidSSNWithoutDashes, StringComparison.InvariantCultureIgnoreCase))
                        {
                            // SSN not in correct format
                            model.DisplayFacilityMessage = Strings.DisplayNone;
                            model.DisplaySSNMessage = Strings.DisplayBlock;
                            model.IsAddTransfer = false;
                        }
                        else
                        {
                            // SSN not found in the DB => allow the user to enter patient name and save the new patient to the DB
                            model.DisplayFacilityMessage = Strings.DisplayNone;
                            model.DisplaySSNMessage = Strings.DisplayNone;
                            model.StyleDisabled = string.Empty;
                            model.IsAddTransfer = true;
                            model.ExtensionFacilityId = QueryStrings["selectedFacility"];
                            model.RootFacilityId = loggedUser.Domain;
                            model.ExtensionPatientId = QueryStrings["patient"];
                            model.RootPatientId = Constants.SSNROOT;
                            PopulatelAndInitModel(model);
                        }
                    }
                }
                else
                {
                    // SSN found in the DB
                    model.IsAddTransfer = true;
                    model.DisplayFacilityMessage = Strings.DisplayNone;
                    model.DisplaySSNMessage = Strings.DisplayNone;
                    model.ExtensionFacilityId = QueryStrings["selectedFacility"];
                    model.RootFacilityId = loggedUser.Domain;
                    model.ExtensionPatientId = QueryStrings["patient"];
                    model.RootPatientId = loggedUser.Domain;
                    PopulatelAndInitModel(model);
                    if (!model.IsVistaValidated)
                        model.StyleDisabled = string.Empty;
                    else
                        model.StyleDisabled = Strings.Disabled;
                }
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Edits the transfer.
        /// </summary>
        /// <param name="transferIdRoot">The transfer id root.</param>
        /// <param name="transferIdExtension">The transfer id extension.</param>
        /// <param name="ien">The ien.</param>
        /// <param name="facilityIdParam">The facility id param.</param>
        /// <param name="selectedSortParam">The selected sort param.</param>
        /// <param name="currentPageParam">The current page param.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult EditTransfer(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TransferViewModel model = new TransferViewModel();
                FillBedBoardParameter(model, QueryStrings["visn"], loggedUser.Domain, QueryStrings["facility"], false);
                PopulateModel(model);
                model.TransferRequestId = new II(loggedUser.Domain, QueryStrings["transferId"]);
                model.TransferRequestIdExtension = QueryStrings["transferId"];
                model.TransferRequestIdRoot = loggedUser.Domain;
                Transfer transfer = FacadeManager.ADTInterface.GetTransferById(model.TransferRequestId, loggedUser);
                TranslateTransferToModel(transfer, model);
                return View("EditTransfer", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Finalize the transfer.
        /// </summary>
        /// <param name="transferIdRoot">The transfer id root.</param>
        /// <param name="transferIdExtension">The transfer id extension.</param>
        /// <param name="ien">The ien.</param>
        /// <param name="facilityIdParam">The facility id param.</param>
        /// <param name="selectedSortParam">The selected sort param.</param>
        /// <param name="currentPageParam">The current page param.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult FinalizeTransfer(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TransferViewModel model = new TransferViewModel();
                FillBedBoardParameter(model, QueryStrings["visn"], loggedUser.Domain, QueryStrings["facility"], false);
                model.TransferRequestId = new II(loggedUser.Domain, QueryStrings["transferId"]);
                model.TransferRequestIdExtension = QueryStrings["transferId"];
                model.TransferRequestIdRoot = loggedUser.Domain;
                PopulateModel(model);
                if (model.FacilityList == null)
                    model.FacilityList = new List<Facility>();
                model.FacilityList.Insert(0, new Facility() { Id = new II(string.Empty, (default(Guid)).ToString()), Code = string.Empty });
                Transfer transfer = FacadeManager.ADTInterface.GetTransferById(new II(loggedUser.Domain, QueryStrings["transferId"]), loggedUser);
                TranslateTransferToModel(transfer, model);
                return View("FinalizeTransfer", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Post Methods

        /// <summary>
        /// Default submit action for the BedBoard screen.
        /// </summary>
        /// <param name="input">The model.</param>
        /// <returns>Returns the appropriate view result.</returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(TransferSelectPatientViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = FacadeManager.EntityInterface.GetFacility(new II(input.VisnRoot, input.SelectedFacilityId));
                if (!string.IsNullOrEmpty(input.SelectedPatientId))
                {
                    Patient patient = FacadeManager.EntityInterface.GetPatientBySsn(new II(Constants.SSNROOT, input.SelectedPatientId), null);
                    return this.RedirectToAction<TransferController>(act => act.AddTranfer(EncryptQueryString(new string[] { "visn", "facility", "message", "isFacilityError", "selectedFacility", "patient", "isGoBackHomePage" }, new string[] { input.VisnExtension, input.FacilityIdParam, "Add Transfer", null, facility.Id.extension, patient.Id.extension, input.IsGoBackHomePage.ToString() }, loggedUser.Salt)));
                }

                Regex regexSSN = new Regex(@"^\d{3}-\d{2}-\d{4}$");
                Regex regexSSNWithoutDashes = new Regex(@"^\d{3}\d{2}\d{4}$");
                Regex regexFirstLastNameFourSSN = new Regex(@"^([A-Za-z]){1}\d{4}$");
                Regex regexLastName = new Regex(@"^[a-zA-Z]+$");
                IList<Patient> patientList = null;
                input.LoggedUser = loggedUser;
                input.PatientSearchText = input.PatientSearchText.Trim();
                if (input.PatientSearchText.Length > 2)
                {
                    if (!regexSSN.IsMatch(input.PatientSearchText) && !regexSSNWithoutDashes.IsMatch(input.PatientSearchText))
                    {
                        if (!regexFirstLastNameFourSSN.IsMatch(input.PatientSearchText))
                        {
                            if (!regexLastName.IsMatch(input.PatientSearchText))
                            {
                                ModelState.AddModelError("PatientSearchText", Strings.SSNWasNotInAValidFormat);
                                FillFacilityList(input);
                                return View(input);
                            }
                            else
                                patientList = FacadeManager.EntityInterface.GetPatientsByLastName(input.PatientSearchText, null);
                        }
                        else
                            patientList = FacadeManager.EntityInterface.GetPatientsByLastNameInitialAndLastFourSSN(input.PatientSearchText, null);
                    }
                    else
                    {
                        string ssn = input.PatientSearchText;
                        if (!input.PatientSearchText.Contains('-'))
                            ssn = string.Format("{0}-{1}-{2}", ssn.Substring(0, 3), ssn.Substring(3, 2), ssn.Substring(5, 4));
                        Patient patient = FacadeManager.EntityInterface.GetPatientBySsn(new II(Constants.SSNROOT, ssn), null);
                        if (patient != null)
                        {
                            patientList = new List<Patient>();
                            patientList.Add(patient);
                        }
                        else
                            return this.RedirectToAction<TransferController>(act => act.AddTranfer(EncryptQueryString(new string[] { "visn", "facility", "message", "isFacilityError", "selectedFacility", "patient", "isGoBackHomePage" }, new string[] { input.VisnExtension, input.FacilityIdParam, "Add Transfer", false.ToString(), facility.Id.extension, ssn, input.IsGoBackHomePage.ToString() }, loggedUser.Salt)));
                    }
                }

                if (patientList != null && patientList.Count > 0)
                {
                    if (patientList.Count == 1)
                        return this.RedirectToAction<TransferController>(act => act.AddTranfer(EncryptQueryString(new string[] { "visn", "facility", "message", "isFacilityError", "selectedFacility", "patient", "isGoBackHomePage" }, new string[] { input.VisnExtension, input.FacilityIdParam, "Add Transfer", null, facility.Id.extension, patientList[0].Id.extension, input.IsGoBackHomePage.ToString() }, loggedUser.Salt)));
                    else
                    {
                        input.PatientList = patientList.OrderBy(patient => patient.LastName).ThenBy(patient => patient.FirstName).ThenBy(patient => patient.SSN.extension).ToList();
                        FillFacilityList(input);
                        return View(input);
                    }
                }
                else
                {
                    if (input.PatientSearchText.Length > 2)
                    {
                        ModelState.AddModelError("PatientNotFound", Strings.SSNNotFound);
                        FillFacilityList(input);
                        return View(input);
                    }
                    else
                    {
                        ModelState.AddModelError("PatientSearchText", Strings.SearchStringToShort);
                        FillFacilityList(input);
                        return View(input);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Add the tranfer.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="buttonCancel">The button cancel.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult AddTranfer(TransferViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!FacadeManager.UserInterface.CheckBMSWritePermission(this.loggedUser, new II(model.RootFacilityId, model.ExtensionFacilityId)))
                    return BaseController.UnauthorizedWriteRequestResult(FacadeManager.EntityInterface.GetFacility(new II(model.RootFacilityId, model.ExtensionFacilityId)).Name);

                SaveTransfer(model, true);
                FillAddEditResultModel(model);
                return View("AddEditResult", model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult EditTransfer(TransferViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(input.ButtonCancel))
                    return this.RedirectToAction<BedBoardController>(act => act.Index(EncryptQueryString(new string[] { "visn", "facility" }, new string[] { input.VisnExtension, input.FacilityIdParam }, loggedUser.Salt)));
                if (!FacadeManager.UserInterface.CheckBMSWritePermission(this.loggedUser, new II(this.loggedUser.Domain, input.ExtensionFacilityId)))
                    return BaseController.UnauthorizedWriteRequestResult(FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, input.ExtensionFacilityId)).Name);

                SaveTransfer(input, false);
                FillAddEditResultModel(input);
                return View("AddEditResult", input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult FinalizeTransfer(TransferViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!string.IsNullOrEmpty(input.ButtonCancel))
                    return this.RedirectToAction<BedBoardController>(act => act.Index(EncryptQueryString(new string[] { "visn", "facility" }, new string[] { input.VisnExtension, input.FacilityIdParam }, loggedUser.Salt)));
                if (!FacadeManager.UserInterface.CheckBMSWritePermission(this.loggedUser, new II(this.loggedUser.Domain, input.ExtensionFacilityId)))
                    return BaseController.UnauthorizedWriteRequestResult(FacadeManager.EntityInterface.GetFacility(new II(input.RootFacilityId, input.ExtensionFacilityId)).Name);
                if (!FacadeManager.UserInterface.CheckBMSWritePermission(this.loggedUser, new II(this.loggedUser.Domain, input.SelectedAdmissionFacilityId)))
                    return BaseController.UnauthorizedWriteRequestResult(FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, input.SelectedAdmissionFacilityId)).Name);

                if (input.Visn == null)
                {
                    input.Visn = FacadeManager.EntityInterface.GetVisn(new II(input.VisnRoot, input.VisnExtension));
                    input.Region = FacadeManager.EntityInterface.GetRegions().ToList().Where(a => a.Id.extension.Equals(input.Visn.Region.Id.extension)).FirstOrDefault().Number.ToString();
                }

                //check if finalize transfer is valid.
                if (string.IsNullOrWhiteSpace(input.SelectedDispositionDate) || string.IsNullOrWhiteSpace(input.SelectedDispositionCode))
                {
                    input.IsValidFinalizeTranfer = false;
                    input.Message = Strings.YouMustEnterADispositionAndADispositionDate;
                    return View("FinalizeResult", input);
                }

                if (input.SelectedDispositionCode == Constants.OTHER_COMMENT && string.IsNullOrWhiteSpace(input.DischargeComment))
                {
                    input.IsValidFinalizeTranfer = false;
                    input.Message = Strings.YouMustEnterADispositionComment;
                    return View("FinalizeResult", input);
                }

                if (input.SelectedDispositionCode == Constants.VA_ADMISSION && (string.IsNullOrWhiteSpace(input.SelectedAdmissionFacilityId) || input.SelectedAdmissionFacilityId == (default(Guid)).ToString()))
                {
                    input.IsValidFinalizeTranfer = false;
                    input.Message = Strings.YouMustEnterAVAFacility;
                    return View("FinalizeResult", input);
                }
                Patient patient = FacadeManager.EntityInterface.GetPatientById(new II(input.RootPatientId, input.ExtensionPatientId), null);
                PopulateModel(input);
                Facility facility = null;
                if (!string.IsNullOrEmpty(input.SelectedAdmissionFacilityId) && input.SelectedAdmissionFacilityId != (default(Guid)).ToString())
                {
                    facility = input.FacilityList.Where(a => a.Id.extension.Equals(input.SelectedAdmissionFacilityId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                    facility.VistaSite = FacadeManager.EntityInterface.GetVistaSite(facility.VistaSite.Id);
                }

                if (input.SelectedDispositionCode == Constants.VA_ADMISSION && (patient.IENList == null || (patient.IENList != null && patient.IENList.Where(a => a.root == facility.VistaSite.Code).Count() == 0)))
                {
                    input.IsValidFinalizeTranfer = false;
                    input.Message = Strings.TransferPatientNotFoundInVista;
                    return View("FinalizeResult", input);
                }

                //finalize transfer
                input.IsValidFinalizeTranfer = true;


                II transferId = new II(input.TransferRequestIdRoot, input.TransferRequestIdExtension);

                DateTime? dispositionDate = null;
                DateTime tempDate = DateTimeNowOnCurrentFacility;
                if (DateTime.TryParse(input.SelectedDispositionDate, out tempDate))
                    dispositionDate = tempDate;


                Transfer transfer = new Transfer()
                {
                    Id = new II(this.loggedUser.Domain, null),
                    Parent = new Act() { Id = transferId },
                    CreationDate = DateTimeNowOnCurrentFacility,
                    CanceledDate = null,
                    Patient = patient,
                    Disposition = input.DispositionList.Where(a => a.code == input.SelectedDispositionCode).FirstOrDefault(),
                    FacilityTo = facility,
                    DispositionDate = dispositionDate,
                    DischargeComment = input.DischargeComment,
                    Edited = DateTimeNowOnCurrentFacility,
                    EnteredBy = this.loggedUser.UserName,
                    LastEditBy = this.loggedUser.UserName
                };

                II transferEventId = FacadeManager.WorkflowInterface.CreateTransferEvent(transfer, loggedUser.Facility);

                FillFinalizeResultModel(input);
                input.PatientSSN = patient.SSN.extension;
                return View("FinalizeResult", input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        #region Private methods

        private void FillFacilityList(TransferSelectPatientViewModel vm)
        {
            Visn visn = FacadeManager.EntityInterface.GetVisn(new II(vm.VisnRoot, vm.VisnExtension));
            if (visn != null)
            {
                vm.VisnNumber = visn.Number;
                vm.FacilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(visn.Id).Where(facility => facility.IsAlive == true).OrderBy(fac => fac.Code).ToList();
                vm.FacilityList = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, vm.FacilityList.ToList());
            }
            else
            {
                vm.VisnNumber = string.Empty;
                vm.FacilityList = new List<Facility>();
            }
            //set default value for selected facility
            vm.FacilityList.Insert(0, new Facility() { Id = new II() { extension = Strings.ResponsibleFacility, root = Strings.ResponsibleFacility }, Code = Strings.ResponsibleFacility.ToUpper() });
        }

        /// <summary>
        /// Fills the bed board parameters.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="ien">The ien.</param>
        /// <param name="facilityIdParam">The facility id param.</param>
        /// <param name="selectedSortParam">The selected sort param.</param>
        /// <param name="currentPageParam">The current page param.</param>
        private void FillBedBoardParameter(TransferViewModel model, string visnExtension, string visnRoot, string facilityIdParam, bool isGoBackHomePage)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.VisnExtension = visnExtension;
                model.VisnRoot = visnRoot;
                model.Visn = FacadeManager.EntityInterface.GetVisn(new II(visnRoot, visnExtension));
                model.Region = FacadeManager.EntityInterface.GetRegions().ToList().Where(a => a.Id.extension.Equals(model.Visn.Region.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Number.ToString();
                model.FacilityIdParam = facilityIdParam;
                model.IsGoBackHomePage = isGoBackHomePage;
                model.LoggedUser = loggedUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the add/edit result model.
        /// </summary>
        private void FillAddEditResultModel(TransferViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (input.Visn == null)
                    input.Visn = FacadeManager.EntityInterface.GetVisn(new II(input.VisnRoot, input.VisnExtension));
                input.SelectedContractName = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision).First<CD>(conceptDescriptor => conceptDescriptor.code == input.SelectedContractCode).displayName;
                input.SelectedEraName = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.Era).Where(a => a.code == input.SelectedEraCode).FirstOrDefault().displayName;
                CD selectedSpecialty = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.VistaSpecialty).Where(a => a.code == input.SelectedSpecialityCode).FirstOrDefault();
                if (selectedSpecialty != null)
                    input.SelectedSpecialtyName = selectedSpecialty.displayName;
                List<Facility> facilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(input.Visn.Id).ToList();
                input.SelectedFacilityName = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, facilityList).Where(a => a.Id.extension.Equals(input.ExtensionFacilityId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Name;
                input.RequestDateTime = input.RequestedDate.HasValue ? input.RequestedDate.Value.ToString(Strings.USDateTime) : string.Empty;
                if (!string.IsNullOrEmpty(input.PatientGender))
                    input.SelectedGenderName = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Gender).First<CD>(conceptDescriptor => conceptDescriptor.code == input.PatientGender).displayName;
                else
                    input.SelectedGenderName = string.Empty;
                input.LoggedUser = loggedUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the finalize result model.
        /// </summary>
        private void FillFinalizeResultModel(TransferViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (input.Visn == null)
                {
                    input.Visn = FacadeManager.EntityInterface.GetVisn(new II(input.VisnRoot, input.VisnExtension));
                    input.Region = FacadeManager.EntityInterface.GetRegions().ToList().Where(a => a.Id.extension.Equals(input.Visn.Region.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Number.ToString();
                }

                input.SelectedDispositionName = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Disposition).Where(cd => cd.code == input.SelectedDispositionCode).FirstOrDefault().displayName;
                List<Facility> facilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(input.Visn.Id).ToList();
                if (!string.IsNullOrEmpty(input.SelectedAdmissionFacilityId) && input.SelectedAdmissionFacilityId != (default(Guid)).ToString())
                    input.SelectedAdmissionFacilityName = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, facilityList).Where(a => a.Id.extension.Equals(input.SelectedAdmissionFacilityId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Name;
                input.RecordNumber = (new Random()).Next(1000, 9999).ToString();
                input.RequestDateTime = input.RequestedDate.HasValue ? input.RequestedDate.Value.ToString(Strings.USDateTime) : string.Empty;
                input.LoggedUser = loggedUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the transfer data to the model.
        /// </summary>
        /// <param name="transfer">The transfer.</param>
        /// <param name="model">The model.</param>
        private void TranslateTransferToModel(Transfer transfer, TransferViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (transfer.Patient != null)
                {
                    Patient patient = FacadeManager.EntityInterface.GetPatientBySsn(transfer.Patient.SSN, null);
                    if (patient != null)
                    {
                        if (patient.FirstName != null && patient.LastName != null)
                            model.PatientName = String.Format("{0}, {1} {2}", patient.LastName, patient.FirstName, patient.MiddleName).Trim();
                        if (patient.SSN != null)
                            model.PatientSSN = patient.SSN.extension;
                        if (patient.Gender != null)
                            model.PatientGender = (patient.Gender != null) ? patient.Gender.code : string.Empty;
                        model.ExtensionPatientId = patient.Id.extension;
                        model.RootPatientId = patient.Id.root;
                        model.IsVistaValidated = patient.IsVistaValidated;
                        model.SCRating = patient.ServiceConnectedPercentage ?? "";
                        if (!patient.IsVistaValidated)
                            model.StyleDisabled = string.Empty;
                        else
                            model.StyleDisabled = Strings.Disabled;
                    }
                }
                if (transfer.Contract != null)
                    model.SelectedContractCode = transfer.Contract.code;
                model.Diagnosis = transfer.Diagnosis;
                model.CurrentLocation = transfer.CurrentLocation;
                model.LocationAdmissionDate = transfer.AdmissionDate;
                model.Comments = transfer.Comment;
                if (transfer.FacilityFrom != null && transfer.FacilityFrom.Id != null)
                {
                    model.ExtensionFacilityId = transfer.FacilityFrom.Id.extension;
                    model.RootFacilityId = transfer.FacilityFrom.Id.root;
                }
                if (transfer.FacilityTo != null && transfer.FacilityTo.Id != null)
                    model.SelectedAdmissionFacilityId = transfer.FacilityTo.Id.extension;
                model.DischargeComment = transfer.DischargeComment;
                model.DispositionDate = (transfer.DispositionDate == null) ? DateTimeNowOnCurrentFacility : transfer.DispositionDate.Value;
                if (transfer.Era != null)
                    model.SelectedEraCode = transfer.Era.code;
                if (transfer.Speciality != null)
                    model.SelectedSpecialityCode = transfer.Speciality.code;
                model.SelectedRequestedDate = (transfer.RequestedDate.HasValue) ? transfer.RequestedDate.Value.ToShortDateString() : string.Empty;
                model.IsNationalWaitList = transfer.IsNationalWaitList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Populates the model with default data.
        /// </summary>
        /// <param name="model">The model.</param>
        private void PopulateModel(TransferViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Facility> facilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(model.Visn.Id).Where(facility => facility.IsAlive == true).OrderBy(facility => facility.Code).ToList();
                model.FacilityList = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, facilityList.ToList());
                model.ContractList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                model.DispositionList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Disposition);
                model.EraList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Era);
                model.SpecialityList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.VistaSpecialty).OrderBy(cd => cd.displayName).ToList();
                model.GenderList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Gender);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Populates and inits the model.
        /// </summary>
        /// <param name="model">The model.</param>
        private void PopulatelAndInitModel(TransferViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Facility> facilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(model.Visn.Id).Where(a => a.IsAlive == true).ToList();
                model.FacilityList = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, facilityList.ToList());
                model.ContractList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                model.DispositionList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Disposition);
                model.EraList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Era);
                model.SpecialityList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.VistaSpecialty).OrderBy(cd => cd.displayName).ToList();
                model.GenderList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.Gender);

                Patient patient = null;
                if (!string.IsNullOrEmpty(model.RootPatientId) && !model.RootPatientId.Equals(Constants.SSNROOT, StringComparison.InvariantCultureIgnoreCase)
                    && !string.IsNullOrEmpty(model.ExtensionPatientId) && !model.ExtensionPatientId.Equals(Guid.Empty.ToString()))
                    patient = FacadeManager.EntityInterface.GetPatientById(new II() { extension = model.ExtensionPatientId, root = model.RootPatientId }, null);
                Facility facility = FacadeManager.EntityInterface.GetFacility(new II() { extension = model.ExtensionFacilityId, root = model.RootFacilityId });
                model.PatientName = (patient != null) ? String.Format("{0}, {1} {2}", patient.LastName, patient.FirstName, patient.MiddleName).Trim() : string.Empty;
                model.PatientSSN = (patient != null) ? patient.SSN.extension : (model.RootPatientId.Equals(Constants.SSNROOT, StringComparison.InvariantCultureIgnoreCase)) ? model.ExtensionPatientId : string.Empty;
                model.PatientGender = (patient != null) ? (patient.Gender != null) ? patient.Gender.code : string.Empty : string.Empty;
                model.IsVistaValidated = (patient != null) ? patient.IsVistaValidated : false;
                model.FacilityName = facility.Name;
                model.RequestedDate = DateTime.Today;
                model.LocationAdmissionDate = DateTime.Today;
                model.DispositionDate = DateTime.Today;
                model.SCRating = patient != null ? patient.ServiceConnectedPercentage : "";
                if (facilityList != null && facilityList.Count > 0)
                {
                    model.SelectedAdmissionFacilityId = facilityList[0].Id.extension;
                }
                if (model.ContractList != null && model.ContractList.Count > 0)
                    model.SelectedContractCode = model.ContractList[0].code;
                if (model.DispositionList != null && model.DispositionList.Count > 0)
                    model.SelectedDispositionCode = model.DispositionList[0].code;
                if (model.EraList != null && model.EraList.Count > 0)
                    model.SelectedEraCode = Constants.OTHER;
                if (model.SpecialityList != null && model.SpecialityList.Count > 0)
                    model.SelectedSpecialityCode = model.SpecialityList[0].code;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Save the transfer.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="isNew">if set to <c>true</c> [is new].</param>
        private void SaveTransfer(TransferViewModel model, bool isNew)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (model.Visn == null)
                {
                    model.Visn = FacadeManager.EntityInterface.GetVisn(new II(model.VisnRoot, model.VisnExtension));
                    model.Region = FacadeManager.EntityInterface.GetRegions().ToList().Where(a => a.Id.extension.Equals(model.Visn.Region.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Number.ToString();
                }

                PopulateModel(model);

                II transferId = null;
                if (isNew)
                    transferId = new II(this.loggedUser.Domain, null);
                else
                    transferId = new II(model.TransferRequestIdRoot, model.TransferRequestIdExtension);

                Patient patient = null;
                if (isNew && model.RootPatientId.Equals(Constants.SSNROOT, StringComparison.InvariantCultureIgnoreCase))
                {
                    patient = new Patient();
                    Dictionary<string, string> names = Utilities.SplitPersonFullName(model.PatientName);
                    patient.FirstName = names[Constants.PERSON_FIRST_NAME];
                    patient.LastName = names[Constants.PERSON_LAST_NAME];
                    patient.MiddleName = names[Constants.PERSON_MIDDLE_NAME];
                    patient.SSN = new II(model.RootPatientId, model.ExtensionPatientId);
                    patient.Gender = model.GenderList.Where(a => a.code == model.PatientGender).FirstOrDefault();
                    patient.ServiceConnectedPercentage = model.SCRating ?? "";
                    FacadeManager.EntityInterface.CreatePatient(patient);
                }

                if (model.RootPatientId.Equals(Constants.SSNROOT, StringComparison.InvariantCultureIgnoreCase))
                    patient = FacadeManager.EntityInterface.GetPatientBySsn(new II(model.RootPatientId, model.ExtensionPatientId), null);
                else
                    patient = FacadeManager.EntityInterface.GetPatientById(new II(model.RootPatientId, model.ExtensionPatientId), null);

                if (!patient.IsVistaValidated && !isNew)
                {
                    Dictionary<string, string> names = Utilities.SplitPersonFullName(model.PatientName);
                    patient.FirstName = names[Constants.PERSON_FIRST_NAME];
                    patient.LastName = names[Constants.PERSON_LAST_NAME];
                    patient.MiddleName = names[Constants.PERSON_MIDDLE_NAME];
                    patient.Gender = model.GenderList.Where(a => a.code == model.PatientGender).FirstOrDefault();
                    patient.ServiceConnectedPercentage = model.SCRating;
                    FacadeManager.EntityInterface.UpdatePatient(patient);
                }

                model.PatientSSN = patient.SSN.extension;

                DateTime tempDate = DateTimeNowOnCurrentFacility;
                DateTime? admissionDate = null;
                DateTime? requestedDate = null;
                if (DateTime.TryParse(model.SelectedLocationAdmissionDate, out tempDate))
                    admissionDate = tempDate;
                if (DateTime.TryParse(model.SelectedRequestedDate, out tempDate))
                    requestedDate = tempDate;

                Facility facility = model.FacilityList.Where(a => a.Id.extension.Equals(model.ExtensionFacilityId, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                facility.VistaSite = FacadeManager.EntityInterface.GetVistaSite(facility.VistaSite.Id);
                facility.VistaSite.Visn = FacadeManager.EntityInterface.GetVisn(facility.VistaSite.Visn.Id);
                facility.VistaSite.Visn.Region = FacadeManager.EntityInterface.GetRegion(facility.VistaSite.Visn.Region.Id);

                Transfer transfer = new Transfer()
                {
                    Id = transferId,
                    CreationDate = DateTime.UtcNow,
                    CanceledDate = null,
                    Patient = patient,
                    FacilityFrom = facility,
                    Era = model.EraList.Where(a => a.code == model.SelectedEraCode).FirstOrDefault(),
                    Contract = model.ContractList.Where(a => a.code == model.SelectedContractCode).FirstOrDefault(),
                    Diagnosis = model.Diagnosis,
                    CurrentLocation = model.CurrentLocation,
                    AdmissionDate = admissionDate,
                    Comment = model.Comments,
                    Speciality = model.SpecialityList.Where(a => a.code == model.SelectedSpecialityCode).FirstOrDefault(),
                    Edited = DateTime.UtcNow,
                    EnteredBy = this.loggedUser.UserName,
                    LastEditBy = this.loggedUser.UserName,
                    RequestedDate = requestedDate,
                    IsNationalWaitList = model.IsNationalWaitList
                };

                if (!isNew)
                    FacadeManager.WorkflowInterface.UpdateTransferRequest(transfer, transfer.FacilityFrom);
                else
                    transferId = FacadeManager.WorkflowInterface.CreateTransferRequest(transfer, transfer.FacilityFrom);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
