﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using Microsoft.Web.Mvc;
using BMS.Facade.Fault;
using BMS.Web.App_GlobalResource;
using BMS.Utils.Properties;
using System.Text;
using BMS.Web.Controllers.Shared;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Event Notification Controller class.
    /// </summary>
    [ValidateInput(false)]
    public class EventNotificationController : BaseController
    {
        /// <summary>
        /// Default action for the EventNotificationController.
        /// </summary>
        /// <returns>Returns Index view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotificationViewModel eventViewModel = new EventNotificationViewModel();
                Fill(eventViewModel);
                return View(eventViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action of Event Notification Controller, used when clicking submit button from AddEdit view.
        /// </summary>
        /// <param name="eventViewModel">The event view model.</param>
        /// <returns>AddEdit view.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(EventNotificationViewModel eventViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (eventViewModel.WardGroupSelected != Strings.SelectWardGroup)
                    return this.RedirectToAction<EventNotificationController>(act => act.AddEdit(EncryptQueryString(new string[] { "wardGroup", "isAddOperation" }, new string[] { eventViewModel.WardGroupSelected, "true" }, loggedUser.Salt)));
                else
                {
                    ModelState.AddModelError("WardGroupSelected", Strings.YourDidNotSelectAWardGroup);
                    Fill(eventViewModel);
                    return View(eventViewModel);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        /// <summary>
        /// Redirect to AddEdit view when clicking Edit link in Index view.
        /// </summary>
        /// <param name="eventName">Name of the event.</param>
        /// <param name="isAddOperation">if set to <c>true</c> [is add operation].</param>
        /// <returns>AddEdit view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult AddEdit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {

                EventNotificationViewModel eventViewModel = new EventNotificationViewModel();
                eventViewModel.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                eventViewModel.WardGroupSelected = QueryStrings["wardGroup"];
                eventViewModel.WardGroupNameSelected = GetLocationName(eventViewModel.WardGroupSelected);
                eventViewModel.Header = Constants.EventNotificationsEdit;
                eventViewModel.LoggedUser = this.loggedUser;
                eventViewModel.EventAddEditDisplay = string.Empty;
                eventViewModel.EventAlreadyExistsDisplay = Strings.StyleDisplayNone;
                eventViewModel.IsAddOperation = bool.Parse(QueryStrings["isAddOperation"]);
                if (eventViewModel.IsAddOperation)
                {
                    //check if event already exists
                    bool eventExists = FacadeManager.ConfigurationInterface.EventNotificationExists(eventViewModel.WardGroupNameSelected, this.loggedUser.Domain, this.loggedUser.Facility.Id);
                    if (eventExists)
                    {
                        eventViewModel.EventAlreadyExistsDisplay = string.Empty;
                        eventViewModel.EventAddEditDisplay = Strings.StyleDisplayNone;
                    }
                    eventViewModel.AdmissionOrder = new EventNotificationItem();
                    eventViewModel.AnticipatedDischarge = new EventNotificationItem();
                    eventViewModel.DischargeAppointmentOrder = new EventNotificationItem();
                    eventViewModel.DischargeOrder = new EventNotificationItem();
                    eventViewModel.TransferOrder = new EventNotificationItem();
                    eventViewModel.BedOOS = new EventNotificationItem();
                    eventViewModel.BedSwitch = new EventNotificationItem();
                }
                else
                {
                    IList<EventNotification> editEventNotification = FacadeManager.ConfigurationInterface.GetEventNotification(eventViewModel.WardGroupNameSelected, this.loggedUser.Domain, this.loggedUser.Facility.Id);
                    foreach (EventNotification eventItem in editEventNotification)
                    {
                        switch (eventItem.Name)
                        {
                            case Constants.ADMISSION_ORDER:
                                eventViewModel.AdmissionOrder = GetEventItem(eventItem);
                                break;
                            case Constants.ANTICIPATED_DISCHARGE_ORDER:
                                eventViewModel.AnticipatedDischarge = GetEventItem(eventItem);
                                break;
                            case Constants.DISCHARGE_APPOINTMENT:
                                eventViewModel.DischargeAppointmentOrder = GetEventItem(eventItem);
                                break;
                            case Constants.DISCH_ORDER:
                                eventViewModel.DischargeOrder = GetEventItem(eventItem);
                                break;
                            case Constants.TRANSFER_ORDER:
                                eventViewModel.TransferOrder = GetEventItem(eventItem);
                                break;
                            case Constants.BED_OUT_OF_SERVICE:
                                eventViewModel.BedOOS = GetEventItem(eventItem);
                                break;
                            case Constants.BED_SWITCH:
                                eventViewModel.BedSwitch = GetEventItem(eventItem);
                                break;
                        }
                    }
                }
                return View(eventViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action used when clicking Add button from Index view.
        /// </summary>
        /// <param name="Event">The event.</param>
        /// <returns>AddEdit view result.</returns>
        [UpdatePermissionAuthFilterAttribute]
        [HttpPost]
        public ActionResult AddEdit(EventNotificationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                model.FlagsList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.StrictDecision);
                model.LoggedUser = loggedUser;
                if (model.AdmissionOrder.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.AdmissionOrder.BedControllerEmail))
                    ModelState.AddModelError("AdmissionOrder.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (model.AnticipatedDischarge.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.AnticipatedDischarge.BedControllerEmail))
                    ModelState.AddModelError("AnticipatedDischarge.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (model.DischargeAppointmentOrder.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.DischargeAppointmentOrder.BedControllerEmail))
                    ModelState.AddModelError("DischargeAppointmentOrder.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (model.DischargeOrder.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.DischargeOrder.BedControllerEmail))
                    ModelState.AddModelError("DischargeOrder.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (model.TransferOrder.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.TransferOrder.BedControllerEmail))
                    ModelState.AddModelError("TransferOrder.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (model.BedOOS.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.BedOOS.BedControllerEmail))
                    ModelState.AddModelError("BedOOS.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (model.BedSwitch.BedControllerNotify.code == Strings.Yes && string.IsNullOrWhiteSpace(model.BedSwitch.BedControllerEmail))
                    ModelState.AddModelError("BedSwitch.BedControllerEmail", Strings.YouMustEnterAnEmailAddress);

                if (!ModelState.IsValid)
                {
                    model.EventAddEditDisplay = string.Empty;
                    model.EventAlreadyExistsDisplay = Strings.StyleDisplayNone;
                    return View(model);
                }
                IList<EventNotification> eventNotificationList = new List<EventNotification>();
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.ADMISSION_ORDER, BedControllerEmailAddress = model.AdmissionOrder.BedControllerEmail, BedControllerNotification = GetFlag(model.AdmissionOrder.BedControllerNotify.code, model.FlagsList) });
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.ANTICIPATED_DISCHARGE_ORDER, BedControllerEmailAddress = model.AnticipatedDischarge.BedControllerEmail, BedControllerNotification = GetFlag(model.AnticipatedDischarge.BedControllerNotify.code, model.FlagsList) });
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.DISCHARGE_APPOINTMENT, BedControllerEmailAddress = model.DischargeAppointmentOrder.BedControllerEmail, BedControllerNotification = GetFlag(model.DischargeAppointmentOrder.BedControllerNotify.code, model.FlagsList) });
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.DISCH_ORDER, BedControllerEmailAddress = model.DischargeOrder.BedControllerEmail, BedControllerNotification = GetFlag(model.DischargeOrder.BedControllerNotify.code, model.FlagsList) });
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.TRANSFER_ORDER, BedControllerEmailAddress = model.TransferOrder.BedControllerEmail, BedControllerNotification = GetFlag(model.TransferOrder.BedControllerNotify.code, model.FlagsList) });
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.BED_OUT_OF_SERVICE, BedControllerEmailAddress = model.BedOOS.BedControllerEmail, BedControllerNotification = GetFlag(model.BedOOS.BedControllerNotify.code, model.FlagsList) });
                eventNotificationList.Add(new EventNotification() { WardGroup = model.WardGroupNameSelected, Name = Constants.BED_SWITCH, BedControllerEmailAddress = model.BedSwitch.BedControllerEmail, BedControllerNotification = GetFlag(model.BedSwitch.BedControllerNotify.code, model.FlagsList) });
                //save events
                FacadeManager.ConfigurationInterface.SaveEventNotification(eventNotificationList, this.loggedUser.Domain, model.IsAddOperation, this.loggedUser.Facility.Id, model.WardGroupNameSelected);
                if (model.IsAddOperation)
                    return this.RedirectToAction<EventNotificationController>(act => act.AddConfirmation());
                else
                    return this.RedirectToAction<EventNotificationController>(act => act.EditConfirmation());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Display AddConfirmation view after an add operation.
        /// </summary>
        /// <param name="Event">The event name.</param>
        /// <returns>AddConfirmation view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult AddConfirmation()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotificationViewModel eventViewModel = new EventNotificationViewModel();
                eventViewModel.LoggedUser = this.loggedUser;
                return View(eventViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Display EditConfirmation view after an edit operation.
        /// </summary>
        /// <returns>EditConfirmation view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult EditConfirmation()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotificationViewModel eventViewModel = new EventNotificationViewModel();
                eventViewModel.LoggedUser = this.loggedUser;
                return View(eventViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Show Delete view when click Delete button.
        /// </summary>
        /// <param name="wardGroupName">The wardGroupName.</param>
        /// <returns>Returns the Delete view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotificationViewModel model = new EventNotificationViewModel();
                model.WardGroupSelected = QueryStrings["wardGroup"];
                model.WardGroupNameSelected = GetLocationName(model.WardGroupSelected);
                model.LoggedUser = loggedUser;
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the specified event view model.
        /// </summary>
        /// <param name="eventViewModel">The event view model.</param>
        /// <returns></returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Delete(EventNotificationViewModel model)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FacadeManager.ConfigurationInterface.DeleteEventNotification(this.loggedUser.Domain, this.loggedUser.Facility.Id, model.WardGroupNameSelected);
                return this.RedirectToAction<EventNotificationController>(act => act.DeleteConfirmation());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Display DeleteConfirmation view after a delete operation.
        /// </summary>
        /// <returns>DeleteConfirmation view result.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult DeleteConfirmation()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotificationViewModel eventViewModel = new EventNotificationViewModel();                
                eventViewModel.LoggedUser = this.loggedUser;
                return View(eventViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Members

        /// <summary>
        /// Gets the flag.
        /// </summary>
        /// <param name="isFlagged">if set to <c>true</c> [is flagged].</param>
        /// <param name="flagList">The flag list.</param>
        /// <returns>CD object.</returns>
        private CD GetFlag(bool isFlagged, CDList flagList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (isFlagged)
                    return flagList.Find(cd => cd.code == Constants.Yes);
                else
                    return flagList.Find(cd => cd.code == Constants.No);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the flag.
        /// </summary>
        /// <param name="isFlagged">The is flagged.</param>
        /// <param name="flagList">The flag list.</param>
        /// <returns>CD object.</returns>
        private CD GetFlag(string isFlagged, CDList flagList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (isFlagged == Constants.Yes)
                    return flagList.Find(cd => cd.code == Constants.Yes);
                else
                    return flagList.Find(cd => cd.code == Constants.No);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the specified event view model.
        /// </summary>
        /// <param name="eventViewModel">The event view model.</param>
        private void Fill(EventNotificationViewModel eventViewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                eventViewModel.LoggedUser = this.loggedUser;
                eventViewModel.WardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id).ToList();
                //Add Select a Ward Group option to the drop down list.
                eventViewModel.WardGroupList.Insert(0, new Division() { Id = new II() { root = this.loggedUser.Domain, extension = Strings.SelectWardGroup }, Name = Strings.SelectWardGroup });
                List<EventNotification> eventsList = FacadeManager.ConfigurationInterface.GetEventNotifications(this.loggedUser.Domain, this.loggedUser.Facility.Id);
                eventViewModel.CurrentEventsList = new List<EventNotification>();
                foreach (EventNotification evn in eventsList)
                {
                    if (eventViewModel.WardGroupList.Where(a => a.Name.Equals(evn.WardGroup, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                        eventViewModel.CurrentEventsList.Add(evn);
                }                
                
                var elementList = (from item in eventViewModel.CurrentEventsList
                                   group item by item.WardGroup into grp
                                   select grp);

                eventViewModel.CurrentsEventsListExtented = new List<DisplayCurrentEvent>();
                DisplayCurrentEvent displayCurrentEvent = null;
                foreach (IGrouping<string, EventNotification> elementGroup in elementList)
                {
                    displayCurrentEvent = BuildDisplayCurrentEvent(elementGroup);
                    displayCurrentEvent.WardGroupExtension = eventViewModel.WardGroupList.Where(a => a.Name.Equals(displayCurrentEvent.WardGroup, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault().Id.extension;
                    eventViewModel.CurrentsEventsListExtented.Add(displayCurrentEvent);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Builds the display current event.
        /// </summary>
        /// <param name="elementGroup">The element group.</param>
        /// <returns></returns>
        private DisplayCurrentEvent BuildDisplayCurrentEvent(IGrouping<string, EventNotification> elementGroup)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                DisplayCurrentEvent element = new DisplayCurrentEvent();
                element.WardGroup = elementGroup.Key;
                element.AdmissionOrder = BuildEventText(elementGroup.Where(a => a.Name == Constants.ADMISSION_ORDER).First());
                element.AnticipatedDischarge = BuildEventText(elementGroup.Where(a => a.Name == Constants.ANTICIPATED_DISCHARGE_ORDER).First());
                element.DischargeAppointmentOrder = BuildEventText(elementGroup.Where(a => a.Name == Constants.DISCHARGE_APPOINTMENT).First());
                element.DischargeOrder = BuildEventText(elementGroup.Where(a => a.Name == Constants.DISCH_ORDER).First());
                element.TransferOrder = BuildEventText(elementGroup.Where(a => a.Name == Constants.TRANSFER_ORDER).First());
                element.BedOOS = BuildEventText(elementGroup.Where(a => a.Name == Constants.BED_OUT_OF_SERVICE).First());
                element.BedSwitch = BuildEventText(elementGroup.Where(a => a.Name == Constants.BED_SWITCH).First());
                return element;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }


        }

        /// <summary>
        /// Builds the event text.
        /// </summary>
        /// <param name="eventNotification">The event notification.</param>
        /// <returns></returns>
        private string BuildEventText(EventNotification eventNotification)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return eventNotification.BedControllerNotification.displayName;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the event item.
        /// </summary>
        /// <param name="eventNotification">The event notification.</param>
        /// <returns></returns>
        private EventNotificationItem GetEventItem(EventNotification eventNotification)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EventNotificationItem result = new EventNotificationItem();
                result.BedControllerEmail = eventNotification.BedControllerEmailAddress;
                result.BedControllerNotify = eventNotification.BedControllerNotification;
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Gets the name of the location.
        /// </summary>
        /// <param name="location">The location.</param>
        /// <returns></returns>
        private string GetLocationName(string location)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<Division> wardGroupList = FacadeManager.EntityInterface.GetDivisions(this.loggedUser.Facility.Id);
                foreach (Division obj in wardGroupList)
                {
                    if (obj.Id.extension.Equals(location, StringComparison.InvariantCultureIgnoreCase))
                        return obj.Name;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion
    }
}
