﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Reflection;

namespace BMS.Utils
{
    /// <summary>
    /// WCF Proxy Manager used to keep a list of all proxies and recreate them on faulted or closed state.
    /// </summary>
    public class ProxyManager
    {
        private static Hashtable _proxies = new Hashtable();
        static BmsLogger logger = new BmsLogger("BMS.Utils.ProxyManager - ");

        /// <summary>
        /// Return a cached proxy.
        /// </summary>
        /// <typeparam name="T">ICommunicationObject proxy type</typeparam>
        /// <returns>The cached proxy.</returns>
        public static T GetProxy<T>(string endpointName) where T : class, ICommunicationObject, IDisposable, new()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string proxyKey = null;
                if (endpointName.Contains(".Windows") || endpointName.StartsWith("EVS."))
                    proxyKey = endpointName;
                else
                {
                    string userName = GetCurrentUserName();
                    if (!string.IsNullOrEmpty(userName))
                        proxyKey = endpointName + "_" + userName;
                    else
                        proxyKey = endpointName;
                }

                lock (_proxies)
                {                    
                    if (!_proxies.Contains(proxyKey))
                        _proxies.Add(proxyKey, new PoolProxy<T>());

                    PoolProxy<T> pool = ((PoolProxy<T>)_proxies[proxyKey]);
                    T proxy = pool.Take();

                    if (proxy != null && proxy.State != CommunicationState.Opened)
                    {
                        proxy.Abort();
                        proxy = null;
                    }
                    // Create the proxy if not found in the pool.
                    if (proxy == null)                    
                        proxy = CreateProxy<T>(endpointName);                    
                    
                    return proxy;
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static T CreateProxy<T>(string endpointName) where T : class, ICommunicationObject, new()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (String.IsNullOrEmpty(endpointName))
                    return new T();
                return (T)Activator.CreateInstance(typeof(T), endpointName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void ReturnProxy<T>(T proxy, string endpointName) where T : class, ICommunicationObject, IDisposable
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool shouldClose = false;

                string proxyKey = null;
                if (endpointName.Contains(".Windows") || endpointName.StartsWith("EVS."))
                    proxyKey = endpointName;
                else
                {
                    string userName = GetCurrentUserName();
                    if (!string.IsNullOrEmpty(userName))
                        proxyKey = endpointName + "_" + userName;
                    else
                        proxyKey = endpointName;
                }

                lock (_proxies)
                {
                    if (_proxies.Contains(proxyKey))
                    {                     
                        PoolProxy<T> pool = ((PoolProxy<T>)_proxies[proxyKey]);
                        shouldClose = !pool.Return(proxy);
                    }
                }                
                // Close the proxy if it's not put back into pool
                if (shouldClose)
                {
                    Tracer.TraceMessage(string.Format("Pool for proxy endpoint {0} is full!", proxyKey));
                    try
                    {
                        proxy.Close();
                    }
                    catch
                    {
                        proxy.Abort();                        
                    }
                    proxy.Dispose();
                    proxy = null;
                }                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void ClearUserProxies(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                string partKey = "_" + userName;
                List<string> keysToRemove = new List<string>();
                lock (_proxies)
                {
                    foreach (string key in _proxies.Keys)
                    {
                        if (key.EndsWith(partKey, StringComparison.InvariantCultureIgnoreCase))
                        {
                            try
                            {
                                keysToRemove.Add(key);
                                MethodInfo method = _proxies[key].GetType().GetMethod("Clear");
                                method.Invoke(_proxies[key], null);
                            }
                            catch (Exception ex)
                            {
                                Tracer.TraceMessage("ProxyManager.ClearUserProxies for user " + userName + " and key " + key + " exception");
                                Tracer.TraceException(ex);
                            }
                        }
                    }
                    foreach (string key in keysToRemove)
                    {
                        _proxies[key] = null;
                        _proxies.Remove(key);
                    }
                }
            }
            catch (Exception ex)
            {
                Tracer.TraceMessage("ProxyManager.ClearUserProxies for user " + userName + " exception");
                Tracer.TraceException(ex);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        /// <summary>
        /// Checks the current state of an existing proxy.
        /// </summary>
        /// <param name="proxy">The communication object (proxy) to be checked.</param>
        /// <returns>False is proxy is unsable, true otherwise.</returns>
        public static bool CheckProxyInstance(ICommunicationObject proxy)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                try
                {
                    if (proxy == null)
                        return false;
                    if (proxy.State == System.ServiceModel.CommunicationState.Faulted)
                        return false;
                    if (proxy.State == System.ServiceModel.CommunicationState.Closed)
                        return false;
                    if (proxy.State == System.ServiceModel.CommunicationState.Closing)
                        return false;
                    return true;
                }
                catch { return false; }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        
        /// <summary>
        /// Verifies whether the exception is non-critical - the proxy can be recreated.
        /// </summary>
        /// <param name="e">Thrown exception</param>
        /// <returns>True is exception is not vital and the proxy can be recreated.</returns>
        public static bool CheckThrownException(System.Exception e)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (e is System.ServiceModel.Security.MessageSecurityException)
                    return true;
                if (e is System.ServiceModel.FaultException)
                    return false;
                if (e is System.ServiceModel.CommunicationException)
                    return true;
                if (e is System.TimeoutException)
                    return true;
                if (e is System.ServiceModel.CommunicationObjectAbortedException)
                    return true;
                return false;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
        
        /// <summary>
        /// Returns the current HttpSession id. If this method is called outside a HttpContext
        /// it tries to find the session id in the WCF OperationContext in HttpRequestMessage headers.
        /// </summary>
        /// <returns>The current HTTP Session ID.</returns>
        public static string GetCurrentSessionID()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                String result = String.Empty;
                if (System.Web.HttpContext.Current != null && System.Web.HttpContext.Current.Session != null && System.Web.HttpContext.Current.Session["User"] != null)
                    return System.Web.HttpContext.Current.Session.SessionID;
                if (OperationContext.Current != null &&
                    OperationContext.Current.IncomingMessageProperties != null && OperationContext.Current.InstanceContext.State == CommunicationState.Opened)
                {
                    try
                    {
                        KeyValuePair<string, object> property = OperationContext.Current.IncomingMessageProperties.FirstOrDefault(a => a.Key == HttpRequestMessageProperty.Name);
                        if (property.Value != null && property.Value is HttpRequestMessageProperty)
                        {
                            HttpRequestMessageProperty val = property.Value as HttpRequestMessageProperty;
                            if (val.Headers.HasKeys() && val.Headers.AllKeys.Contains(Constants.HTTP_SESSION_ID_HTTP_HEADER))
                                return val.Headers[Constants.HTTP_SESSION_ID_HTTP_HEADER];
                        }
                    }
                    catch (System.ObjectDisposedException)
                    {
                        return string.Empty;
                    }
                }
                return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Returns the current HttpSession id. If this method is called outside a HttpContext
        /// it tries to find the session id in the WCF OperationContext in HttpRequestMessage headers.
        /// </summary>
        /// <returns>The current HTTP Session ID.</returns>
        public static string GetCurrentSessionID2()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                String result = String.Empty;
                if (System.Web.HttpContext.Current != null && System.Web.HttpContext.Current.Session != null)
                    return System.Web.HttpContext.Current.Session.SessionID;
                if (OperationContext.Current != null &&
                    OperationContext.Current.IncomingMessageProperties != null && OperationContext.Current.InstanceContext.State == CommunicationState.Opened)
                {
                    try
                    {
                        KeyValuePair<string, object> property = OperationContext.Current.IncomingMessageProperties.FirstOrDefault(a => a.Key == HttpRequestMessageProperty.Name);
                        if (property.Value != null && property.Value is HttpRequestMessageProperty)
                        {
                            HttpRequestMessageProperty val = property.Value as HttpRequestMessageProperty;
                            if (val.Headers.HasKeys() && val.Headers.AllKeys.Contains(Constants.HTTP_SESSION_ID_HTTP_HEADER))
                                return val.Headers[Constants.HTTP_SESSION_ID_HTTP_HEADER];
                        }
                    }
                    catch (System.ObjectDisposedException)
                    {
                        return string.Empty;
                    }
                }
                return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static string GetCurrentUserName()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                String result = String.Empty;
                if (System.Web.HttpContext.Current != null && System.Web.HttpContext.Current.Session != null && System.Web.HttpContext.Current.Session["UserName"] != null)                
                    return System.Web.HttpContext.Current.Session["UserName"].ToString();
                                
                return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
