﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using PAP = InfoWorld.Security.Authorization.PolicyAdministrationPoint;

namespace BMS.ServicesWrapper.Security
{

    public interface ISecurityWrapper
    {
        /// <summary>
        /// Retrieves the current domain for the logged on user. 
        /// </summary>
        /// <returns>the domain name</returns>
        string GetCurrentDomain();

        /// <summary>
        /// Gets the user.
        /// </summary>
        /// <param name="userId">The user id.</param>
        /// <returns></returns>
        User GetUser(string userId);

        /// <summary>
        /// Check permission forthe username.
        /// </summary>
        /// <param name="userName"></param>
        /// <param name="permissionName"></param>
        /// <returns></returns>
        bool CheckPermission(string userName, string permissionName);        

        /// <summary>
        /// Sets the available domains for the username.
        /// </summary>
        /// <param name="userName"></param>
        /// <param name="domains"></param>
        void SetAvailableDomains(string username, string userId, IList<string> domains);

        /// <summary>
        /// Gets the available domains for the username.
        /// </summary>
        /// <typeparam name="?"></typeparam>
        /// <param name="?"></param>
        IList<string> GetAvailableDomains(string username, string userId);

        /// <summary>
        /// Gets the available domains.
        /// </summary>
        /// <returns></returns>
        IList<string> GetAvailableDomains();

        /// <summary>
        /// Gets the user operations for facility.
        /// </summary>
        /// <param name="facilityId"></param>
        /// <returns></returns>
        IList<UserOperations> GetUserOperations(II facilityId);

        /// <summary>
        /// Adds the user operations.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        void AddUserOperations(string userName, List<PermissionFacility> facilities);

        /// <summary>
        /// Removes the user operations.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        void RemoveUserOperations(string userName, List<PermissionFacility> facilities);

        /// <summary>
        /// Saves the user roles.
        /// </summary>
        /// <param name="userRoles">The user roles.</param>
        void SaveUserRoles(UserRoles userRoles);

        /// <summary>
        /// Gets the user roles.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="userId">The user id.</param>
        /// <returns></returns>
        UserRoles GetUserRoles(String userName, string userId);

        /// <summary>
        /// Gets the default user profile.
        /// </summary>
        /// <returns></returns>
        User GetDefaultUserProfile();

        /// <summary>
        /// Gets the active directory domain.
        /// </summary>
        /// <returns></returns>
        List<string> GetActiveDirectoryDomains();

        /// <summary>
        /// Gets the user is super user.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <returns></returns>
        bool GetUserIsSuperUser(string userName);

        /// <summary>
        /// Checks the BMS read permission.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        bool CheckBMSReadPermission(string userName, II facilityId);

        /// <summary>
        /// Checks the BMS write permission.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        bool CheckBMSWritePermission(string userName, II facilityId);

        /// <summary>
        /// Gets the facilities for which the user has read and write permissions.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        /// <returns></returns>
        List<PermissionFacility> GetFacilitiesForReadAndWrite(string userName, List<Facility> facilities);

        /// <summary>
        /// Gets the facilities for which the user has read permission, including the user groups.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <param name="facilities">The facilities.</param>
        /// <returns></returns>
        List<Facility> GetFacilitiesWithReadAccess(string userName, List<Facility> facilities);

        /// <summary>
        /// Gets the users by role.
        /// </summary>
        /// <param name="roleName">Name of the role.</param>
        /// <returns></returns>
        List<BMS.Facade.Data.User> GetUsersByRole(string roleName);

        /// <summary>
        /// Gets the users by role and facility.
        /// </summary>
        /// <param name="roleName">Name of the role.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        List<BMS.Facade.Data.User> GetUsersByRoleAndFacility(string roleName, II facilityId);

        /// <summary>
        /// Gets all user roles.
        /// </summary>
        /// <returns></returns>
        List<string> GetAllUserRoles();

        /// <summary>
        /// Gets the administrative functions client.
        /// </summary>
        PAP.AdministrativeFunctionsClient AdministrativeFunctionsClient { get; }

        /// <summary>
        /// Fills the cache.
        /// </summary>
        void FillCache();

        /// <summary>
        /// Clears the user permission cache.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        void ClearUserPermissionCache(string userName);

        /// <summary>
        /// Caches the policies for organizations.
        /// </summary>
        void CachePoliciesForOrganizations();

        /// <summary>
        /// Caches the user roles.
        /// </summary>
        /// <param name="user">The user.</param>
        void CacheUserRoles(InfoWorld.Security.Authorization.PolicyAdministrationPoint.User user);

        /// <summary>
        /// Searches for AD users.
        /// </summary>
        IList<User> SearchUsers(string domain, string searchText);

        /// <summary>
        /// Gets the full name of the user (username@domain).
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <returns></returns>
        string GetFullUserName(string userName);

        /// <summary>
        /// Gets the name of the user by user.
        /// </summary>
        /// <param name="userName">Name of the user.</param>
        /// <returns></returns>
        User GetUserByUserName(string userName);

        /// <summary>
        /// Checks the user role exists.
        /// </summary>
        /// <param name="username">The username.</param>
        /// <param name="userid">The userid.</param>
        /// <returns></returns>
        string CheckUserRoleExists(string username, string userid);
    }
}
