﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using BMS.ServicesWrapper.EVS;
using BMS.Facade.Data;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using FC = BMS.Facade.Data;
using InfoWorld.EVS.CTSMAPI;

using SC = BMS.DataContracts;
using BMS.VistaWorker.Exceptions;
using BMS.ServicesWrapper.WF;
using System.ServiceModel;
using BMS.Facade.Fault;

namespace BMS.VistaWorker.Writer.Concrete.WF
{
    /// <summary>
    /// The order writer.
    /// </summary>
    class OrderWriter : BaseWorkFlowWriter<Order>
    {
        /// <summary>
        /// Sends the event.
        /// </summary>
        /// <param name="entity">The entity.</param>
        protected override void SendEvent(Order entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (entity.Patient == null)
                    throw new AbortEntityProcessingException("Abort: Order with id=" + entity.IEN + " has no patient.");
                FC.HospitalLocation hospitallocation = EIS.GetHospitalLocation(entity.PatientLocationId, VistaSite.Id);

                IList<CDWithProperties> cdpList = GetOrderableItem(entity.OrdersOrderableItems);
                string orderType = null;
                if (cdpList != null && cdpList.Count > 0)
                    orderType = GetOrderType(cdpList[0]);
                if (string.IsNullOrEmpty(orderType))
                {
                    string msj = string.Format("Order with id={0} has no type.", entity.IEN);
                    throw new AbortEntityProcessingException(msj);
                }

                string domainId = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetCurrentDomain();
                Dictionary<string, string> signedNames = null;
                if (entity.OrderAction.SignedBy != null)
                    signedNames = Utilities.SplitPersonFullName(entity.OrderAction.SignedBy.Name);
                Dictionary<string, string> providerNames = null;
                if (entity.OrderAction.Provider != null)
                    providerNames = Utilities.SplitPersonFullName(entity.OrderAction.Provider.Name);

                if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_ADMISSION))
                {
                    SC.AdmissionOrder wfAdmission = new SC.AdmissionOrder()
                    {
                        Id = new II(domainId, null),                        
                        PatientId = MakeWfPatient(entity.Patient),
                        OrderedDate = entity.OrderAction.DateTimeOrdered,
                        SignedDate = entity.OrderAction.DateTimeSigned,
                        Ien = entity.IEN,
                        OrderText = entity.OrderAction.OrderText,
                        ReleasedDate = entity.OrderAction.ReleaseDateTime,
                        SignedById = null,
                        ProviderId = null,
                        HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                        FacilityList = EIS.GetFacilities(VistaSite.Id).Select(facility => facility.Id).ToList(),
                        IsActive = (entity.OrderStatusId.HasValue && entity.OrderStatusId == 6) ? true : false
                    };

                    ConvertToUtc(wfAdmission);
                    try
                    {
                        BMS.ServicesWrapper.BMService.BMSFactory.BedManagerOperationsClientWindows.CreateAdmissionOrder(wfAdmission);
                        string msj = string.Format("Create admission order for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);
                    }
                    catch (FaultException<GenericWFServiceFault> e)
                    {
                        string msj = string.Format("Admission order for patient {0} failed:\n{1}",
                            entity.Patient.Name, e.Detail);
                        throw new AbortEntityProcessingException(msj);
                    }
                    return;
                }

                if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_DISCHARGE))
                {
                    SC.DischargeOrder wfDischarge = new SC.DischargeOrder()
                    {
                        Id = new II(domainId, null),
                        PatientId = MakeWfPatient(entity.Patient),
                        OrderedDate = entity.OrderAction.DateTimeOrdered,
                        SignedDate = entity.OrderAction.DateTimeSigned,
                        Ien = entity.IEN,
                        OrderText = entity.OrderAction.OrderText,
                        ReleasedDate = entity.OrderAction.ReleaseDateTime,
                        SignedById = null,
                        ProviderId = null,
                        HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                        FacilityList = EIS.GetFacilities(VistaSite.Id).Select(facility => facility.Id).ToList(),
                        IsActive = (entity.OrderStatusId.HasValue && entity.OrderStatusId == 6) ? true : false,
                        OrderableItem = GetVistaOrderableItem(entity.OrdersOrderableItems),                        
                    };

                    ConvertToUtc(wfDischarge);
                    try
                    {
                        BMS.ServicesWrapper.BMService.BMSFactory.BedManagerOperationsClientWindows.CreateDischargeOrder(wfDischarge);
                        string msj = string.Format("Create discharge request for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);
                    }
                    catch (FaultException<GenericWFServiceFault> e)
                    {
                        string msj = string.Format("Discharge request  for patient {0} failed:\n{1}",
                            entity.Patient.Name, e.Detail);
                        throw new AbortEntityProcessingException(msj);
                    }
                    return;
                }

                if (orderType.ToUpper().Equals(Constants.ORDERABLE_ITEM_TYPE_TRANSFER))
                {
                    SC.MovementOrder wfMovement = new SC.MovementOrder()
                    {
                        Id = new II(domainId, null),                        
                        PatientId = MakeWfPatient(entity.Patient),
                        OrderedDate = entity.OrderAction.DateTimeOrdered,
                        SignedDate = entity.OrderAction.DateTimeSigned,
                        Ien = entity.IEN,
                        OrderText = entity.OrderAction.OrderText,
                        ReleasedDate = entity.OrderAction.ReleaseDateTime,
                        SignedById = null,
                        ProviderId = null,
                        HospitalLocationId = (hospitallocation != null) ? hospitallocation.Id : null,
                        VistaSiteId = (VistaSite != null) ? VistaSite.Id : null,
                        FacilityList = EIS.GetFacilities(VistaSite.Id).Select(facility => facility.Id).ToList(),
                        IsActive = (entity.OrderStatusId.HasValue && entity.OrderStatusId == 6) ? true : false
                    };
                    ConvertToUtc(wfMovement);
                    try
                    {
                        BMS.ServicesWrapper.BMService.BMSFactory.BedManagerOperationsClientWindows.CreateMovementOrder(wfMovement);
                        string msj = string.Format("Create movement request for patient {0} with succes", entity.Patient.Name);
                        Logger.LogInformation(msj);
                    }
                    catch (FaultException<GenericWFServiceFault> e)
                    {
                        string msj = string.Format("Movement requestt  for patient {0} failed:\n{1}",
                            entity.Patient.Name, e.Detail);
                        throw new AbortEntityProcessingException(msj);
                    }
                    return;
                }

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CD MakeConcept(OrderableItem o)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return new CD()
                {
                    code = o.IEN.ToString(),
                    displayName = o.Name,
                    codeSystem = Constants.OrderableItemCodeSystem,
                    codeSystemName = Util.Vocabulary.OrderableItem.ToString()
                };
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private CD GetVistaOrderableItem(IList<OrdersOrderableItem> ordersOrderableItem)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ordersOrderableItem != null && ordersOrderableItem.Count > 0)
                {
                    //select OrderableItem of the order
                    OrdersOrderableItem ordItem = ordersOrderableItem[0];
                    CodeFilterParameters codeFilterParam = new CodeFilterParameters();
                    codeFilterParam.MaxSelectedCodes = int.MaxValue;
                    codeFilterParam.VocabularyDomain = Util.Vocabulary.OrderableItem.ToString();
                    codeFilterParam.MatchAlgorithm = MatchAlgorithm.StartsWithIgnoreCase;
                    codeFilterParam.MatchProperty = MatchProperty.Code;
                    codeFilterParam.MatchText = VistaSite.Code;
                    return EVSFactory.Instance.SearchConceptsWithPaging(codeFilterParam).Where(a => a.code == VistaSite.Code + "_" + ordItem.OrderableItem.IEN.ToString()).FirstOrDefault();
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private IList<CDWithProperties> GetOrderableItem(IList<OrdersOrderableItem> ordersOrderableItem)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ordersOrderableItem != null && ordersOrderableItem.Count > 0)
                {
                    //select OrderableItem of the order
                    OrdersOrderableItem ordItem = ordersOrderableItem[0];
                    //construct parameters to get ADT Orderable Items from EVS
                    CodeFilterParametersWithProperty codeFilterParamWithProperties = new CodeFilterParametersWithProperty();
                    codeFilterParamWithProperties.VocabularyDomain = Util.Vocabulary.ADTOrderableItem.ToString();
                    codeFilterParamWithProperties.Properties = new List<string>();
                    codeFilterParamWithProperties.Properties.Add("IEN");
                    codeFilterParamWithProperties.Properties.Add("OrderableItemType");
                    //get ADT Orderable Items from EVS
                    IList<CDWithProperties> cdList = EVS.GetCodesWithProperties(codeFilterParamWithProperties);
                    //search order's OrderableItem in the ADT Orderable Items list
                    int vistaSiteCodeLength = VistaSite.Code.Length;
                    int orderableItemIenLength = ordItem.OrderableItem.IEN.Length;
                    IList<CDWithProperties> cdpList = cdList.Where(a => (a.code.Substring(0, vistaSiteCodeLength) == VistaSite.Code && a.code.Substring(a.code.Length - orderableItemIenLength, orderableItemIenLength) == ordItem.OrderableItem.IEN.ToString())).ToList();
                    return cdpList;
                }
                return null;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private List<II> CreateFacilitiesList(IList<CDWithProperties> cdpList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (cdpList == null)
                    return null;
                List<II> facilitiesList = new List<II>();
                foreach (CDWithProperties cdp in cdpList)
                {
                    string[] cdpCodeArray = cdp.code.Split('_');
                    Facility facility = EIS.GetFacilityByCode(cdpCodeArray[1]);
                    if (facility != null)
                        facilitiesList.Add(facility.Id);                    
                }
                return facilitiesList;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private string GetOrderType(CDWithProperties cdp)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (cdp != null)
                {
                    // get the type property of the OrderableItem
                    ConceptProperty cp = cdp.Properties.Where(a => a.PropertyName.text == "OrderableItemType").FirstOrDefault();
                    if (cp != null)
                    {
                        //get the type property value(type code) of the OrderableItem
                        string code = cp.PropertyValue.text;
                        //get the OrderableItemType list from EVS
                        IList<CD> list = EVS.GetCodes(new CodeFilterParametersWithProperty() { VocabularyDomain = Util.Vocabulary.OrderableItemType.ToString() });
                        //search type by type code
                        CD cd = list.Where(a => a.code == code).FirstOrDefault();
                        //return type display name(Admission/Discharge/Transfer)
                        return cd.displayName;
                    }
                }
                return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
