﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.EVS;
using BMS.ServicesWrapper.EIS;
using BMS.Utils;
using EisPatient = BMS.Facade.Data.Patient;
using SC = BMS.DataContracts;
using BMS.VistaWorker.Writer.Abstract;
using BMS.VistaWorker.Writer.Concrete.CatchExceptions;
using System.ServiceModel;
using BMS.Facade.Fault;
using BMS.Facade;

namespace BMS.VistaWorker.Writer.Concrete
{
    class PatientWriter : BaseEisWriter<Patient, EisPatient>
    {
        protected override EisPatient GetBmsEntity(Patient entity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return EIS.GetPatientBySsn(new II() { root = Constants.SSNROOT, extension = entity.SocialSecurityNumber }, null);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public static void TranslatePatientFromVistAToBMS(Patient entity, EisPatient bmsEntity, BMS.Facade.Data.VistaSite site)
        {
            IList<CD> genderList = EVS.GetCodes(new CodeFilterParameters() { VocabularyDomain = BMS.Facade.Data.Util.Vocabulary.Gender.ToString() });
            Dictionary<string, string> names = Utilities.SplitPersonFullName(entity.Name);
            bmsEntity.FirstName = names[Constants.PERSON_FIRST_NAME];
            CD gender = genderList.Where(a => a.code.ToLower() == entity.Sex.ToString().ToLower()).FirstOrDefault();
            if (gender == null)
                gender = genderList.Where(a => a.code == Constants.GENDER_UNKNOWN_CODE).First();
            bmsEntity.Gender = gender;
            bmsEntity.Ien = entity.IEN;
            bmsEntity.LastName = names[Constants.PERSON_LAST_NAME];
            bmsEntity.MiddleName = names[Constants.PERSON_MIDDLE_NAME];
            bmsEntity.SSN = new II(Constants.SSNROOT, entity.SocialSecurityNumber);
            bmsEntity.IsVistaValidated = true;
            if (entity.TreatingSpecialtyId.HasValue)
            {
                IList<CD> treatingSpecialtyList = EVS.GetCodes(new CodeFilterParameters() { VocabularyDomain = BMS.Facade.Data.Util.Vocabulary.VistaTreatingSpecialty.ToString() });
                if (treatingSpecialtyList != null)
                    bmsEntity.TreatingSpecialty = treatingSpecialtyList.Where(a => a.code == site.Code + "_" + entity.TreatingSpecialtyId.Value.ToString()).FirstOrDefault();
                else
                    bmsEntity.TreatingSpecialty = null;
            }
            else
                bmsEntity.TreatingSpecialty = null;
            bmsEntity.AdmittingDiagnosis = entity.AdmittingDiagnosis;
            bmsEntity.ServiceConnectedPercentage = entity.ServiceConnectedPercentage.HasValue ?
                entity.ServiceConnectedPercentage.Value.ToString() :
                "";
            bmsEntity.DateOfBirth = entity.DateOfBirth;
            if (entity.AttendingPhysicianId.HasValue)
            {
                Dictionary<string, string> attendingPhysicianNames = null;
                attendingPhysicianNames = Utilities.SplitPersonFullName(entity.AttendingPhysician.Name);
                BMS.Facade.Data.Person attendingPhysician = EIS.GetMedicalPerson(entity.AttendingPhysicianId.ToString(), site.Id);
                if (attendingPhysician == null)
                {
                    //save attending physician
                    attendingPhysician = new BMS.Facade.Data.Person()
                                    {
                                        Ien = entity.AttendingPhysician.IEN,
                                        FirstName = attendingPhysicianNames[Constants.PERSON_FIRST_NAME],
                                        MiddleName = attendingPhysicianNames[Constants.PERSON_MIDDLE_NAME],
                                        LastName = attendingPhysicianNames[Constants.PERSON_LAST_NAME],
                                        VistaSite = new II() { root = EISConstants.SITE, extension = site.Id.extension }
                                    };
                    EIS.CreateMedicalPerson(attendingPhysician);
                }
                else
                {
                    if (attendingPhysician.FirstName != attendingPhysicianNames[Constants.PERSON_FIRST_NAME] ||
                   attendingPhysician.LastName != attendingPhysicianNames[Constants.PERSON_LAST_NAME] ||
                   attendingPhysician.MiddleName != attendingPhysicianNames[Constants.PERSON_MIDDLE_NAME])
                    {
                        //update attending physician
                        attendingPhysician.FirstName = attendingPhysicianNames[Constants.PERSON_FIRST_NAME];
                        attendingPhysician.MiddleName = attendingPhysicianNames[Constants.PERSON_MIDDLE_NAME];
                        attendingPhysician.LastName = attendingPhysicianNames[Constants.PERSON_LAST_NAME];
                        EIS.UpdateMedicalPerson(attendingPhysician);
                    }
                }
                bmsEntity.AttendingPhysicianId = attendingPhysician.Id;
            }
            else
                bmsEntity.AttendingPhysicianId = null;

        }

        protected override void TranslateFromVistaToBMS(Patient entity, EisPatient bmsEntity)
        {
            TranslatePatientFromVistAToBMS(entity, bmsEntity, VistaSite);
        }

        protected override void InsertBmsEntity(EisPatient bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.CreatePatient(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected override void UpdateBmsEntity(EisPatient bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.UpdatePatient(bmsEntity);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        protected override void DeleteBmsEntity(EisPatient bmsEntity)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EIS.DeletePatient(bmsEntity.Id);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
