﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Data;
using BMS.VistaWorker.Abstract;
using InfoWorld.HL7.ITS;
using BMS.ServicesWrapper.EVS;
using BMS.VistaWorker.Writer.Concrete.CatchExceptions;

namespace BMS.VistaWorker.Writer.Concrete
{
    /// <summary>
    /// Abstract base writer for evs entities.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    abstract class BaseEvsWriter<T> : BaseEisWriter<T, CD> where T : IConcept
    {

        public BaseEvsWriter() : base(CatchExceptionFactory.CatchEvsExceptions) { }

        protected abstract string CodeSystem { get; }
        protected abstract string CodeSystemName { get; }

        /// <summary>
        /// Gets the concept.
        /// </summary>
        /// <param name="concept">The EVS concept.</param>
        /// <returns></returns>
        protected override CD GetBmsEntity(T concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IList<CD> cds = EVS.GetCodes(new CodeFilterParameters() { VocabularyDomain = CodeSystemName, MaxSelectedCodes = int.MaxValue });
                string code = VistaSite.Code + "_" + concept.IEN.ToString();
                return cds.FirstOrDefault(cd => cd.code == code);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Translates the VistA entity to an existing EVS concept (for use in BMS).
        /// </summary>
        /// <param name="entity">The VistA entity.</param>
        /// <param name="concept">The EVS concept.</param>
        protected override void TranslateFromVistaToBMS(T entity, CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                concept.code = VistaSite.Code + "_" + entity.IEN.ToString();
                concept.codeSystem = CodeSystem;
                concept.codeSystemName = CodeSystemName;
                concept.displayName = entity.Name;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Inserts the BMS entity.
        /// </summary>
        /// <param name="concept">The concept.</param>
        protected override void InsertBmsEntity(CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVS.InsertConcept(concept);
                EVS.FlushCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        /// <summary>
        /// Updates the BMS entity.
        /// </summary>
        /// <param name="concept">The concept.</param>
        protected override void UpdateBmsEntity(CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVS.UpdateConcept(concept);
                EVS.FlushCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Deletes the BMS entity.
        /// </summary>
        /// <param name="concept">The concept.</param>
        protected override void DeleteBmsEntity(CD concept)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EVS.DeleteConcept(concept);
                EVS.FlushCache();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


    }
}
