﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using BMS.VistaWorker.Abstract;
using BMS.VistaWorker.Writer;
using BMS.VistaWorker.Reader.Abstract;
using BMS.VistaWorker.Reader.LinqToSql;
using BMS.VistaWorker.Reader.Concrete;
using System.Configuration;

namespace BMS.VistaWorker.Reader
{
    /// <summary>
    /// Factory for reader manager - creates a reader manager based on connection strings.
    /// </summary>
    public static class ReaderManagerFactory
    {

        public const string KeyVistaWorkerDelayTimeSpan = "VistaWorker.WorkerDelayTimeSpan";
        public const string KeyVistaWorkerPeriodTimeSpan = "VistaWorker.WorkerPeriodTimeSpan";
        public const string KeyVistaWorkerBulkSize = "VistaWorker.BulkSize";

        /// <summary>
        /// Creates a reader manager with timer.
        /// </summary>
        /// <param name="connectionString">The connection string.</param>
        /// <param name="delay">The delay.</param>
        /// <param name="period">The period.</param>
        /// <param name="deleteOnProcessingSuccess">if set to <c>true</c> [delete on processing success].</param>
        /// <returns></returns>
        public static ReaderManagerTimer MakeReaderManagerWithTimer(String connectionString, int? bulkSize = null, TimeSpan? delay = null, TimeSpan? period = null)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                TimeSpan pDelay = delay ?? TimeSpan.Parse(ConfigurationManager.AppSettings[KeyVistaWorkerDelayTimeSpan]);
                TimeSpan pPeriod = period ?? TimeSpan.Parse(ConfigurationManager.AppSettings[KeyVistaWorkerPeriodTimeSpan]);
                ReaderManager readerManager = MakeReaderManager(connectionString, bulkSize);
                ReaderManagerTimer readerManagerTimer = new ReaderManagerTimer(readerManager, pDelay, pPeriod);
                return readerManagerTimer;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Creates a reader manager.
        /// </summary>
        /// <param name="connectionString">The connection string.</param>
        /// <returns></returns>
        private static ReaderManager MakeReaderManager(String connectionString, int? bulkSize = null)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                int pBulkSize = bulkSize ?? int.Parse(ConfigurationManager.AppSettings[KeyVistaWorkerBulkSize]);
                IRepository repository = new Repository(connectionString, pBulkSize);
                ReaderManager readerManager = new ReaderManager(repository);
                return readerManager;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


    }
}
