﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Security;
using System.ServiceModel.Channels;
using BMS.Facade.Data;
using BMS.ServicesWrapper.BMService;
using System.ServiceModel;
using System.ServiceModel.Security;
using BMS.ServicesWrapper.Security;
using InfoWorld.HL7.ITS;
using InfoWorld.Security.Authentication;
using BMS.Utils;
using BMS.Facade;

namespace BMS.Authentication
{
    /// <summary>
    /// Custom membership provider implementation
    /// </summary>
    public class BmsMembershipProvider : MembershipProvider
    {
        /// <summary>
        /// Passes the user name and password to the security service
        /// </summary>
        /// <param name="username">user name</param>
        /// <param name="password">password</param>
        /// <returns></returns>
        public override bool ValidateUser(string username, string password)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                HttpContext.Current.Request.Cookies.Remove("SamlToken");
                HttpContext.Current.Session["User"] = null;
                HttpContext.Current.Session["UserName"] = null;

                string fullUsername;
                bool loginSucces = LogIn(username, password, out fullUsername);
                if (!loginSucces)
                    return false;

                HttpContext.Current.Session["UserName"] = fullUsername;
                User user = new User() { UserName = fullUsername };
                Security.DurableIssuedToken.IssuedTokenCache itc = BMS.Security.DurableIssuedToken.PersistenceIssuedTokenCacheFactory.GetIssuedTokenCache();
                user.Id = new II(user.Domain, itc.TryGetUserSidFromToken(HttpContext.Current.Session.SessionID));

                FillUserData(user);

                user.Salt = Utils.CustomEncryption.GetRandomSalt(user.UserName);
                HttpContext.Current.Session["User"] = user;
                return true;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        private bool LogIn(string username, string password, out string fullUsername, bool retry = false)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                AuthenticationServiceClient clientAutentificare = new AuthenticationServiceClient();
                clientAutentificare.ClientCredentials.UserName.UserName = username;
                clientAutentificare.ClientCredentials.UserName.Password = password;

                try
                {
                    fullUsername = clientAutentificare.Login();
                    clientAutentificare.Close();
                    return true;
                }
                catch (SecurityNegotiationException)
                {
                    clientAutentificare.Abort();

                    User user = BMS.ServicesWrapper.Security.SecurityFactory.InstanceWindows.GetUserByUserName(username);
                    var availableDomains = SecurityFactory.InstanceWindows.GetAvailableDomains(user.UserName, user.Id.extension);
                    if (!availableDomains.Any())
                    {
                        string firstDomain = SecurityFactory.InstanceWindows.GetAvailableDomains().First();
                        SecurityFactory.InstanceWindows.SetAvailableDomains(user.UserName, user.Id.extension, new string[] { firstDomain });

                        if (retry)
                            throw new InvalidProgramException("Available domains set stack overflow");
                        return LogIn(username, password, out fullUsername, true);
                    }
                    else
                        throw;
                }
                catch (MessageSecurityException)
                {
                    clientAutentificare.Abort();
                    fullUsername = null;
                    return false;
                }
                catch (Exception)
                {
                    clientAutentificare.Abort();
                    throw;
                }

            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void FillUserData(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<CD> YesNoValues = FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                CD yes = YesNoValues.First(x => x.code == Constants.Yes);
                CD no = YesNoValues.First(x => x.code == Constants.No);
                // default set all roles with no/false
                user.EMSUser = user.GuestUser = user.NationalUser = user.RegionalUser = user.SiteUser = user.SupportUser = user.VisnUser = user.AdminUser = user.AuditLogUser = user.EMSSupervisorUser = no;
                user.IsSuperUser = false;
                List<string> roles = BMS.ServicesWrapper.Security.SecurityFactory.Instance.GetAllUserRoles();
                if (roles != null && roles.Count > 0)
                {
                    if (roles.Contains(UserRolePermissions.SuperUsers))
                    {
                        // set all roles with yes/true because user is super user
                        user.EMSUser = user.GuestUser = user.NationalUser = user.RegionalUser = user.SiteUser = user.SupportUser = user.VisnUser = user.AdminUser = user.AuditLogUser = user.EMSSupervisorUser = yes;
                        user.IsSuperUser = true;
                    }
                    else
                    {
                        foreach (string role in roles)
                        {
                            switch (role)
                            {
                                case UserRolePermissions.EmsStaff:
                                    user.EMSUser = yes;
                                    break;
                                case UserRolePermissions.Guests:
                                    user.GuestUser = yes;
                                    break;
                                case UserRolePermissions.NationalUsers:
                                    user.NationalUser = yes;
                                    break;
                                case UserRolePermissions.RegionalUsers:
                                    user.RegionalUser = yes;
                                    break;
                                case UserRolePermissions.SiteUsers:
                                    user.SiteUser = yes;
                                    break;
                                case UserRolePermissions.SupportUsers:
                                    user.SupportUser = yes;
                                    break;
                                case UserRolePermissions.VisnUsers:
                                    user.VisnUser = yes;
                                    break;
                                case UserRolePermissions.Administrators:
                                    user.AdminUser = yes;
                                    break;
                                case UserRolePermissions.AuditLogUsers:
                                    user.AuditLogUser = yes;
                                    break;
                                case UserRolePermissions.EMSSupervisorUsers:
                                    user.EMSSupervisorUser = yes;
                                    break;
                            }
                        }
                    }
                }

                if (!user.IsSuperUser)
                    CheckHasRole(user);

                user.AvailableDomains = new string[0];
                user.Domain = BMS.ServicesWrapper.Security.SecurityFactory.Instance.GetCurrentDomain();

                User userProfile = Facade.FacadeManager.ConfigurationInterface.GetUserProfile(user.UserName, user.Domain);
                user.Facility = userProfile.Facility;
                user.VistaSite = userProfile.VistaSite;
                user.Visn = userProfile.Visn;
                user.Region = userProfile.Region;
                user.DefaultFacility = userProfile.DefaultFacility;
                if (userProfile.DefaultFacility == null)
                {
                    if (roles != null && roles.Count > 1)
                    {
                        List<string> definedRoles = Enum.GetNames(typeof(Util.UserRoles)).ToList();
                        roles = ((from a in roles
                                  where !(from b in definedRoles select b).Contains(a)
                                  select a)).ToList();
                        roles.RemoveAll(a => a.Equals(user.UserName, StringComparison.InvariantCultureIgnoreCase));
                        if (roles != null && roles.Count > 0)
                        {
                            User profile = null;
                            foreach (string role in roles)
                            {
                                profile = Facade.FacadeManager.ConfigurationInterface.GetUserProfile(role, user.Domain);
                                if (profile.DefaultFacility != null)
                                {
                                    userProfile.DefaultFacility = profile.DefaultFacility;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void CheckHasRole(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                bool hasRole = user.GuestUser.IsYes() ||
                    user.EMSUser.IsYes() ||
                    user.NationalUser.IsYes() ||
                    user.RegionalUser.IsYes() ||
                    user.SiteUser.IsYes() ||
                    user.SupportUser.IsYes() ||
                    user.VisnUser.IsYes() ||
                    user.AdminUser.IsYes() ||
                    user.AuditLogUser.IsYes() ||
                    user.EMSSupervisorUser.IsYes();
                if (!hasRole)
                    throw new BMS.Authentication.UserHasNoRoleException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override string ApplicationName
        {
            get { return String.Empty; }                
            set { }
        }

        #region not implemented

        public override bool ChangePassword(string username, string oldPassword, string newPassword)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override bool ChangePasswordQuestionAndAnswer(string username, string password, string newPasswordQuestion, string newPasswordAnswer)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override MembershipUser CreateUser(string username, string password, string email, string passwordQuestion, string passwordAnswer, bool isApproved, object providerUserKey, out MembershipCreateStatus status)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override bool DeleteUser(string username, bool deleteAllRelatedData)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override bool EnablePasswordReset
        {
            get { return false; }
        }

        public override bool EnablePasswordRetrieval
        {
            get { return false; }
        }

        public override MembershipUserCollection FindUsersByEmail(string emailToMatch, int pageIndex, int pageSize, out int totalRecords)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override MembershipUserCollection FindUsersByName(string usernameToMatch, int pageIndex, int pageSize, out int totalRecords)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override MembershipUserCollection GetAllUsers(int pageIndex, int pageSize, out int totalRecords)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override int GetNumberOfUsersOnline()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override string GetPassword(string username, string answer)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override MembershipUser GetUser(string username, bool userIsOnline)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override MembershipUser GetUser(object providerUserKey, bool userIsOnline)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override string GetUserNameByEmail(string email)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override int MaxInvalidPasswordAttempts
        {
            get { return 0; }
        }

        public override int MinRequiredNonAlphanumericCharacters
        {
            get { return 0; }
        }

        public override int MinRequiredPasswordLength
        {
            get { return 0; }
        }

        public override int PasswordAttemptWindow
        {
            get { return 0; }
        }

        public override MembershipPasswordFormat PasswordFormat
        {
            get { return MembershipPasswordFormat.Clear; }
        }

        public override string PasswordStrengthRegularExpression
        {
            get { return null; }
        }

        public override bool RequiresQuestionAndAnswer
        {
            get { return false; }
        }

        public override bool RequiresUniqueEmail
        {
            get { return false; }
        }

        public override string ResetPassword(string username, string answer)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override bool UnlockUser(string userName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public override void UpdateUser(MembershipUser user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                throw new NotImplementedException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

    }
}
