﻿using System.Collections.Generic;
using BMS.Facade.Service;
using BMS.Facade.Data;
using BMS.ServicesWrapper.Security;
using BMS.Facade.Fault;
using BMS.Facade.Translators;
using System.Web.SessionState;
using System;
using BMS.Utils;
using InfoWorld.HL7.ITS;
using System.Linq;
using BMS.ServicesWrapper.EIS;
using BMS.ServicesWrapper.BMService;

namespace BMS.Facade.Implementation
{
    public class UserOperationsImplementation : IUserOperations
    {

        public User GetProfile()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                User user = System.Web.HttpContext.Current.Session["User"] as User;
                if (user == null)
                    throw new UserProfileEmptyException();
                return user;                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public User GetDefaultUserProfile()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return SecurityFactory.Instance.GetDefaultUserProfile();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public User GetUser(string userId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return SecurityFactory.Instance.GetUser(userId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public bool CheckPermission(string permissionName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                User profile = GetProfile();
                if (profile.IsSuperUser)
                    return true;
                string userName = profile.UserName;
                bool result = SecurityFactory.Instance.CheckPermission(userName, permissionName);
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<UserOperations> GetUserOperations(II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return SecurityFactory.Instance.GetUserOperations(facilityId).Sort("UserName").ToList();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public void AddUserOperations(string userName, List<PermissionFacility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SecurityFactory.Instance.AddUserOperations(userName, facilities);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void RemoveUserOperations(string userName, List<PermissionFacility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                SecurityFactory.Instance.RemoveUserOperations(userName, facilities);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<User> SearchUsers(string domain, string searchUsername)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (string.IsNullOrWhiteSpace(searchUsername))
                    return null;
                return SecurityFactory.Instance.SearchUsers(domain, searchUsername);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public IList<string> GetUserDomains()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<string> domains = SecurityFactory.Instance.GetActiveDirectoryDomains();
                if (!domains.Contains("Local"))
                    domains.Add("Local");
                return domains.ToArray();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static bool ToBool(CD cd)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (cd.code == "No")
                    return false;
                if (cd.code == "Yes")
                    return true;
                throw new InvalidOperationException();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SaveUserRoles(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UserRoles roles = new UserRoles();
                roles.UserName = user.UserName;
                roles.UserId = user.Id.extension;

                roles.IsEMSUser = ToBool(user.EMSUser);
                roles.IsGuestUser = ToBool(user.GuestUser);
                roles.IsNationalUser = ToBool(user.NationalUser);
                roles.IsRegionalUser = ToBool(user.RegionalUser);
                roles.IsSiteUser = ToBool(user.SiteUser);
                roles.IsAdministrator = ToBool(user.AdminUser);
                roles.IsSupportUser = ToBool(user.SupportUser);
                roles.IsVisnUser = ToBool(user.VisnUser);
                roles.IsAuditLogUser = ToBool(user.AuditLogUser);
                roles.IsAuditLogUser = ToBool(user.AuditLogUser);
                roles.IsEMSSupervisorUser = ToBool(user.EMSSupervisorUser);

                SecurityFactory.Instance.SaveUserRoles(roles);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveUserData(User user, List<PermissionFacility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                user.IsSuperUser = SecurityFactory.Instance.GetUserIsSuperUser(user.UserName);
                if (!user.IsSuperUser)
                {
                    List<Facility> allfacilities = FacadeManager.EntityInterface.GetFacilities().ToList();
                    List<PermissionFacility> oldPermissionFacilites = FacadeManager.UserInterface.GetFacilitiesForReadAndWrite(user.UserName, allfacilities);
                    List<PermissionFacility> permissionsToRemove = new List<PermissionFacility>();
                    if (oldPermissionFacilites != null && oldPermissionFacilites.Count > 0 && facilities != null && facilities.Count > 0)
                    {
                        PermissionFacility oldPf = null;
                        foreach (PermissionFacility pf in facilities)
                        {
                            oldPf = oldPermissionFacilites.Where(a => a.Facility.Id.extension.Equals(pf.Facility.Id.extension, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault();
                            if (oldPf != null && oldPf.ReadAccess == pf.ReadAccess && oldPf.WriteAccess == pf.WriteAccess)
                            {
                                oldPermissionFacilites.Remove(oldPf);
                                permissionsToRemove.Add(pf);
                            }
                        }
                        foreach (PermissionFacility pf in permissionsToRemove)                        
                            facilities.Remove(pf);
                    }
                    if (oldPermissionFacilites != null && oldPermissionFacilites.Count > 0)
                        RemoveUserOperations(user.UserName, oldPermissionFacilites);
                    if (facilities != null && facilities.Count > 0)
                        AddUserOperations(user.UserName, facilities);
                }
                SaveUserRoles(user);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void SaveUserData(User user, PermissionFacility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                user.IsSuperUser = SecurityFactory.Instance.GetUserIsSuperUser(user.UserName);
                List<PermissionFacility> permissionFacilites = FacadeManager.UserInterface.GetFacilitiesForReadAndWrite(user.UserName, new List<Facility>() { user.Facility });
                if (permissionFacilites != null && permissionFacilites.Count > 0)
                    RemoveUserOperations(user.UserName, permissionFacilites);
                if (facility != null)
                    SaveUserFacilities(user, new List<PermissionFacility>() { facility });
                SaveUserRoles(user);                
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }

        }

        private void ClearUserFacilities(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!user.IsSuperUser)
                {
                    List<Facility> allfacilities = FacadeManager.EntityInterface.GetFacilities().ToList();
                    List<PermissionFacility> permissionFacilites = FacadeManager.UserInterface.GetFacilitiesForReadAndWrite(user.UserName, allfacilities);
                    if (permissionFacilites != null && permissionFacilites.Count > 0)
                        RemoveUserOperations(user.UserName, permissionFacilites);
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private void SaveUserFacilities(User user, List<PermissionFacility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!user.IsSuperUser)
                    AddUserOperations(user.UserName, facilities);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static CD CreateStrictDecisionCD(bool value)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return value ?
                    YesNoValues.First(x => x.code == Constants.Yes) :
                    YesNoValues.First(x => x.code == Constants.No);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private static CDList YesNoValues
        {
            get
            {
                DateTime entryInLogMethodTime = DateTime.UtcNow;
                if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
                }
                try
                {
                    return FacadeManager.VocabularyInterface.GetVocabulary(Util.Vocabulary.StrictDecision);
                }
                finally
                {
                    if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                    {
                        InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                    }
                }
            }
        }      

        public void FillUserData(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UserRoles userRoles = user.Id != null ? SecurityFactory.InstanceWindows.GetUserRoles(user.UserName, user.Id.extension) :
                                     new UserRoles();

                user.EMSUser = CreateStrictDecisionCD(userRoles.IsEMSUser);
                user.GuestUser = CreateStrictDecisionCD(userRoles.IsGuestUser);
                user.NationalUser = CreateStrictDecisionCD(userRoles.IsNationalUser);
                user.RegionalUser = CreateStrictDecisionCD(userRoles.IsRegionalUser);
                user.SiteUser = CreateStrictDecisionCD(userRoles.IsSiteUser);
                user.SupportUser = CreateStrictDecisionCD(userRoles.IsSupportUser);
                user.VisnUser = CreateStrictDecisionCD(userRoles.IsVisnUser);
                user.AdminUser = CreateStrictDecisionCD(userRoles.IsAdministrator);
                user.AuditLogUser = CreateStrictDecisionCD(userRoles.IsAuditLogUser);
                user.EMSSupervisorUser = CreateStrictDecisionCD(userRoles.IsEMSSupervisorUser);
                user.AvailableDomains = new string[0];
                SecurityFactory.InstanceWindows.CheckUserRoleExists(user.UserName, user.Id.extension);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public bool CheckBMSReadPermission(User user, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (user.IsSuperUser)
                    return true;
                return SecurityFactory.Instance.CheckBMSReadPermission(user.UserName, facilityId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public bool CheckBMSWritePermission(User user, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (user.IsSuperUser)
                    return true;
                return SecurityFactory.Instance.CheckBMSWritePermission(user.UserName, facilityId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<PermissionFacility> GetFacilitiesForReadAndWrite(string userName, List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return SecurityFactory.Instance.GetFacilitiesForReadAndWrite(userName, facilities);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<Facility> GetFacilitiesWithReadAccess(User user, List<Facility> facilities)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (user.IsSuperUser)
                    return facilities;
                return SecurityFactory.Instance.GetFacilitiesWithReadAccess(user.UserName, facilities);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<User> GetUsersByRole(string roleName)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return SecurityFactory.Instance.GetUsersByRole(roleName);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<User> GetUsersByRoleAndFacility(string roleName, II facilityId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                return SecurityFactory.Instance.GetUsersByRoleAndFacility(roleName, facilityId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void InsertLoggedUser(string sessionId, string userName, Facility facility, Visn visn)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.InsertLoggedUsers(
                        new DataContracts.LoggedUser()
                        {
                            SessionId = sessionId,
                            UserName = userName,
                            FacilityId = (facility != null) ? facility.Id : null,
                            VisnId = (visn != null) ? visn.Id : null
                        });
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public Dictionary<string, int> GetLoggedUsersCount(string domainId, string facilityIdList)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Dictionary<string, int> result = new Dictionary<string, int>();
                bool _isFacility = false;
                if (!string.IsNullOrEmpty(facilityIdList))
                    _isFacility = true;
                List<string> keys = new List<string>();
                if (_isFacility)
                {
                    foreach (string facilityId in facilityIdList.Split(','))
                        keys.Add(facilityId.ToLower());
                }
                else
                {
                    List<Visn> visns = FacadeManager.EntityInterface.GetVisns().ToList();
                    foreach (Visn visn in visns)
                        keys.Add(visn.Id.extension.ToLower());
                }
                List<DataContracts.LoggedUser> loggedUsers = BMSFactory.BedManagerOperationsClient.GetLoggedUsers(domainId, facilityIdList);
                foreach (string key in keys)
                {
                    if (_isFacility)
                        result.Add(key, loggedUsers.Where(a => a.FacilityId.extension == key).ToList().Count);
                    else
                        result.Add(key, loggedUsers.Where(a => a.VisnId.extension == key).ToList().Count);
                }
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void DeleteEmsStaff(II emsStaffId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.DeleteEmsStaff(emsStaffId);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void CreateEmsStaff(EmsStaff emsStaff)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.CreateEmsStaff(emsStaff.ToDataContract());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public void UpdateEmsStaff(EmsStaff emsStaff)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BMSFactory.BedManagerOperationsClient.UpdateEmsStaff(emsStaff.ToDataContract());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public EmsStaff GetEmsStaff(II emsStaffId)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                EmsStaff result = BMSFactory.BedManagerOperationsClient.GetEmsStaff(emsStaffId).ToFacadeContract();
                if (result.Facility != null)
                    result.Facility = EISFactory.InstanceFromWCF.GetFacility(result.Facility.Id);
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public List<EmsStaff> FilterEmsStaff(Facility facility)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                List<EmsStaff> result = BMSFactory.BedManagerQueryClient.FilterEmsStaff(facility.Id).Select<DataContracts.EmsStaff, EmsStaff>(a => a.ToFacadeContract()).ToList();
                foreach (EmsStaff es in result)
                    es.Facility = facility;
                return result;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }        

        public bool CheckUserHasRole(User user)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UserRoles userRoles = user.Id != null ? SecurityFactory.InstanceWindows.GetUserRoles(user.UserName, user.Id.extension) : new UserRoles();
                return userRoles.IsGuestUser || userRoles.IsEMSUser || userRoles.IsNationalUser || userRoles.IsRegionalUser || userRoles.IsSiteUser || userRoles.IsSupportUser ||
                        userRoles.IsVisnUser || userRoles.IsAdministrator || userRoles.IsAuditLogUser || userRoles.IsEMSSupervisorUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }
    }
}
