module.exports = {
	writeNote: function(params, session, ewd) {
		this.checkAndSetupWriteNoteParams(params);
		params.timestamp = this.getSystemTime(params, session, ewd);
		
		if (this.isCosignerRequired(params, session, ewd) && (!params.cosignerDUZ || params.cosignerDUZ == '')) {
			throw new Error('Missing cosigner');
		}
		
		var newNoteId = this.createNote(params, session, ewd);
		params.noteId = newNoteId;
		this.lockNote(newNoteId, session, ewd);
		var updatedNoteId = this.updateNote(params, session, ewd);
		if (updatedNoteId != newNoteId) {
			this.unlockNote(newNoteId, session, ewd);
			throw new Error('Problem when updating note after create: ' + updatedNoteId);
		}
		
		var result = this.setNoteText(params, session, ewd);
		this.unlockNote(newNoteId, session, ewd);
		return result;
	},
	
	checkAndSetupWriteNoteParams: function(params) {
		if (!params.patientId || params.patientId == '') {
			throw new Error('Missing patientId');
		}
		if (!params.titleIEN || params.titleIEN == '') {
			throw new Error('Missing titleIEN');
		}
		
		if (!params.encounterString || params.encounterString == '') {
			throw new Error('Missing encounterString');
		}
		params.encounter = this.getEncounterFromString(params.encounterString);
		
		if (!params.text || params.text == '') {
			throw new Error('Missing text');
		}
		if (!params.authorDUZ || params.authorDUZ == '') {
			throw new Error('Missing authorDUZ');
		}
		
		// if RAPTOR doesn't pass userId, assume user is author
		if (!params.userId || params.userId == '') {
			params.userId = params.authorDUZ;
		}
		
		return {ok:true};
	},
	
	setNoteText: function(params, session, ewd) {
		var maxCharsPerLine = 80;
		var linesToWrite = [];
		var lines = params.text.split("|");
		for (var i = 0; i < lines.length; i++) {
			var curLine = this.trimEnd(lines[i]);
			if (curLine == "") {
				linesToWrite.push(curLine);
				continue;
			}
			
			while (curLine.length > maxCharsPerLine) {
				var idx = this.getFirstWhitespaceBeforeIndex(curLine, maxCharsPerLine);
				if (idx < 0) {
					idx = maxCharsPerLine;
				}
				linesToWrite.push(curLine.substr(0, idx));
				while (idx < curLine.length && this.isWhitespace(curLine[idx])) {
					idx++;
				}
				curLine = curLine.substr(idx);
			}
			
			if (curLine.length > 0) {
				linesToWrite.push(curLine);
			}
		}
		
		var npages = Math.floor((linesToWrite.length / 300) + 1);
		var nlinePerPage = 0;
		var pageNum = 1;
		
		var list = { "TEXT" : {} };
		for (var i = 0; i < linesToWrite.length; i++) {
			list["TEXT"][(i+1).toString()] = {};
			list["TEXT"][(i+1).toString()]["0"] = this.filterString(linesToWrite[i]); // TODO - per MDWS line 1300 - need to filter string for VistA
			nlinePerPage++;
			if (nlinePerPage == 300) {
				list["HDR"] = pageNum.toString() + "^" + npages.toString() + "^";
				// send to Vista and check for error
				this.writeNoteText(params.noteId, list, pageNum, npages, session, ewd); 
				list["TEXT"] = {}; // clear this out so it's not being re-set each time
				pageNum++;
				nlinePerPage = 0;
			}
		}
		
		if (nlinePerPage > 0) {
			list["HDR"] = pageNum.toString() + "^" + npages.toString() + "^";
			this.writeNoteText(params.noteId, list, pageNum, npages, session, ewd); 
			pageNum++;
		}
		
		return { id: params.noteId, totalPages: npages, lastPageRecd: pageNum };
	},
	
	writeNoteText: function(noteId, list, lastPageSent, totalPages, session, ewd) {
		var params = { rpcName: "TIU SET DOCUMENT TEXT" };
		params.rpcArgs = [];
		params.rpcArgs.push({ type: "LITERAL", value: noteId });
		params.rpcArgs.push({ type: "LIST", value: list });
		params.rpcArgs.push({ type: "LITERAL", value: "0" });
		
		var response = vistaLib.runRpc(params, session, ewd);
		
		var pieces = response.value.split("^");
		if (pieces[0] == "0") {
			throw new Error("Problem writing note: " + JSON.stringify(response));
		}
		if (pieces[0] != noteId) {
			throw new Error("IEN mismatch");
		}
		if (Number(pieces[1]) != lastPageSent) {
			throw new Error("Last page received not last page sent");
		}
		if (Number(pieces[2]) != totalPages) {
			throw new Error("Total pages mismatch");
		}
	},
	
	filterString: function(s) {
		var result = "";
		for (var i = 0; i < s.length; i++) {
			var charCode = s.charCodeAt(i);
			if (charCode == 9) {
				result += "        ";
			} else if ((charCode >= 32 && charCode <= 127) || (charCode >= 161 || charCode <= 255)) {
				result += s[i];
			} else if (charCode >= 128 && charCode <= 159) {
				result += "?";
			} else if (charCode == 10 || charCode == 13 || charCode == 160) {
				result += " ";
			}
		}
		return result;
	},
	
	getFirstWhitespaceBeforeIndex: function(s, index) {
		while (index >= 0 && !this.isWhitespace(s[index])) {
			index--;
		}
		return index;
	},
	
	isWhitespace: function(c) {
		return (c == ' ' || c == '\t' || c == '\n' || c == '\r')
	},
	
	trimEnd: function(s) {
		return s.replace(/\s+$/, "");
	},
	
	getEncounterFromString: function(encounterString) {
		var pieces = encounterString.split(";");
		if (pieces.length != 3) {
			throw new Error('Invalid encounter string: does not contain 3 parts');
		}
		if (pieces[2] != "A" && pieces[2] != "H" && pieces[2] != "E")
		{
			throw new Error("Invalid encounter string: type must be A, H or E");
		}
        return {
            locationId: pieces[0],
            timestamp: pieces[1],
            type: pieces[2]
		}
	},
	
	isCosignerRequired: function(params, session, ewd) {
		if (this.userRequiresCosignature(params.userId, session, ewd)) {
			return true;
		}
		if (this.userRequiresCosignatureForNote(params.userId, params.titleIEN, session, ewd)) {
			return true;
		}
		if (params.userId !== params.authorDUZ) {
			if (this.userRequiresCosignature(params.authorDUZ, session, ewd)) {
				return true;
			}
			if (this.userRequiresCosignatureForNote(params.authorDUZ, params.titleIEN, session, ewd)) {
				return true;
			}
		}
		return false;
	},
	
	userRequiresCosignature: function(duz, session, ewd) {
		return this.userRequiresCosignatureForNote(duz, "3", session, ewd); // not sure what '3' means... from MDWS
	},

	userRequiresCosignatureForNote: function(duz, titleIEN, session, ewd) {
		var params = {};
		params.rpcName = "TIU REQUIRES COSIGNATURE";
		params.rpcArgs = [];
		params.rpcArgs.push({ type: "LITERAL", value: titleIEN });
		params.rpcArgs.push({ type: "LITERAL", value: "0" });
		params.rpcArgs.push({ type: "LITERAL", value: duz });
		
		return (vistaLib.runRpc(params, session, ewd)).value == "1";
	},
	
	updateNote: function(params, session, ewd) {
		params.rpcName = "TIU UPDATE RECORD";
		params.rpcArgs = [];
		params.rpcArgs.push({ type: "LITERAL", value: params.noteId });

		var list = { };
		list[".01"] = params.titleIEN;
		list["1202"] = params.authorDUZ;
		if (params.cosignerDUZ && params.cosignerDUZ != '') {
			list["1208"] = params.cosignerDUZ;
		}
		// skipping consult ID (key '1405') from MDWS code because only writing RAPTOR notes!
		list["1301"] = params.timestamp;
		if (params.subject && params.subject != '') {
			if (params.subject.length > 80) {
				params.subject = params.subject.substring(0, 80);
			}
		}
		list["1701"] = params.subject || "";
		// skipping parent ID (key '2101') from MDWS code because only writing RAPTOR notes!		
		params.rpcArgs.push({ type: "LIST", value: list});
		// slipping ProcId and ProcTimestamp because they don't appear to be used anywhere in MDWS
		return vistaLib.runRpc(params, session, ewd).value;
	},
	
	createNote: function(params, session, ewd) {
		var params2 = {};
		params2.rpcName = "TIU CREATE RECORD";
		params2.rpcArgs = [];
		params2.rpcArgs.push({ type: "LITERAL", value: params.patientId });
		params2.rpcArgs.push({ type: "LITERAL", value: params.titleIEN });
		params2.rpcArgs.push({ type: "LITERAL", value: "" }); // blank per MDWS code
		params2.rpcArgs.push({ type: "LITERAL", value: "" }); // blank per MDWS code
		params2.rpcArgs.push({ type: "LITERAL", value: "" }); // blank per MDWS code

		var list = { };
		list["1202"] = params.authorDUZ;
		list["1301"] = params.timestamp;
		list["1205"] = params.encounter.locationId;
		if (params.cosignerDUZ && params.cosignerDUZ != '') {
			list["1208"] = params.cosignerDUZ;
		}
		// skipping consult ID (key '1405') from MDWS code because only writing RAPTOR notes!
		if (params.subject && params.subject != '') {
			if (params.subject.length > 80) {
				params.subject = params.subject.substring(0, 80);
			}
		}
		list["1701"] = params.subject || "";
		// skipping parent ID (key '2101') from MDWS code because only writing RAPTOR notes!
		params2.rpcArgs.push({ type: "LIST", value: list});

		var ts = this.convertUtcStringToVistaDateTime(params.encounter.timestamp);
		ts = this.trimZeroes(ts);
		var visitString = params.encounter.locationId + ";" + ts + ";" + params.encounter.type;
		params2.rpcArgs.push({ type: "LITERAL", value: visitString }); 
		
		params2.rpcArgs.push({ type: "LITERAL", value: "1" }); // per MDWS
		
		return this.toNoteFromCreate(vistaLib.runRpc(params2, session, ewd));
	},
	
	toNoteFromCreate: function(response) {
		if (!response.value || response.value.indexOf("0^") == 0) {
			throw new Error('Unable to create note: ' + JSON.stringify(response));
		}
		return response.value; // new note IEN
	},
	
	lockNote: function(noteId, session, ewd) {
		var params = {};
		params.rpcName = "TIU LOCK RECORD";
		params.rpcArgs = [{ type: "LITERAL", value: noteId }]; 
		var response = vistaLib.runRpc(params, session, ewd);
		//return response;
		return this.toLockResponse(response);
	},
	
	unlockNote: function(noteId, session, ewd) {
		var params = {};
		params.rpcName = "TIU UNLOCK RECORD";
		params.rpcArgs = [{ type: "LITERAL", value: noteId }]; 
		var response = vistaLib.runRpc(params, session, ewd);
		//return response;
		return this.toLockResponse(response);
	},
	
	toLockResponse: function(response) {
		if (!response || !response.value || response.value == '') {
			throw new Error('Invalid lock/unlock response');
		}
		var pieces = response.value.split("^");
		if (pieces[0] !== "0") {
			throw new Error('Problem locking/unlocking note: ' + pieces[1]);
		}
		return true;
	},
	
	convertUtcStringToVistaDateTime: function(s) {
	    var year = Number(s.substr(0,4));
        var result = (year - 1700).toString();
        result += s.substr(4);
        return result;
	},
	
	trimZeroes: function(s) {
		if (!s || s == '' || s.indexOf('.') < 0) {
			return s;
		}
		var endIdx = s.length - 1;
		while (s[endIdx] == '0') {
			endIdx--;
		}
		return s.substr(0, endIdx + 1);
	},

	signNote: function(params, session, ewd) {
		var encyptedSig = vistaLib.encryptRpcParameter(params.eSig, session, ewd);
		
		var params2 = {};
		params2.rpcName = "TIU SIGN RECORD";
		params2.rpcArgs = [];
		
		params2.rpcArgs.push({ type: "LITERAL", value: params.noteIEN });
		params2.rpcArgs.push({ type: "LITERAL", value: encyptedSig });
		
		return vistaLib.runRpc(params2, session, ewd);
	},
	
	getNoteTitles: function(params, session, ewd) {
		params.rpcName = "TIU LONG LIST OF TITLES";
		params.rpcArgs = [];
		params.rpcArgs.push({ type: "LITERAL", value: "3" }); // not sure what this three means... from MDWS broker code
		params.rpcArgs.push({ type: "LITERAL", value: vistaStringUtils.adjustForNameSearch(params.target) });
		return this.toNoteTitles(vistaLib.runRpc(params, session, ewd));
	},
	
	toNoteTitles: function(response) {
		var result = {};
		
		for (var i = 1; response.value.hasOwnProperty(i.toString()); i++) {
			var pieces = response.value[i.toString()].split("^");
			if (result.hasOwnProperty(pieces[0])) { // may be multiple note titles for an ID - add them to pre-existing array
				result[pieces[0]].push(pieces[1]);
			}
			else {
				result[pieces[0]] = [pieces[1]];
			}
		}
		
		return result;
	},
	
	getSurgeryReportsWithText: function(params, session, ewd) {
		params.rpcName = "ORWSR RPTLIST";
		params.rpcArgs = [{ type: "LITERAL", value: params.patientId }];
		var response = vistaLib.runRpc(params, session, ewd);
		
		// TODO - consider auto-detecting this type of thing in runRpc...
		// returns a global array reference - need to get it after calling RPC
		var resultRef = response.value;
		if (!resultRef || resultRef == "") {
			return null;
		}
		var resultGlo = vistaLib.getGlobalNodeFromRef(resultRef, ewd);
		results = resultGlo._getDocument();
		resultGlo._delete();
		// done getting data by ref

		return this.toSurgeryReports(results, params, session, ewd);
	},
	
	toSurgeryReports: function(reports, params, session, ewd) {

	//		{
	//		  "1": "CAMP MASTER;500^10007^3061208.073^LEFT INGUINAL HERNIA REPAIR WITH MESH^PROVIDER,ONE^+^LAB WORK-Yes^STATUS-(COMPLETED)^SPEC-GENERAL(OR WHEN NOT DEFINED BELOW)^DICT-^TRANS-",
	//		  "2": "CAMP MASTER;500^10003^3020311.0915^RIH^ZZTDNURSE,ONE^^LAB WORK-Yes^STATUS-(COMPLETED)^SPEC-GENERAL(OR WHEN NOT DEFINED BELOW)^DICT-^TRANS-"
	//		}

		var result = [];
		for (var i = 1; reports.hasOwnProperty(i.toString()); i++) {
			var current = {};
			var pieces = reports[i.toString()].split("^");
			
			current.id = pieces[1]; // assign ID
			params.ien = pieces[1]; // set IEN for call to fetch report text
			current.text = this.getSurgeryReportText(params, session, ewd); // immediately assign text
			// continue parsing pieces
			current.timestamp = pieces[2];
			current.title = pieces[3];
			current.author = { name: pieces[4] };
			
			var facilityPieces = pieces[0].split(";");
			current.facility = { id: facilityPieces[1], name: facilityPieces[0] };
			
			result.push(current);
		}
		
		return result;
	},
	
	getSurgeryReportText: function(params, session, ewd) {
		params.rpcName = "ORWRP REPORT TEXT";
		params.rpcArgs = [];
		params.rpcArgs.push({ type: "LITERAL", value: params.patientId });
		params.rpcArgs.push({ type: "LITERAL", value: "28:SURGERY (LOCAL ONLY)~" });
		params.rpcArgs.push({ type: "LITERAL", value: "" });
		params.rpcArgs.push({ type: "LITERAL", value: "" });
		params.rpcArgs.push({ type: "LITERAL", value: params.ien });
		params.rpcArgs.push({ type: "LITERAL", value: "0" });
		params.rpcArgs.push({ type: "LITERAL", value: "0" });
		
		var rpcRef = vistaLib.runRpc(params, session, ewd);
		// TODO - consider auto-detecting this type of thing in runRpc...
		// returns a global array reference - need to get it after calling RPC
		var resultRef = rpcRef.value;
		if (!resultRef || resultRef == "") {
			return null;
		}
		var resultGlo = vistaLib.getGlobalNodeFromRef(resultRef, ewd);
		results = resultGlo._getDocument();
		resultGlo._delete();
		// done getting data by ref
		
		var reportText = "";
		for (var i = 1; results.hasOwnProperty(i.toString()); i++) {
			reportText += (results[i.toString()][0] + "\r\n");
		}
		return reportText;
	},
	
	getSystemTime: function(params, session, ewd) {
		var params2 = { rpcName: "ORWU DT" };
		params2.rpcArgs = [{ type: "LITERAL", value: "NOW" }];
		return vistaLib.runRpc(params2, session, ewd).value;
	}
};

var vistaLib = require('VistALib');
var vistaStringUtils = require('vistaStringUtils');