package gov.va.med.pharmacy.peps.presentation.common.controller.update;

import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidationType.CROSS_SITE_SCRIPTING_REFLECTED;
import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidator.validateStringInput;
import gov.va.med.pharmacy.peps.common.exception.ValidationException;
import gov.va.med.pharmacy.peps.common.exception.ValueObjectValidationException;
import gov.va.med.pharmacy.peps.common.utility.ESAPIValidationType;
import gov.va.med.pharmacy.peps.common.utility.ESAPIValidator;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileActionVo;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.common.vo.PaginatedList;
import gov.va.med.pharmacy.peps.common.vo.Status;
import gov.va.med.pharmacy.peps.common.vo.StatusVo;
import gov.va.med.pharmacy.peps.common.vo.UpdateSearchCriteria;
import gov.va.med.pharmacy.peps.common.vo.validator.ErrorKey;
import gov.va.med.pharmacy.peps.common.vo.validator.Errors;
import gov.va.med.pharmacy.peps.presentation.common.controller.AbstractSearchController;
import gov.va.med.pharmacy.peps.presentation.common.controller.ControllerConstants;
import gov.va.med.pharmacy.peps.service.common.scheduler.FdbJobNames;
import gov.va.med.pharmacy.peps.service.common.scheduler.FdbSchedulerControlBean;
import gov.va.med.pharmacy.peps.service.common.scheduler.JobCommands;
import gov.va.med.pharmacy.peps.service.common.session.FdbSchedulerProcessService;
import gov.va.med.pharmacy.peps.service.common.session.NDFUpdateFileService;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Controller for managing and creating NDF Update Files.
 */
@Controller
public class NdfUpdateFileController extends AbstractSearchController {

    /** Key for storing the pending file id in the Model object. */
    private static final String PENDING_FILE_ID = "pendingFileId";    

    /** The NDFUpdateFileService. */
    @Resource
    private NDFUpdateFileService ndfUpdateFileService;
    
    /** The FdbSchedulerProcessService. */
    @Resource
    private FdbSchedulerProcessService fdbSchedulerProcessService;      

    /** The ndf update file action. */
    private NdfUpdateFileActionVo ndfUpdateFileAction;
    
    /** The initiated file. */
    private NdfUpdateFileVo initiatedFile;

    /** The Logger. */
    private static final Logger LOG = LogManager.getLogger(NdfUpdateFileController.class);

    /** Status Choices for files installed to QA, to approve or reject. */
    private static final List<Status> QA_RESULTS_STATUS_CHOICES = Arrays.asList(Status.NONE, Status.APPROVED,
            Status.REJECTED);

    /**
     * Inits the binder.
     *
     * @param binder the WebDataBinder, for security purposes.
     */
    @InitBinder
    protected void initBinderDTO(WebDataBinder binder) {
        binder.setDisallowedFields("dtoVar04292015");
    }

    /**
     * Manage NDF Update File Mapping GET.
     *
     * @param ndfPendingFileAction the ndf pending file action
     * @param model Model
     * @return String the view name
     */
    @RequestMapping(value = "/manageUpdateFiles.go", method = RequestMethod.GET)
    public String getUpdateFilesForTab(
            @ModelAttribute(ControllerConstants.PENDING_ACTION_KEY) NdfUpdateFileActionVo ndfPendingFileAction,
            Model model) {
        pageTrail.clearTrail();
        pageTrail.addPage("updateFileSearch", "Manage Update Files", true); 
        ndfPendingFileAction = ndfUpdateFileAction;        

        List<NdfUpdateFileVo> pendingItems = retrievePendingNdfUpdate();  
        NdfUpdateFileVo pendingItem = null;
        if(pendingItems != null && !pendingItems.isEmpty()){            
            pendingItem = pendingItems.get(0);
        }

        NdfUpdateFileVo initiatedItem = null;
        if(initiatedFile != null){
            initiatedItem = initiatedFile;            
        }
        else{
            List<NdfUpdateFileVo> initiatedItems = retrieveInitiatedNdfUpdate();          
            if(initiatedItems != null && !initiatedItems.isEmpty()){            
                initiatedItem = initiatedItems.get(0);
            }            
        }

        if(ndfUpdateFileAction != null){      
            ndfPendingFileAction.setActionSelected(ndfUpdateFileAction.getActionSelected());            
            model.addAttribute("ndfPendingAction", ndfPendingFileAction);            
        }          


        model.addAttribute("pendingFile", pendingItem); 
        model.addAttribute("initiatedFile", initiatedItem);


        clearInitatedFile();

        return "update.file.search";
    }

    /**
     * Gets the potential filtered paged update files.
     *
     * @param request the request
     * @param response the response
     * @return the filtered paged update files
     */
    @RequestMapping(value = "/historyUpdateFiles.rest", method = RequestMethod.POST,  produces = "application/json")
    @ResponseBody
    public String getFilteredPagedUpdateFiles(
            HttpServletRequest request, HttpServletResponse response) {

        UpdateSearchCriteria searchcriteria = convertToUpdateSearchCriteria(request);
        PaginatedList<NdfUpdateFileVo> pagedUpdateFiles = retrievePagedUpdatedItems(searchcriteria);

        ObjectMapper mapper = new ObjectMapper();
        String jsonPayload = "ERROR";
        try {
            jsonPayload = mapper.writeValueAsString(pagedUpdateFiles);

        } catch (Throwable t) {
            LOG.error(t.getMessage());
        }
        String cleanJsonPayload = validateStringInput(jsonPayload, CROSS_SITE_SCRIPTING_REFLECTED);
        return cleanJsonPayload;
    }

    /**
     * Gets the status list drop down values.
     *
     * @return the status list drop down values
     */
    @RequestMapping(value = "/fileStatusDropDownList.rest", method = RequestMethod.GET,  produces = "application/json")
    @ResponseBody
    public String getStatusListDropDownValues() {

        List<StatusVo> statuses = retrieveAllStatuses();

        ObjectMapper mapper = new ObjectMapper();
        String jsonPayload = "ERROR";
        try {
            jsonPayload = mapper.writeValueAsString(statuses);
        } catch (Throwable t) {
            LOG.error(t.getMessage());
        }

        String cleanJsonPayload = validateStringInput(jsonPayload, CROSS_SITE_SCRIPTING_REFLECTED);
        return cleanJsonPayload;
    }

    /**
     * Convert request information to update search criteria.
     *
     * @param request the request
     * @return the update search criteria
     */
    private UpdateSearchCriteria convertToUpdateSearchCriteria(HttpServletRequest request) {
        UpdateSearchCriteria searchCriteria = new UpdateSearchCriteria();
        if (StringUtils.isNotBlank(request.getParameter("beginningDateRange"))){
            searchCriteria.setBeginningDateRange(
                    convertStringToDate((String)request.getParameter("beginningDateRange")));
        }

        if (StringUtils.isNotBlank(request.getParameter("endingDateRange"))){
            searchCriteria.setEndingDateRange(
                    convertStringToDateAtEndofDay((String)request.getParameter("endingDateRange")));
        }
        if (StringUtils.isNotBlank(request.getParameter("statusId"))){
            searchCriteria.setStatusId(Long.parseLong((String)request.getParameter("statusId")));
        }

        if (StringUtils.isNotBlank(request.getParameter("startRow"))){
            searchCriteria.setStartRow(Integer.parseInt((String)request.getParameter("startRow")));
        }

        if (StringUtils.isNotBlank(request.getParameter("pageSize"))){
            searchCriteria.setPageSize(Integer.parseInt((String)request.getParameter("pageSize")));
        }

        return searchCriteria;
    }

    /**
     * Convert string to date.
     *
     * @param object the object
     * @return the date
     */
    private Date convertStringToDate(String object){
        Date date = null;
        try {
            date = DateUtils.parseDate(object, "MM/dd/yyyy");

        } catch (ParseException e) {

            e.printStackTrace();
        }
        return date;
    }

    /**
     * Convert string to date at end of day.
     *
     * @param object the object
     * @return the date
     */
    private Date convertStringToDateAtEndofDay(String object){
        Date date = null;
        try {
            date = DateUtils.parseDate(object, "MM/dd/yyyy");
            date = DateUtils.addHours(date, 23);
            date = DateUtils.addMinutes(date, 59);
        } catch (ParseException e) {

            e.printStackTrace();
        }
        return date;
    }
    
    /**
     * Manage NDF Update File Mapping GET.
     *
     * @param incomingAction the incoming action
     * @param fileId the file id to update
     * @param model Model
     * @return String the view name
     * @throws ValidationException Thrown if one of the field validations fails.
     */
    @RequestMapping(value = "/updateStatus.go", method = RequestMethod.GET)
    public String updateNDFUpdateFileStatus(
            @ModelAttribute(ControllerConstants.PENDING_ACTION_KEY) NdfUpdateFileActionVo incomingAction,
            @RequestParam(value = PENDING_FILE_ID, required = false) String fileId,        
            Model model) throws ValidationException{  

        ndfUpdateFileAction = incomingAction;

        pageTrail.clearTrail();
        pageTrail.addPage("updateFileSearch", "Manage Update Files", true); 

        NdfUpdateFileVo toUpdate = retrievePendingById(fileId);
        
        if(toUpdate != null){
            if(incomingAction != null){
                if (incomingAction.getComments() != null)
                toUpdate.setComments(ESAPIValidator.validateStringInput(incomingAction.getComments(), 
                        ESAPIValidationType.SQL_INJECTION).trim());
                
                if(incomingAction.getActionSelected().isApproved()){ 
                    ndfUpdateFileService.approve(toUpdate, getUser());                    
                }                                
                else if(incomingAction.getActionSelected().isRejected()){ 
                    ndfUpdateFileService.reject(toUpdate, getUser());
                }
                else{
                    //no action selected
                    Errors errors = new Errors();
                    errors.addError(FieldKey.UPDATE_FILE, ErrorKey.EMPTY_SELECTION);                    
                    throw new ValueObjectValidationException(errors);                    
                }
            }            
        }        
        clearActionFile();
        return REDIRECT + "/manageUpdateFiles.go";
    }

    /**
     * Initiate the File processing.
     *
     * @param model Model Spring MVC model object
     * @return String the view name Tiles view name
     * @throws ValidationException Thrown if a previous file is not in a final state 
     */
    @RequestMapping(value = "/createFile.go", method = RequestMethod.GET)
    public String initiateFileProcessing(Model model) throws ValidationException {

        pageTrail.clearTrail();
        pageTrail.addPage("updateFileSearch", "Manage Update Files", true);  

        NdfUpdateFileVo latestNdfUpdateFileVo = ndfUpdateFileService.retrieveCurrentStatus();
        if (latestNdfUpdateFileVo == null){
            initiatedFile = ndfUpdateFileService.createInitiatedFile(getUser());
        }        
        else if (StatusVo.NON_FINAL_STATE_STATUSES.contains(latestNdfUpdateFileVo.getStatus().getStatusId())){           
            throw new ValidationException(ValidationException.NON_FINAL_STATE_FILE_IN_PROCESS);
        }
        else{
            if (StatusVo.PPSN_ERROR == latestNdfUpdateFileVo.getStatus().getStatusId()){

                //reuse the NdfUpdateFileVo
                latestNdfUpdateFileVo.getStatus().setStatusId(StatusVo.INITIATED);
                latestNdfUpdateFileVo.setComments(null);
                latestNdfUpdateFileVo.setDirectoryPath(null);
                latestNdfUpdateFileVo.setFileName(null);
                ndfUpdateFileService.updateNdfUpdateFile(latestNdfUpdateFileVo, getUser());
                initiatedFile = latestNdfUpdateFileVo; 
            }
            else{
                initiatedFile = ndfUpdateFileService.createInitiatedFile(getUser());
            }
               
            scheduleOnDemandUpdateFileJob();
        }

        pageTrail.goToPreviousFlowUrl();
        return REDIRECT + "/manageUpdateFiles.go";
    }

    /**
     * Schedule the NDF Update Job to run one minute from now.  This appears to be a different instance of the Job than is 
     * used for the scheduled Job, which causes the disallowConcurrentExecution annotation in NdfUpdateJob to not have the 
     * desired effect.  It would be better to get a reference to the JobFactory to get the same instance of the NdfUpdateJob 
     * and add a trigger to run the job asynchronously, or triggerJob directly, which might be synchronous, causing the user 
     * to wait.  
     */
    private void scheduleOnDemandUpdateFileJob() {
        JobCommands pcommand = JobCommands.SCHEDULE;
        FdbJobNames pJobName = FdbJobNames.NDF_UPDATE_JOB;      
        FdbSchedulerControlBean schedulerControl = new  FdbSchedulerControlBean(); 
        schedulerControl.setJobCommands(JobCommands.STATUS);
    
        Calendar now = Calendar.getInstance(); 
        now.add(Calendar.MINUTE, 1);
        int dayOfWeek = now.get(Calendar.DAY_OF_WEEK);
        int hours = now.get(Calendar.HOUR_OF_DAY);        
        int mins = now.get(Calendar.MINUTE);         
        schedulerControl.setJobCommands(pcommand);        
        schedulerControl.setJobName(pJobName.getJobName());
        schedulerControl.setFdbJobNames(pJobName);
        schedulerControl.setDayOfWeek(dayOfWeek);
        schedulerControl.setHour(hours);
        schedulerControl.setMins(mins);
    
        fdbSchedulerProcessService.executeJobCommand(schedulerControl, true);
    }

    /**
     * Get the status list for the Approval/Rejection options.
     *
     * @return List
     */
    @ModelAttribute("StatusMap")
    public final List<Status> getStatusList() {
        return QA_RESULTS_STATUS_CHOICES;
    }

    /**
     * Retrieve paged updated items.
     *
     * @param searchCriteria the search criteria
     * @return the paginated list
     */
    private PaginatedList<NdfUpdateFileVo> retrievePagedUpdatedItems(UpdateSearchCriteria searchCriteria) {

        return ndfUpdateFileService.retrievePagedFileUpdates(searchCriteria);
    }

    /**
     * Retrieves a list of all update files statuses.
     * 
     * @return An array list of all file update statuses
     */
    private List<StatusVo> retrieveAllStatuses() {

        List<StatusVo> statuses = new ArrayList<StatusVo>();
        statuses = ndfUpdateFileService.retrieveAllFileStatuses();

        return statuses;
    }

    /**
     * Retrieves a list of all update files that the current user is allowed to see.
     * 
     * @return An array list of all update files
     */
    private List<NdfUpdateFileVo> retrievePendingNdfUpdate() {

        List<NdfUpdateFileVo> pendingItems = new ArrayList<NdfUpdateFileVo>();
        pendingItems = ndfUpdateFileService.retrievePendingNdfUpdate();

        return pendingItems;
    }

    /**
     * Retrieves a list of all update files that the current user is allowed to see.
     * 
     * @return An array list of all update files
     */
    private List<NdfUpdateFileVo> retrieveInitiatedNdfUpdate() {

        List<NdfUpdateFileVo> initiatedItems = new ArrayList<NdfUpdateFileVo>();        
        initiatedItems = ndfUpdateFileService.retrieveInitiatedNdfUpdate();

        return initiatedItems;
    }

    /**
     * Retrieves the file specified and sets it as the pendingItem.
     *
     * @param id the file id to retrieve
     * @return An array list of all update files
     */
    private NdfUpdateFileVo retrievePendingById(String id) {

        NdfUpdateFileVo pendingItem = new NdfUpdateFileVo();              
        pendingItem = ndfUpdateFileService.retrieveById(Long.parseLong(id));

        return pendingItem;
    }        

    /**
     * Clear action file.
     */
    private void clearActionFile(){
        ndfUpdateFileAction = null;  
    }

    /**
     * Clear initiated file.
     */
    private void clearInitatedFile(){
        initiatedFile = null;  
    }

}
