/**
 * Source file created in 2011 by Southwest Research Institute
 */

package gov.va.med.pharmacy.peps.presentation.common.controller;

import gov.va.med.pharmacy.peps.common.vo.EntityType;
import gov.va.med.pharmacy.peps.common.vo.MatchType;
import gov.va.med.pharmacy.peps.common.vo.MatchTypeVo;
import gov.va.med.pharmacy.peps.common.vo.ProductVo;
import gov.va.med.pharmacy.peps.common.vo.RematchSuggestionVo;
import gov.va.med.pharmacy.peps.common.vo.RematchVo;
import gov.va.med.pharmacy.peps.common.vo.Role;
import gov.va.med.pharmacy.peps.common.vo.UnmatchRematchVo;
import gov.va.med.pharmacy.peps.service.common.session.FDBUpdateProcessService;
import gov.va.med.pharmacy.peps.service.common.session.NDFUpdateFileService;
import gov.va.med.pharmacy.peps.service.common.session.RematchSuggestionService;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

/**
 * The controller used to process the reports
 */
@Controller("unmatchedProductsController")
public class UnmatchedProductsController extends AbstractSearchController {

    private static final Logger LOG = LogManager.getLogger(UnmatchedProductsController.class);
    private static final String MATCH_TYPE_MAP = "matchTypeMap";
    private static final String MATCH_VO = "matchVo";
    private static final String SELECTED_MATCH_TYPE = "selectedMatchType";
    private static final String MATCH_TYPE_DOT = "MatchType.";
    private static final String RESULTS_FROM_PRODUCT_SEARCH = "resultsFromProductSearch";

    private static SortedMap<MatchType, String> MATCH_MAP;
    private static String UNMATCHED_PRODUCTS_VIEW = "unmatched.product.search";
    private static String UNMATCHED_PRODUCTS_REMATCH = "unmatch.rematch";
    private static SortedMap<MatchTypeVo, String> matchTypeMap;

    @Autowired
    private RematchSuggestionService rematchSuggestionService;
    @Autowired
    private NDFUpdateFileService ndfUpdateFileService;
    @Autowired
    private FDBUpdateProcessService fDBUpdateProcessService;

    private RematchVo matchVo = new RematchVo();
    private MatchType selectedMatch;

    private ProductVo productVo = new ProductVo();

    @InitBinder
    protected void initBinderDTO(WebDataBinder binder) {
        binder.setDisallowedFields("dtoVar04292015");
    }

    /**
     * Request mapping for unmatched products
     * 
     * @throws Exception
     *             Exception
     */
    @RequestMapping(value = "/unmatchedproductscriteria.go", method = { RequestMethod.GET })
    public String unmatchedProductsTab(@ModelAttribute(MATCH_VO) RematchVo mvo, @RequestParam(
            value = "isFirstRun", required = false, defaultValue = "true") String isFirstRun, Model model,
            HttpServletRequest request, HttpSession session, Locale locale) throws Exception {

        boolean hasRun = StringUtils.isEmpty(isFirstRun) ? true : Boolean.parseBoolean(isFirstRun);
        Long currentStatus = ndfUpdateFileService.retrieveCurrentStatus().getStatus().getStatusId();
        if (currentStatus >= 7) {
            if ( currentStatus == 8 ) {
                rematchSuggestionService.removeObsoleteProducts();
            }
            rematchSuggestionService.addUnmatchedProducts(getUser());
        }

        if (hasRun) {
            initialize(locale);
            mvo.setMatchType(MatchType.INACTIVATION);

            model.addAttribute(MATCH_TYPE_MAP, MATCH_MAP);
            model.addAttribute(MATCH_VO, mvo);

            return UNMATCHED_PRODUCTS_VIEW;
        } else {

            this.matchVo.setMatchType(mvo.getMatchType());
            mvo.clearAll();

            model.addAttribute(SELECTED_MATCH_TYPE, selectedMatch);
            model.addAttribute(MATCH_TYPE_MAP, MATCH_MAP);
            model.addAttribute(MATCH_VO, mvo);

            return UNMATCHED_PRODUCTS_VIEW;
        }
    }

    @RequestMapping(value = "/searchUnmatched.go", method = { RequestMethod.GET })
    public String searchUnmatched(@ModelAttribute(MATCH_VO) RematchVo mvo, Model model, HttpServletRequest request,
            HttpSession session, HttpServletResponse pResponse, Locale locale) throws Exception {

        pageTrail.clearTrail();
        pageTrail.addPage("reports", "Product Rematch", true);

        List<RematchSuggestionVo> searchResults = null;

        if (mvo.getMatchType() != null) {
            this.matchVo.setMatchType(mvo.getMatchType());
        } else {
            if (selectedMatch != null) {
                this.matchVo.setMatchType(selectedMatch);
                mvo.setMatchType(selectedMatch);
            }
        }

        if (mvo.getMatchType() != null) {
            searchResults = rematchSuggestionService.retrieveUnmatched(Long.valueOf(mvo.getMatchType().getId()));
        }

        List<UnmatchRematchVo> unmatchedSearch = new ArrayList<UnmatchRematchVo>();

        if (searchResults != null) {
            for (RematchSuggestionVo searchResult : searchResults) {

                Long productId = rematchSuggestionService.retrieveProductId(searchResult.getOldIen());
                Long rematchedproductId = null;
                if (searchResult.getNewIen() != null) {
                    rematchedproductId = rematchSuggestionService.retrieveProductId(searchResult.getNewIen());
                }

                UnmatchRematchVo unmatchVo = new UnmatchRematchVo();
                ProductVo product =
                        (ProductVo) fDBUpdateProcessService.retrieveManagedItem(productId.toString(), EntityType.PRODUCT);
                ProductVo rematchedProduct = null;
                if (rematchedproductId != null) {
                    rematchedProduct =
                            (ProductVo) fDBUpdateProcessService.retrieveManagedItem(rematchedproductId.toString(),
                                    EntityType.PRODUCT);
                }

                unmatchVo.setId(searchResult.getEplRematchSuggestionId());

                unmatchVo.setProductId(productId);
                unmatchVo.setCmop(product.getCmopId());
                unmatchVo.setProductName(product.getVaProductName());
                unmatchVo.setIen(searchResult.getOldIen());

                unmatchVo.setMatchedIen(searchResult.getNewIen());
                unmatchVo.setMatchedProductId(rematchedProduct != null ? rematchedproductId : null);
                unmatchVo.setMatchedProductName(rematchedProduct != null ? rematchedProduct.getVaProductName() : null);
                unmatchVo.setMatchedCmop(rematchedProduct != null ? rematchedProduct.getCmopId() : null);
                unmatchVo.setMatchedScope(searchResult.getMatchScope() != null ? searchResult.getMatchScope() ? "Automatic"
                        : "Suggest" : null);

                unmatchedSearch.add(unmatchVo);
            }
        }

        this.matchVo.setUnmatchedList(unmatchedSearch);
        mvo.setUnmatchedList(unmatchedSearch);
        mvo.setMatchType(this.matchVo.getMatchType());

        model.addAttribute("unmatchedProductsList", mvo.getUnmatchedList());
        model.addAttribute(SELECTED_MATCH_TYPE, selectedMatch);
        model.addAttribute(MATCH_TYPE_MAP, MATCH_MAP);
        model.addAttribute(MATCH_VO, mvo);

        return UNMATCHED_PRODUCTS_VIEW;
    }

    /**
     * 
     * Redirects the matching page to the associate product search
     *
     * @return redirect to associate product page
     */
    @RequestMapping(value = "/searchForExistingProductToRematch.go", method = RequestMethod.GET)
    public String searchForExistingProduct() {
        RematchVo addStateBean = (RematchVo) flowScope.get(MATCH_VO);

        if (addStateBean == null) {
            return REDIRECT + "/unmatchedproductscriteria.go";
        }

        flowInputScope.put("productMatches", addStateBean.getProductList());
        flowInputScope.put("rematchList", addStateBean.getRematchList());

        return REDIRECT + "/associateProductToRematch.go?entityType=product&isFirstRun=true";
    }

    /**
     * matchResultProductSearch
     * 
     * @param entityType
     *            entityType
     * @param itemId
     *            itemId
     * @return URL
     */
    @SuppressWarnings("unchecked")
    @RequestMapping(value = "/{entity_type}/{itemId}/rematchResultProductSearch.go", method = RequestMethod.GET)
    public String matchResultProductSearch(@PathVariable(value = "entity_type") EntityType entityType, @PathVariable(
            value = "itemId") String itemId) {

        ProductVo product = (ProductVo) fDBUpdateProcessService.retrieveManagedItem(itemId, entityType);
        List<UnmatchRematchVo> rematches = (List<UnmatchRematchVo>) flowScope.get("rematchList");
        List<UnmatchRematchVo> products = (List<UnmatchRematchVo>) flowScope.get("productMatches");
        List<RematchSuggestionVo> suggestions = (List<RematchSuggestionVo>) flowScope.get("rematchSuggestions");

        if (product != null) {
            boolean containsProduct = false;

            for (UnmatchRematchVo existingMatch : rematches) {
                if (existingMatch.getMatchedProductId().equals(Long.parseLong(product.getId()))) {
                    containsProduct = true;
                }
            }

            if (rematches.isEmpty() || !containsProduct) {
                UnmatchRematchVo productMatch = products.get(0);
                UnmatchRematchVo unmatchVo = new UnmatchRematchVo();
                unmatchVo.setId(productMatch.getId());

                unmatchVo.setProductId(productMatch.getProductId());
                unmatchVo.setCmop(productMatch.getCmop());
                unmatchVo.setProductName(productMatch.getProductName());
                unmatchVo.setIen(productMatch.getIen());

                unmatchVo.setMatchedIen(String.valueOf(product.getNdfProductIen()));
                unmatchVo.setMatchedProductId(Long.parseLong(product.getId()));
                unmatchVo.setMatchedProductName(product.getVaProductName());
                unmatchVo.setMatchedCmop(product.getCmopId());

                rematches.add(unmatchVo);
                flowInputScope.put(RESULTS_FROM_PRODUCT_SEARCH, rematches);
            }
        }

        return REDIRECT + "/rematch.go";

    }

    /**
     * Rematch.
     *
     * @param mvo
     *            the ReMatchVo
     * @param id
     *            the id
     * @param pModel
     *            the model
     * @return the string
     */
    @RequestMapping(value = "/rematch.go", method = RequestMethod.GET)
    public String rematch(@ModelAttribute(MATCH_VO) RematchVo mvo, @RequestParam(value = "id", required = false) String id,
            Model pModel) {
        RematchVo addStateBean = (RematchVo) flowScope.get(MATCH_VO);
        List<UnmatchRematchVo> rematches = (List<UnmatchRematchVo>) flowScope.get("rematchList");
        List<UnmatchRematchVo> products = (List<UnmatchRematchVo>) flowScope.get("productMatches");

        if (addStateBean == null) {
            addStateBean = this.matchVo;

            flowScope.put(MATCH_VO, addStateBean);
        }

        Long selectedProductId = null;

        if (mvo.getMatchType() != null) {
            this.matchVo.setMatchType(mvo.getMatchType());
        } else {
            if (this.matchVo.getMatchType() != null) {
                this.matchVo.getMatchType();
            }
        }

        if (rematches == null || rematches.isEmpty()) {

            if (id != null) {
                products = new ArrayList<UnmatchRematchVo>();
                rematches = new ArrayList<UnmatchRematchVo>();

                List<UnmatchRematchVo> searchResults = this.matchVo.getUnmatchedList();
                if (searchResults != null && !searchResults.isEmpty()) {
                    for (UnmatchRematchVo searchResult : searchResults) {
                        if (searchResult.getId().equals(id)) {
                            selectedProductId = searchResult.getProductId();
                            products.add(searchResult);
                            if (this.matchVo.getMatchType().equals(MatchType.OTHER_CHANGE)) {
                                UnmatchRematchVo possibleRematch = new UnmatchRematchVo();
                                possibleRematch.setMatchedProductId(searchResult.getProductId());
                                possibleRematch.setMatchedProductName(searchResult.getProductName());
                                possibleRematch.setMatchedCmop(searchResult.getCmop());
                                possibleRematch.setMatchedIen(searchResult.getIen());
                                rematches.add(possibleRematch);
                            }

                        }
                    }
                }

                List<Object[]> matchedValues = rematchSuggestionService.findRematchProducts(selectedProductId);

                for (Object[] matchedValue : matchedValues) {

                    UnmatchRematchVo possibleRematch = new UnmatchRematchVo();
                    possibleRematch.setMatchedProductId(((BigDecimal) matchedValue[0]).longValue());
                    possibleRematch.setMatchedProductName((String) matchedValue[1]);
                    possibleRematch.setMatchedCmop((String) matchedValue[2]);
                    possibleRematch.setMatchedIen(((BigDecimal) matchedValue[3]).toString());

                    rematches.add(possibleRematch);

                }

                if (products != null && !products.isEmpty()) {
                    boolean containsProduct = false;
                    UnmatchRematchVo productMatch = products.get(0);
                    if (productMatch.getMatchedProductId() != null) {
                        if (rematches != null && !rematches.isEmpty()) {
                            for (UnmatchRematchVo existingMatch : rematches) {
                                if (existingMatch.getMatchedProductId().equals(productMatch.getMatchedProductId())) {
                                    containsProduct = true;
                                }
                            }
                        }

                        if (!containsProduct) {
                            rematches.add(productMatch);

                        }
                    }
                }

            }

        }

        this.matchVo.setProductList(products);
        this.matchVo.setRematchList(rematches);

        mvo.setProductList(products);
        mvo.setRematchList(rematches);
        mvo.setMatchType(this.matchVo.getMatchType());

        pModel.addAttribute(MATCH_TYPE_MAP, MATCH_MAP);
        pModel.addAttribute(MATCH_VO, mvo);
        pModel.addAttribute("nationalMgrOrSuperRole", isNationalManagerOrSupervisorRole());

        return UNMATCHED_PRODUCTS_REMATCH;
    }

    @RequestMapping(value = "/match.go", method = RequestMethod.GET)
    public String match(@ModelAttribute(MATCH_VO) RematchVo mvo, Model pModel) {

        selectedMatch = matchVo.getMatchType();

        String matchedProductIen = null;

        List<UnmatchRematchVo> rematchLists = this.matchVo.getRematchList();

        if (rematchLists != null && !rematchLists.isEmpty()) {
            if (mvo.getChosenMatch() != null) {
                for (UnmatchRematchVo matchProduct : rematchLists) {
                    if (matchProduct.getMatchedProductId().equals(mvo.getChosenMatch().getId())) {
                        matchedProductIen = matchProduct.getMatchedIen();
                    }
                }
                rematchSuggestionService.updateRematch((Long.valueOf(mvo.getTableId())), matchedProductIen,
                        Boolean.valueOf(mvo.getChosenMatch().isAuto()));
            } else {
                rematchSuggestionService.updateRematch((Long.valueOf(mvo.getTableId())), null, null);
            }
        }

        this.matchVo.getProductList().clear();
        this.matchVo.getRematchList().clear();
        this.matchVo.getUnmatchedList().clear();

        pModel.addAttribute(MATCH_TYPE_MAP, MATCH_MAP);
        pModel.addAttribute(MATCH_VO, mvo);
        pModel.addAttribute("nationalMgrOrSuperRole", isNationalManagerOrSupervisorRole());

        return REDIRECT + "/searchUnmatched.go";
    }

    @RequestMapping(value = "/cancelMatch.go", method = RequestMethod.GET)
    public String cancelMatch(@ModelAttribute(MATCH_VO) RematchVo mvo, @RequestParam(value = "id", required = false) String id,
            Model pModel) {

        selectedMatch = matchVo.getMatchType();

        this.matchVo.getProductList().clear();
        this.matchVo.getRematchList().clear();
        this.matchVo.getUnmatchedList().clear();

        return REDIRECT + "/searchUnmatched.go";
    }

    /**
     * is NationalManager Supervisor Role
     * 
     * @return true if user has PSS_PPSN_MANAGER or role, otherwise false
     */
    private boolean isNationalManagerOrSupervisorRole() {

        return (getUser().hasRole(Role.PSS_PPSN_MANAGER) || getUser().hasRole(Role.PSS_PPSN_SUPERVISOR));
    }

    /**
     * @param locale
     *            locale
     */
    private void initialize(Locale locale) {
        pageTrail.clearTrail();
        pageTrail.addPage("unmatchedProductSearch", "Rematch", true);
        MATCH_MAP = new TreeMap<MatchType, String>();

        for (MatchType element : MatchType.values()) {

            MATCH_MAP.put(element, getMessageSource().getMessage(MATCH_TYPE_DOT + element.name(), null, locale));
        }

    }

    // Getters and Setters

    /**
     * Set reportService
     * 
     * @param nreportsService
     *            Service to use
     */
    public void setReportsService(RematchSuggestionService nrematchSuggestionService) {
        rematchSuggestionService = nrematchSuggestionService;
    }

    /**
     * Get reportService
     * 
     * @return the reportsService
     */
    public RematchSuggestionService getRematchSuggestionService() {
        return rematchSuggestionService;
    }

    /**
     * Set reportType
     * 
     * @param reportType
     *            the reportType to set
     */
    public void setMatchType(MatchType matchType) {
        matchVo.setMatchType(matchType);
    }

    /**
     * Get reportType
     * 
     * @return the reportType
     */
    public MatchType getMatchType() {
        return matchVo.getMatchType();
    }

    /**
     * getMatchTypeMap
     * 
     * @return the reportType
     */
    public SortedMap<MatchTypeVo, String> getMatchTypeMap() {
        return matchTypeMap;
    }

    /**
     * setMatchTypeMap
     * 
     * @param matchTypeMap
     *            report type to use
     */
    public void setMatchTypeMap(SortedMap<MatchTypeVo, String> matchTypeMap) {
        UnmatchedProductsController.matchTypeMap = matchTypeMap;
    }

    /**
     * Get reportVo
     * 
     * @return reportVo
     */
    public RematchVo getMatchTypeVo() {
        return matchVo;
    }

    /**
     * Set reportVo
     * 
     * @param reportVo
     *            reportVo
     */
    public void setMatchTypeVo(RematchVo matchVo) {
        this.matchVo = matchVo;
    }

}
