package gov.va.med.pharmacy.peps.service.common.utility;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;
import gov.va.med.pharmacy.peps.external.common.utility.MumpsConverter;
import gov.va.med.pharmacy.peps.service.common.utility.NDFUpdateFileMapping;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.junit.Test;


public class MessageFormatterTest {

    private static final String SAMPLE_MESSAGE_A = "this could be a valuable message";
    private static final String[] SAMPLE_MESSAGE_B = new String[] {
        "",
        "",
        "The following active VA Products are no longer marked for CMOP.",
        "All local drug file entries matched to these VA Products will",
        "be UNMARKED for CMOP. In order to have these entries dispensed",
        "by CMOP any local DRUG file (#50) entries matched to these",
        "products must be re-matched to another VA product that is actively",
        "marked for CMOP dispensing.",
        "",
        "    ASPIRIN 500MG BUFFERED TAB     [A0526]",
        "    ASPIRIN 325MG TAB,EC,UD     [A1752]",
        "",
        "The following products will be inactivated on the date listed. No alternative",
        "products have been found.",
        "",
        "    DRESSING,PROMOGRAN MATRIX 19.1IN X 19.1IN JJ#PG019     [XU657]",
        "    02/15/2014",
        "    DRESSING,PROMOGRAN PRISMA MATRIX 4.34SQ IN JJ#MA028     [XT106]",
        "    02/15/2014",
        "    EPOPROSTENOL NA 1.5MG/VIL INJ     [E0217]",
        "    02/10/2014",
        "    DRESSING,PROMOGRAN PRISMA MATRIX 19.1SQ IN JJ#MA123     [XT105]",
        "    02/15/2014",
        "    EPOPROSTENOL NA 0.5MG/VIL INJ     [E0216]",
        "    02/10/2014",
        "    DRESSING,PROMOGRAN MATRIX 4.34IN X 4.34IN JJ#PG004     [XU658]",
        "    02/15/2014",
        "",
        "The National Formulary Indicator has changed for the following",
        "VA Products. The National Formulary Indicator will automatically",
        "be changed in your local DRUG file (#50). Please review the",
        "VISN and Local Formulary designations of these products and",
        "make appropriate changes.",
        "",
        "  FORMULARY ITEMS",
        "    NONE",
        "",
        "  NON-FORMULARY ITEMS",
        "    NONE",
        "",
        "The Auto-Create Possible Dosages settings have been edited for the",
        "following VA Products. Please review your local dosages for products",
        "matched to these entries. Edits to your site's possible dosages or",
        "local possible dosages may be needed.",
        "",
        "    IBUPROFEN 100MG/ML INJ,VIL,8ML     [I0491]",
        "      Auto-Create Default Possible Dosage? YES",
        "    IBUPROFEN 800MG TAB,UD     [I0254]",
        "      Auto-Create Default Possible Dosage? YES",
        ""
    };
    private static final String[] EXPECTED_MESSAGE_B = new String[] {
        "MESSAGE^0^0",
        "^^50^50^3141213", // 13-DEC-2014
        "MESSAGE^1^0",
        "",
        "MESSAGE^2^0",
        "",
        "MESSAGE^3^0",
        "The following active VA Products are no longer marked for CMOP.",
        "MESSAGE^4^0",
        "All local drug file entries matched to these VA Products will",
        "MESSAGE^5^0",
        "be UNMARKED for CMOP. In order to have these entries dispensed",
        "MESSAGE^6^0",
        "by CMOP any local DRUG file (#50) entries matched to these",
        "MESSAGE^7^0",
        "products must be re-matched to another VA product that is actively",
        "MESSAGE^8^0",
        "marked for CMOP dispensing.",
        "MESSAGE^9^0",
        "",
        "MESSAGE^10^0",
        "    ASPIRIN 500MG BUFFERED TAB     [A0526]",
        "MESSAGE^11^0",
        "    ASPIRIN 325MG TAB,EC,UD     [A1752]",
        "MESSAGE^12^0",
        "",
        "MESSAGE^13^0",
        "The following products will be inactivated on the date listed. No alternative",
        "MESSAGE^14^0",
        "products have been found.",
        "MESSAGE^15^0",
        "",
        "MESSAGE^16^0",
        "    DRESSING,PROMOGRAN MATRIX 19.1IN X 19.1IN JJ#PG019     [XU657]",
        "MESSAGE^17^0",
        "    02/15/2014",
        "MESSAGE^18^0",
        "    DRESSING,PROMOGRAN PRISMA MATRIX 4.34SQ IN JJ#MA028     [XT106]",
        "MESSAGE^19^0",
        "    02/15/2014",
        "MESSAGE^20^0",
        "    EPOPROSTENOL NA 1.5MG/VIL INJ     [E0217]",
        "MESSAGE^21^0",
        "    02/10/2014",
        "MESSAGE^22^0",
        "    DRESSING,PROMOGRAN PRISMA MATRIX 19.1SQ IN JJ#MA123     [XT105]",
        "MESSAGE^23^0",
        "    02/15/2014",
        "MESSAGE^24^0",
        "    EPOPROSTENOL NA 0.5MG/VIL INJ     [E0216]",
        "MESSAGE^25^0",
        "    02/10/2014",
        "MESSAGE^26^0",
        "    DRESSING,PROMOGRAN MATRIX 4.34IN X 4.34IN JJ#PG004     [XU658]",
        "MESSAGE^27^0",
        "    02/15/2014",
        "MESSAGE^28^0",
        "",
        "MESSAGE^29^0",
        "The National Formulary Indicator has changed for the following",
        "MESSAGE^30^0",
        "VA Products. The National Formulary Indicator will automatically",
        "MESSAGE^31^0",
        "be changed in your local DRUG file (#50). Please review the",
        "MESSAGE^32^0",
        "VISN and Local Formulary designations of these products and",
        "MESSAGE^33^0",
        "make appropriate changes.",
        "MESSAGE^34^0",
        "",
        "MESSAGE^35^0",
        "  FORMULARY ITEMS",
        "MESSAGE^36^0",
        "    NONE",
        "MESSAGE^37^0",
        "",
        "MESSAGE^38^0",
        "  NON-FORMULARY ITEMS",
        "MESSAGE^39^0",
        "    NONE",
        "MESSAGE^40^0",
        "",
        "MESSAGE^41^0",
        "The Auto-Create Possible Dosages settings have been edited for the",
        "MESSAGE^42^0",
        "following VA Products. Please review your local dosages for products",
        "MESSAGE^43^0",
        "matched to these entries. Edits to your site's possible dosages or",
        "MESSAGE^44^0",
        "local possible dosages may be needed.",
        "MESSAGE^45^0",
        "",
        "MESSAGE^46^0",
        "    IBUPROFEN 100MG/ML INJ,VIL,8ML     [I0491]",
        "MESSAGE^47^0",
        "      Auto-Create Default Possible Dosage? YES",
        "MESSAGE^48^0",
        "    IBUPROFEN 800MG TAB,UD     [I0254]",
        "MESSAGE^49^0",
        "      Auto-Create Default Possible Dosage? YES",
        "MESSAGE^50^0",
        ""
    };
    private MessageFormatter testSubject;
    private DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

    private Date getSampleDate() throws ParseException {
        Date date = dateFormat.parse("2014-12-13");
        return date;
    }

    private String[] expectedMessageAsMessage2Format(String[] input) {
        return Arrays.stream(input)
                .map(s -> s.replaceFirst("MESSAGE", "MESSAGE2"))
                .toArray(size -> new String[size]);
    }

    @Test
    public void should_return_overall_message_header_when_no_message_present() throws Exception {
        testSubject = new MessageFormatter(getSampleDate());
        List<String> result = testSubject.apply(NDFUpdateFileMapping.MESSAGE);
        assertThat(result.size(), is(2));
        assertThat(result.get(0), is(NDFUpdateFileMapping.MESSAGE + "0" + NDFUpdateFileMapping.ENDER));
        String mumpsFormattedDate = MumpsConverter.convertDate(getSampleDate()); // In this case.. "3141213"
        assertThat(result.get(1), is("^^0^0^" + mumpsFormattedDate));
    }

    @Test
    public void should_include_message_line_header_followed_by_message_line() throws Exception {
        testSubject = new MessageFormatter(getSampleDate(), SAMPLE_MESSAGE_A);
        List<String> result = testSubject.apply(NDFUpdateFileMapping.MESSAGE);
        assertThat(result.size(), is(4));
        assertThat(result.get(2), is(NDFUpdateFileMapping.MESSAGE + "1" + NDFUpdateFileMapping.ENDER));
        assertThat(result.get(3), is(SAMPLE_MESSAGE_A));
    }

    @Test
    public void should_format_large_example_found_in_wild() throws Exception {
        testSubject = new MessageFormatter(getSampleDate(), SAMPLE_MESSAGE_B);
        List<String> result = testSubject.apply(NDFUpdateFileMapping.MESSAGE);
        assertThat(result, is(Arrays.asList(EXPECTED_MESSAGE_B)));
    }

    @Test
    public void should_format_large_message2_example() throws Exception {
        testSubject = new MessageFormatter(getSampleDate(), SAMPLE_MESSAGE_B);
        List<String> result = testSubject.apply(NDFUpdateFileMapping.MESSAGE2);
        assertThat(result, is(Arrays.asList(expectedMessageAsMessage2Format(EXPECTED_MESSAGE_B))));
    }
}
