package gov.va.med.pharmacy.peps.service.common.capability.impl;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.nio.channels.SeekableByteChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import gov.va.med.pharmacy.peps.common.utility.VistaFileNumber;
import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.dao.EplNdfOutgoingDifferencesDao;
import gov.va.med.pharmacy.peps.domain.common.dao.EplRematchSuggestionDao;
import gov.va.med.pharmacy.peps.external.common.utility.MumpsConverter;
import gov.va.med.pharmacy.peps.service.common.utility.MessageFormatter;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class MessageDataCapabilityImplTest {

    public static final Long LONG_20150911 = 1441975560000L;
    public static final Date DATE_20150911 = new Date(LONG_20150911);
    private static final String VISTA_FIELD_NUMBER_DOT_01 = ".01";
    private static final String BAR_DELIMITED_MSG_PT1 = "this is";
    private static final String BAR_DELIMITED_MSG_PT2 = "a test";
    private static final String BAR_DELIMITED_MSG_PT3 = "of the";
    private static final String BAR_DELIMITED_MSG_PT4 = "whatever";
    private static final String BAR_DELIMITED_MESSAGE = BAR_DELIMITED_MSG_PT1
            + "|" + BAR_DELIMITED_MSG_PT2 + "|" + BAR_DELIMITED_MSG_PT3 + "|"
            + BAR_DELIMITED_MSG_PT4;
    private static final String VISTA_FIELD_NUMBER_2 = "2";
    private static final String PRIOR_MESSAGE_VALUE = "arbitrary HL7 message, probably";
    private static final String NEWLINE = System.getProperty("line.separator");
    private static final List<Object[]> EMPTY_LIST = new ArrayList<Object[]>();
    private static final List<Object[]> ONE_STRENGTH_CHANGE_ITEM = getListOfOneStrengthChangeItem();
    private static final List<Object[]> MULTIPLE_STRENGTH_CHANGE_ITEMS = getListOfMultipleStrengthChangeItems();
    private static final Date START_DATE = DATE_20150911;

    private static final String[] EXPECTED_STRENGTH_CHANGE_MESSAGE_PARTS = new String[] {
            PRIOR_MESSAGE_VALUE,
            "MESSAGE^1^0",
            BAR_DELIMITED_MSG_PT1,
            "MESSAGE^2^0",
            BAR_DELIMITED_MSG_PT2,
            "MESSAGE^3^0",
            BAR_DELIMITED_MSG_PT3,
            "MESSAGE^4^0",
            BAR_DELIMITED_MSG_PT4,
            "MESSAGE^5^0",
            "",
            "MESSAGE^6^0",
            "    1 va product name     [1 cmop id]",
            "MESSAGE^7^0",
            "      Old Value: 1 epl ndf outgoing old",
            "MESSAGE^8^0",
            "      New Value: 1 epl ndf outgoing new",
            "MESSAGE^9^0",
            ""
    };

    private static final String[] EXPECTED_STRENGTH_CHANGE_MESSAGE_PARTS_2 = new String[] {
        PRIOR_MESSAGE_VALUE,
        "MESSAGE^1^0",
        BAR_DELIMITED_MSG_PT1,
        "MESSAGE^2^0",
        BAR_DELIMITED_MSG_PT2,
        "MESSAGE^3^0",
        BAR_DELIMITED_MSG_PT3,
        "MESSAGE^4^0",
        BAR_DELIMITED_MSG_PT4,
        "MESSAGE^5^0",
        "",
        "MESSAGE^6^0",
        "    1 va product name     [1 cmop id]",
        "MESSAGE^7^0",
        "      Old Value: 1 epl ndf outgoing old",
        "MESSAGE^8^0",
        "      New Value: 1 epl ndf outgoing new",
        "MESSAGE^9^0",
        "    2 va product name     [2 cmop id]",
        "MESSAGE^10^0",
        "      Old Value: 2 epl ndf outgoing old",
        "MESSAGE^11^0",
        "      New Value: 2 epl ndf outgoing new",
        "MESSAGE^12^0",
        "    3 va product name     [3 cmop id]",
        "MESSAGE^13^0",
        "      Old Value: 3 epl ndf outgoing old",
        "MESSAGE^14^0",
        "      New Value: 3 epl ndf outgoing new",
        "MESSAGE^15^0",
        ""
    };

    private static final String[] EXPECTED_ADD_PRODUCT_MESSAGE_PARTS = new String[] {
        "",
        "MESSAGE^1^0",
        BAR_DELIMITED_MSG_PT1,
        "MESSAGE^2^0",
        BAR_DELIMITED_MSG_PT2,
        "MESSAGE^3^0",
        BAR_DELIMITED_MSG_PT3,
        "MESSAGE^4^0",
        BAR_DELIMITED_MSG_PT4,
        "MESSAGE^5^0",
        "",
        "MESSAGE^6^0",
        "    1 product name     [1 cmop id]",
        "MESSAGE^7^0",
        "    (DISPENSE UNIT - 1 dispense unit name)",
        "MESSAGE^8^0",
        "    1 ndc number",
        "MESSAGE^9^0",
        ""
    };

    private static final String[] EXPECTED_ADD_PRODUCT_MESSAGE_PARTS_2 = new String[] {
        "",
        "MESSAGE^1^0",
        BAR_DELIMITED_MSG_PT1,
        "MESSAGE^2^0",
        BAR_DELIMITED_MSG_PT2,
        "MESSAGE^3^0",
        BAR_DELIMITED_MSG_PT3,
        "MESSAGE^4^0",
        BAR_DELIMITED_MSG_PT4,
        "MESSAGE^5^0",
        "",
        "MESSAGE^6^0",
        "    1 product name     [1 cmop id]",
        "MESSAGE^7^0",
        "    (DISPENSE UNIT - 1 dispense unit name)",
        "MESSAGE^8^0",
        "    1 ndc number",
        "MESSAGE^9^0",
        "    2 product name     [2 cmop id]",
        "MESSAGE^10^0",
        "    (DISPENSE UNIT - 2 dispense unit name)",
        "MESSAGE^11^0",
        "    2 ndc number",
        "MESSAGE^12^0",
        "    3 product name     [3 cmop id]",
        "MESSAGE^13^0",
        "    (DISPENSE UNIT - 3 dispense unit name)",
        "MESSAGE^14^0",
        "    3 ndc number",
        "MESSAGE^15^0",
        ""
    };

    private static final String[] MINIMAL_MESSAGE_REPORT = new String[] {
        "MESSAGE^0^0",
//        "^^13^13^" + MumpsConverter.convertDate(new Date()), // Test subject does exactly this, unfortunately!
        "^^12^12^" + MumpsConverter.convertDate(new Date()), // Test subject does exactly this, unfortunately!
        "MESSAGE^1^0",
        "",
//        "MESSAGE^2^0",
//        "",
//        "MESSAGE^3^0",
//        "this is",
//        "MESSAGE^4^0",
//        "a test",
//        "MESSAGE^5^0",
//        "of the",
//        "MESSAGE^6^0",
//        "whatever",
//        "MESSAGE^7^0",
//        "",
//        "MESSAGE^8^0",
//        "  FORMULARY ITEMS",
//        "MESSAGE^9^0",
//        "    NONE",
//        "MESSAGE^10^0",
//        "",
//        "MESSAGE^11^0",
//        "  NON-FORMULARY ITEMS",
//        "MESSAGE^12^0",
//        "    NONE",
//        "MESSAGE^13^0",
//        ""
        "MESSAGE^2^0",
        "this is",
        "MESSAGE^3^0",
        "a test",
        "MESSAGE^4^0",
        "of the",
        "MESSAGE^5^0",
        "whatever",
        "MESSAGE^6^0",
        "",
        "MESSAGE^7^0",
        "  FORMULARY ITEMS",
        "MESSAGE^8^0",
        "    NONE",
        "MESSAGE^9^0",
        "",
        "MESSAGE^10^0",
        "  NON-FORMULARY ITEMS",
        "MESSAGE^11^0",
        "    NONE",
        "MESSAGE^12^0",
        ""
    };

    private static final String[] SAMPLE_MESSAGE_REPORT = new String[] {
            "MESSAGE^0^0",
            // Test subject does exactly this, unfortunately!
            "^^55^55^" + MumpsConverter.convertDate(new Date()),
            "MESSAGE^1^0", 
            "", 
            "MESSAGE^2^0", 
            "this is", 
            "MESSAGE^3^0",
            "a test", 
            "MESSAGE^4^0", 
            "of the", 
            "MESSAGE^5^0", 
            "whatever",
            "MESSAGE^6^0", 
            "", 
            "MESSAGE^7^0",
            "    productName_0     [cmopId_0]", 
            "MESSAGE^8^0",
            "    (DISPENSE UNIT - dispenseUnitName_0)", 
            "MESSAGE^9^0",
            "    ndcNumber_0", 
            "MESSAGE^10^0",
            "    productName_1     [cmopId_1]", 
            "MESSAGE^11^0",
            "    (DISPENSE UNIT - dispenseUnitName_1)", 
            "MESSAGE^12^0",
            "    ndcNumber_1", 
            "MESSAGE^13^0", 
            "", 
            "MESSAGE^14^0", 
            "this is",
            "MESSAGE^15^0", 
            "a test", 
            "MESSAGE^16^0", 
            "of the", 
            "MESSAGE^17^0",
            "whatever", 
            "MESSAGE^18^0", 
            "", 
            "MESSAGE^19^0",
            "    productName_0     [cmopId_0]", 
            "MESSAGE^20^0", 
            "",
            "MESSAGE^21^0", 
            "this is", 
            "MESSAGE^22^0", 
            "a test",
            "MESSAGE^23^0", 
            "of the", 
            "MESSAGE^24^0", 
            "whatever",
            "MESSAGE^25^0", 
            "", 
            "MESSAGE^26^0",
            "    productName_0     [cmopId_0]", 
            "MESSAGE^27^0",
            "    09/11/2015", 
            "MESSAGE^28^0", 
            "      USE", 
            "MESSAGE^29^0",
            "    newProductName_0     [newCmopId_0]", 
            "MESSAGE^30^0",
            "    productName_1     [cmopId_1]", 
            "MESSAGE^31^0",
            "    09/11/2015", 
            "MESSAGE^32^0", 
            "      USE", 
            "MESSAGE^33^0",
            "    newProductName_1     [newCmopId_1]", 
            "MESSAGE^34^0", 
            "",
            "MESSAGE^35^0", 
            "this is", 
            "MESSAGE^36^0", 
            "a test",
            "MESSAGE^37^0", 
            "of the", 
            "MESSAGE^38^0", 
            "whatever",
            "MESSAGE^39^0", 
            "", 
            "MESSAGE^40^0",
            "    productName_0     [cmopId_0]", 
            "MESSAGE^41^0",
            "    09/11/2015", 
            "MESSAGE^42^0",
            "    productName_1     [cmopId_1]", 
            "MESSAGE^43^0",
            "    09/11/2015", 
            "MESSAGE^44^0", 
            "", 
            "MESSAGE^45^0", 
            "this is",
            "MESSAGE^46^0", 
            "a test", 
            "MESSAGE^47^0", 
            "of the", 
            "MESSAGE^48^0",
            "whatever", 
            "MESSAGE^49^0", 
            "", 
            "MESSAGE^50^0",
            "  FORMULARY ITEMS", 
            "MESSAGE^51^0", 
            "    NONE", 
            "MESSAGE^52^0",
            "", 
            "MESSAGE^53^0", 
            "  NON-FORMULARY ITEMS", 
            "MESSAGE^54^0",
            "    NONE", 
            "MESSAGE^55^0", 
            ""
    };

    static abstract class SampleData {
        protected abstract int getColumnCount();
        protected abstract String getColumnTitle(int col);
        private String getValue(String columnName, int rowNumber) {
            return columnName + "_" + rowNumber;
        }
        List<Object[]> getSample(int rows) {
            List<Object[]> result = new ArrayList<Object[]>();
            for (int row = 0; row < rows; row++) {
                result.add(getSampleFor(row));
            }
            return result;
        }
        String[] getSampleFor(int row) {
            String[] result = new String[getColumnCount()];
            for (int col = 0; col < getColumnCount(); col++) {
                String proposedValue = getValue(getColumnTitle(col), row);
                // can't pass our usual dummy data for a Date, so if the column is a date, just use this.
                if (proposedValue.contains("Date")) proposedValue = LONG_20150911 + "";
                result[col] = proposedValue;
            }
            return result;
        }
    }

    static class AddedProductsReportData extends SampleData {
        String[] colTitles = new String[] { "newValue", "productName",
                "cmopId", "dispenseUnitName", "ndcNumber" };

        @Override
        protected String getColumnTitle(int col) {
            return colTitles[col];
        }

        @Override
        protected int getColumnCount() {
            return 5;
        }
    }

    static class InactivatedCmopProductsData extends SampleData {
        String[] colTitles = new String[] { "newValue", "productName",
                "cmopId" };

        @Override
        protected String getColumnTitle(int col) {
            return colTitles[col];
        }

        @Override
        protected int getColumnCount() {
            return 3;
        }
    }

    static class FutureInactiveRematchReportData extends SampleData {
        String[] colTitles = new String[] { "eplId", "vistaIen", "productName",
                "cmopId", "futureDate", "newIen", "newProductName", "newCmopId" };

        @Override
        protected String getColumnTitle(int col) {
            return colTitles[col];
        }

        @Override
        protected int getColumnCount() {
            return 8;
        }
    }

    static class FutureInactiveChangeData extends SampleData {
        String[] colTitles = new String[] { "eplId", "productName", "cmopId",
                "futureDate" };
        @Override
        protected String getColumnTitle(int col) {
            return colTitles[col];
        }

        @Override
        protected int getColumnCount() {
            return 4;
        }
    }

    private MessageDataCapabilityImpl testSubject;

    @Mock
    private EplNdfOutgoingDifferencesDao mockEplNdfOutgoingDifferencesDao;

    @Mock
    private NationalSettingDomainCapability mockNationalSettingDomainCapability;

    @Mock
    private NationalSettingVo mockNationalSettingVo;

    @Mock
    private NdfUpdateProcessFile mockNdfUpdateFile;

    @Mock
    private EplRematchSuggestionDao mockEplRematchSuggestionDao;

    @Mock
    private SeekableByteChannel mockSeekableByteChannel;

    private static List<Object[]> getListOfOneStrengthChangeItem() {
        List<Object[]> result = new ArrayList<Object[]>();
        result.add(getStrengthChangeItem(1));
        return result;
    }

    private static List<Object[]> getListOfMultipleStrengthChangeItems() {
        List<Object[]> result = new ArrayList<Object[]>();
        result.add(getStrengthChangeItem(1));
        result.add(getStrengthChangeItem(2));
        result.add(getStrengthChangeItem(3));
        return result;
    }

    private static Object[] getStrengthChangeItem(int num) {
        Object[] result = new Object[4];
        result[0] = num + " epl ndf outgoing new";
        result[1] = num + " epl ndf outgoing old";
        result[2] = num + " va product name";
        result[3] = num + " cmop id";
        return result;
    }

    private static List<Object[]> getListOfOneAddedProduct() {
        List<Object[]> result = new ArrayList<Object[]>();
        result.add(getAddedProduct(1));
        return result;
    }

    private static List<Object[]> getListOfMultipleAddedProducts() {
        List<Object[]> result = new ArrayList<Object[]>();
        result.add(getAddedProduct(1));
        result.add(getAddedProduct(2));
        result.add(getAddedProduct(3));
        return result;
    }

    private static Object[] getAddedProduct(int num) {
        Object[] result = new Object[5];
        result[0] = num + " new value";
        result[1] = num + " product name";
        result[2] = num + " cmop id";
        result[3] = num + " dispense unit name";
        result[4] = num + " ndc number";
        return result;
    }

    @Before
    public void setup() {
        testSubject = new MessageDataCapabilityImpl();
        testSubject.setEplNdfOutgoingDifferencesDao(mockEplNdfOutgoingDifferencesDao);
        testSubject.setNationalSettingDomainCapability(mockNationalSettingDomainCapability);
        testSubject.setEplRematchSuggestionDao(mockEplRematchSuggestionDao);
    }

    @Test
    public void should_return_unchanged_counter_and_message_when_no_strength_change_products() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer();
        when(mockEplNdfOutgoingDifferencesDao.messageGenericChange(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_2, START_DATE)).thenReturn(EMPTY_LIST);

        // act
       /* int counter = testSubject.strengthChangeReport(0, message, START_DATE);

        // assert
        assertThat(counter, is(0));
        assertThat(message.length(), is(0));*/
    }

    @Test
    public void should_add_boring_message_when_one_strength_change_item_has_null_national_setting() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer(PRIOR_MESSAGE_VALUE);
        when(mockEplNdfOutgoingDifferencesDao.messageGenericChange(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_2, START_DATE)).thenReturn(ONE_STRENGTH_CHANGE_ITEM);
        when(mockNationalSettingVo.getStringValue()).thenReturn(null);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_STRENGTH_TEXT.toString())).thenReturn(mockNationalSettingVo);

        // act
       /* int counter = testSubject.strengthChangeReport(0, message, START_DATE);

        // assert (usually prefer to make just one assertion, but the class under test does way too much here)
        assertThat(counter, is(6));
        assertThat(message.toString(), containsString(PRIOR_MESSAGE_VALUE));
        assertThat(message.toString(), containsString("missing text"));*/
    }

    @Test
    public void should_add_msg_when_one_strength_change_item_has_national_setting_value() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer(PRIOR_MESSAGE_VALUE);
        when(mockEplNdfOutgoingDifferencesDao.messageGenericChange(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_2, START_DATE)).thenReturn(ONE_STRENGTH_CHANGE_ITEM);
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_STRENGTH_TEXT.toString())).thenReturn(mockNationalSettingVo);

       /* // act
        int counter = testSubject.strengthChangeReport(0, message, START_DATE);

        // assert (usually prefer to make just one simple assertion, but the class under test does way too much here)
        assertThat(counter, is(9));
        String expectedReport = Arrays.stream(EXPECTED_STRENGTH_CHANGE_MESSAGE_PARTS).collect(Collectors.joining(NEWLINE));
        assertThat(message.toString(), is(expectedReport));
*/
    }

    @Test
    public void should_add_msg_when_multiple_strength_change_items_have_national_setting_value() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer(PRIOR_MESSAGE_VALUE);
        when(mockEplNdfOutgoingDifferencesDao.messageGenericChange(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_2, START_DATE)).thenReturn(MULTIPLE_STRENGTH_CHANGE_ITEMS);
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_STRENGTH_TEXT.toString())).thenReturn(mockNationalSettingVo);

        // act
       /* int counter = testSubject.strengthChangeReport(0, message, START_DATE);

        // assert (usually prefer to make just one simple assertion, but the class under test does way too much here)
        assertThat(counter, is(15));
        String expectedReport = Arrays.stream(EXPECTED_STRENGTH_CHANGE_MESSAGE_PARTS_2).collect(Collectors.joining(NEWLINE));
        assertThat(message.toString(), is(expectedReport));*/
    }

    /*
     * temporal coupling in the method tested here--it must be called first as it takes the responsibility to start the message.
     */
    @Test
    public void should_start_message_when_no_added_products_are_present() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer();
        when(mockEplNdfOutgoingDifferencesDao.findAddedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_DOT_01, START_DATE)).thenReturn(null);

        // act
      /*  int counter = testSubject.addedProductReport(START_DATE);

        // assert (usually prefer to make just one simple assertion, but the class under test does way too much here)
        assertThat(counter, is(1));
        assertThat(message.toString(), is(NEWLINE + "MESSAGE^" + counter + "^0" + NEWLINE));*/
    }

    @Test
    public void should_start_message_and_add_products_when_one_added_product_is_present() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer();
        when(mockEplNdfOutgoingDifferencesDao.findAddedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_DOT_01, START_DATE)).thenReturn(getListOfOneAddedProduct());
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_ADDEDPRODUCT_TEXT.toString())).thenReturn(mockNationalSettingVo);

        // act
        /*int counter = testSubject.addedProductReport(START_DATE);

        // assert (usually prefer to make just one simple assertion, but the class under test does way too much here)
        assertThat(counter, is(9));
        String expectedReport = Arrays.stream(EXPECTED_ADD_PRODUCT_MESSAGE_PARTS).collect(Collectors.joining(NEWLINE));
        assertThat(message.toString(), is(expectedReport));*/
    }

    @Test
    public void should_start_message_and_add_products_when_multiple_added_products_are_present() throws Exception {
        // arrange
        StringBuffer message = new StringBuffer();
        when(mockEplNdfOutgoingDifferencesDao.findAddedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), VISTA_FIELD_NUMBER_DOT_01, START_DATE)).thenReturn(getListOfMultipleAddedProducts());
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_ADDEDPRODUCT_TEXT.toString())).thenReturn(mockNationalSettingVo);

        // act
       /* int counter = testSubject.addedProductReport( START_DATE);

        // assert (usually prefer to make just one simple assertion, but the class under test does way too much here)
        assertThat(counter, is(15));
        String expectedReport = Arrays.stream(EXPECTED_ADD_PRODUCT_MESSAGE_PARTS_2).collect(Collectors.joining(NEWLINE));
        assertThat(message.toString(), is(expectedReport));*/
    }

    @Test
    @Ignore
    /**
     * This was written to capture the exact behavior of the 
     * messageReportData() method prior to refactoring. Intent is to
     * just use the refactored implementation now, so this is left as
     * a point of reference.
     * 
     * @throws Exception
     */
    public void should_generate_minimal_message_report() throws Exception {
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_NATIONALFORMULARY_TEXT.toString())).thenReturn(mockNationalSettingVo);

        testSubject.messageReportData(START_DATE, mockNdfUpdateFile);

        String expectedReport = Arrays.stream(MINIMAL_MESSAGE_REPORT)
                .collect(Collectors.joining(NEWLINE));
        verify(mockNdfUpdateFile).putNextRow(expectedReport);
    }

    @Test
    public void should_generate_new_minimal_message_report() throws Exception {
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_NATIONALFORMULARY_TEXT.toString())).thenReturn(mockNationalSettingVo);

       /* testSubject.messageReportData_2(START_DATE, mockNdfUpdateFile);

        String expectedReport = Arrays.stream(MINIMAL_MESSAGE_REPORT)
                .collect(Collectors.joining(NEWLINE));
        verify(mockNdfUpdateFile).putNextRow(expectedReport);*/
    }

    @Test
    public void should_generate_minimal_message2_report() throws Exception {
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE2_DDI_TEXT.toString())).thenReturn(mockNationalSettingVo);

        testSubject.message2ReportData(START_DATE, mockNdfUpdateFile);
    }

    @Test
    @Ignore
    /**
     * This was written to capture the exact behavior of the 
     * messageReportData() method prior to refactoring. Intent is to
     * just use the refactored implementation now, so this is left as
     * a point of reference.
     * 
     * @throws Exception
     */
    public void should_generate_sample_message_report() throws Exception {
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_NATIONALFORMULARY_TEXT.toString())).thenReturn(mockNationalSettingVo);
        AddedProductsReportData addedProductsReportData = new AddedProductsReportData();
        when(mockEplNdfOutgoingDifferencesDao.findAddedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), ".01", START_DATE)).thenReturn(addedProductsReportData.getSample(2));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_ADDEDPRODUCT_TEXT.toString())).thenReturn(mockNationalSettingVo);
        InactivatedCmopProductsData inactivatedCmopProductsData = new InactivatedCmopProductsData();
        when(mockEplNdfOutgoingDifferencesDao.findInactivatedCMOPProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "7", START_DATE)).thenReturn(inactivatedCmopProductsData.getSample(1));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_INACTIVATEDCMOP_TEXT.toString())).thenReturn(mockNationalSettingVo);
        FutureInactiveRematchReportData futureInactiveRematchedReportData = new FutureInactiveRematchReportData();
        when(mockEplNdfOutgoingDifferencesDao.futureInactiveRematched()).thenReturn(futureInactiveRematchedReportData.getSample(2));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_PROPOSEDINACTIVATEDWSUGG_TEXT.toString())).thenReturn(mockNationalSettingVo);
        FutureInactiveChangeData futureInactiveChangeData = new FutureInactiveChangeData();
        when(mockEplNdfOutgoingDifferencesDao.futureInactiveChange()).thenReturn(futureInactiveChangeData.getSample(2));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_FUTUREINACTIVE_TEXT.toString())).thenReturn(mockNationalSettingVo);

        testSubject.messageReportData(START_DATE, mockNdfUpdateFile);

        String expectedReport = Arrays.stream(SAMPLE_MESSAGE_REPORT)
                .collect(Collectors.joining(NEWLINE));
        verify(mockNdfUpdateFile).putNextRow(expectedReport);
    }

    @Test
    public void should_generate_new_sample_message_report() throws Exception {
        when(mockNationalSettingVo.getStringValue()).thenReturn(BAR_DELIMITED_MESSAGE);
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_NATIONALFORMULARY_TEXT.toString())).thenReturn(mockNationalSettingVo);
        AddedProductsReportData addedProductsReportData = new AddedProductsReportData();
        when(mockEplNdfOutgoingDifferencesDao.findAddedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), ".01", START_DATE)).thenReturn(addedProductsReportData.getSample(2));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_ADDEDPRODUCT_TEXT.toString())).thenReturn(mockNationalSettingVo);
        InactivatedCmopProductsData inactivatedCmopProductsData = new InactivatedCmopProductsData();
        when(mockEplNdfOutgoingDifferencesDao.findInactivatedCMOPProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "7", START_DATE)).thenReturn(inactivatedCmopProductsData.getSample(1));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_INACTIVATEDCMOP_TEXT.toString())).thenReturn(mockNationalSettingVo);
        FutureInactiveRematchReportData futureInactiveRematchedReportData = new FutureInactiveRematchReportData();
        when(mockEplNdfOutgoingDifferencesDao.futureInactiveRematched()).thenReturn(futureInactiveRematchedReportData.getSample(2));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_PROPOSEDINACTIVATEDWSUGG_TEXT.toString())).thenReturn(mockNationalSettingVo);
        FutureInactiveChangeData futureInactiveChangeData = new FutureInactiveChangeData();
        when(mockEplNdfOutgoingDifferencesDao.futureInactiveChange()).thenReturn(futureInactiveChangeData.getSample(2));
        when(mockNationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_FUTUREINACTIVE_TEXT.toString())).thenReturn(mockNationalSettingVo);

       /* testSubject.messageReportData_2(START_DATE, mockNdfUpdateFile);

        String expectedReport = Arrays.stream(SAMPLE_MESSAGE_REPORT)
                .collect(Collectors.joining(NEWLINE));
        verify(mockNdfUpdateFile).putNextRow(expectedReport);*/
    }

}
