package gov.va.med.pharmacy.peps.external.common.utility.test;

import gov.va.med.pharmacy.peps.service.common.utility.FileInfo;
import gov.va.med.pharmacy.peps.service.common.utility.TransferFile;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;

import org.apache.commons.io.FileUtils;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.impl.StandardFileSystemManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Test;

import com.jcraft.jsch.JSchException;

/**
 * The Class TransferFileIT.
 */
public class TransferFileIT extends TestCase {

    /** The Constant LOG. */
    private static final Logger LOG = LogManager.getLogger(TransferFileIT.class);

    /**
     * Test local file to remote file transfer.
     *
     * @throws IllegalArgumentException
     *             the illegal argument exception
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     * @throws JSchException
     *             the j sch exception
     */
    @Test
    public final void testLocalToRemote() throws IllegalArgumentException, IOException, JSchException {
        File tempFile = File.createTempFile("LocalToRemote", ".txt");
        FileUtils.writeStringToFile(tempFile, "Some text here");

        FileInfo fileInfo = createFileInfo(tempFile.getName(), "pending");

        String inputFileDirectory = tempFile.getAbsolutePath();

        boolean fileTransmitted = TransferFile.moveTo(inputFileDirectory, fileInfo);
        LOG.info(fileTransmitted);
        // cleanup test files
        tempFile.delete();
        deleteRemoteTestFile(fileInfo);
        assertTrue(fileTransmitted);
    }

    /**
     * Test remote file to remote file transfer.
     *
     * @throws IllegalArgumentException
     *             the illegal argument exception
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     * @throws JSchException
     *             the j sch exception
     */
    @Test
    public final void testRemoteToRemote() throws IllegalArgumentException, IOException, JSchException {
        String fileName = "RemoteToRemote.txt";
        FileInfo fileInfoPending = createFileInfo(fileName, "pending");
        // create the initial file to be used during the move
        FileObject testFile = createRemoteTestFile(fileInfoPending);
        LOG.info(testFile.getName());
        FileInfo fileInfoApproved = createFileInfo(fileName, "approved");

        boolean fileTransmitted = TransferFile.moveTo(fileInfoPending, fileInfoApproved);

        // cleanup test file
        deleteRemoteTestFile(fileInfoApproved);
        assertTrue(fileTransmitted);
    }

    /**
     * Creates a FileInfo object when given fileName and folder of the file.
     *
     * @param fileName
     *            the file name
     * @param folder
     *            the folder
     * @return the file info
     */
    private FileInfo createFileInfo(String fileName, String folder) {
        FileInfo fileInfo = new FileInfo();

        fileInfo.setHostName("vaauspresftp01.aac.domain");
        fileInfo.setPort("22");
        fileInfo.setUserName("presftp");
        fileInfo.setAuthorization("Im@g!N3");
        fileInfo.setDirectoryPath("/home/presftp/pps-n/dev/localhost/" + folder + "/");
        fileInfo.setFileName(fileName);

        return fileInfo;
    }

    /**
     * Creates the remote test file.
     *
     * @param fileInfo
     *            the file info
     * @return the file object
     */
    public final FileObject createRemoteTestFile(FileInfo fileInfo) {

        StandardFileSystemManager remoteFsm = new StandardFileSystemManager();
        FileObject remoteFile = null;
        try {

            remoteFsm.init();
            if (fileInfo != null) {
                remoteFile = remoteFsm.resolveFile(fileInfo.getEncodedFilePath(), TransferFile.createFileSystemOptions());
                remoteFile.createFile();
            }

        } catch (FileSystemException e) {
            LOG.error("Exception creating test file", e);

        } finally {
            if (remoteFsm != null) {
                remoteFsm.close();
            }
        }
        return remoteFile;
    }

    /**
     * Delete remote test file for cleanup of tests.
     *
     * @param fileInfo
     *            the file info
     * @return the file object
     */
    public final FileObject deleteRemoteTestFile(FileInfo fileInfo) {

        StandardFileSystemManager remoteFsm = new StandardFileSystemManager();
        FileObject remoteFile = null;
        try {

            remoteFsm.init();
            if (fileInfo != null) {
                remoteFile =
                        remoteFsm.resolveFile(fileInfo.getEncodedURIDirectory() + File.separator + fileInfo.getFileName(),
                                TransferFile.createFileSystemOptions());
                remoteFile.delete();
            }

        } catch (FileSystemException e) {
            LOG.error("Exception creating test file ");

        } finally {
            if (remoteFsm != null) {
                remoteFsm.close();
            }
        }
        return remoteFile;
    }

}
