package gov.va.med.pharmacy.peps.service.common.utility;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemException;
import org.apache.commons.vfs2.FileSystemOptions;
import org.apache.commons.vfs2.impl.DefaultFileSystemManager;
import org.apache.commons.vfs2.impl.StandardFileSystemManager;
import org.apache.commons.vfs2.provider.sftp.SftpFileSystemConfigBuilder;

/**
 * The Class TransferFile.
 */
public final class TransferFile {

    /**
     * Instantiates a new transfer file.
     */
    protected TransferFile() {
        // prevents calls from subclass
        throw new UnsupportedOperationException();
    }

    /** The Constant LOG. */
    private static final Log LOG = LogFactory.getLog(TransferFile.class);

    /**
     * Move to.
     *
     * @param source
     *            the source
     * @param destination
     *            the destination
     * @return true, if successful
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    protected static boolean moveTo(FileObject source, FileObject destination) throws IOException {
        LOG.debug("Method: moveTo Message: Being Process");
        boolean success = false;

        if (source != null && source.exists()) {
            source.moveTo(destination);
            success = true;
        } else {
            LOG.debug("Source file doesn't exist");
            success = false;
        }

        LOG.debug("Method: moveTo Message: Process Complete Returning: " + success);
        return success;
    }

    /**
     * Creates the file system options.
     *
     * @return the file system options
     * @throws FileSystemException
     *             the file system exception
     */
    public static final FileSystemOptions createFileSystemOptions() throws FileSystemException {
        LOG.debug("Method: createFileSystemOptions Message: Attempting  to create FileSystemOptions.");
        FileSystemOptions fsOptions = new FileSystemOptions();
        try {
            SftpFileSystemConfigBuilder.getInstance().setStrictHostKeyChecking(fsOptions, "no");
            SftpFileSystemConfigBuilder.getInstance().setUserDirIsRoot(fsOptions, false);
        } catch (FileSystemException e) {
            LOG.debug("Failed to create FileSystemOptions");
        }
        LOG.debug("Method: createFileSystemOptions Message: Created FileSystemOptions.");
        return fsOptions;
    }

    /**
     * Move to method that supports moving a file from a remote file directory to a different remove directory using SFTP.
     *
     * @param sourceInfo
     *            the source info
     * @param destinationInfo
     *            the destination info
     * @return true, if successful
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    public static boolean moveTo(FileInfo sourceInfo, FileInfo destinationInfo) throws IOException {
        StandardFileSystemManager sourceFsm = new StandardFileSystemManager();
        StandardFileSystemManager destinationFsm = new StandardFileSystemManager();
        
        FileObject source = null;
        FileObject destination = null;

        Boolean successfulTransmission;
        try {

            sourceFsm.init();
            destinationFsm.init();

            source = sourceFsm.resolveFile(sourceInfo.getEncodedFilePath(), createFileSystemOptions());

           
            destination = destinationFsm.resolveFile(destinationInfo.getEncodedFilePath(), createFileSystemOptions());

            moveTo(source, destination);

            successfulTransmission = true;
        } catch (FileSystemException fse) {
            LOG.error("Exception caught in TransferFile.moveTo() " + fse.getMessage(), fse);
            throw fse;
        } finally {
        	safeClose(source);
        	safeClose(destination);
            safeClose(sourceFsm);
            safeClose(destinationFsm);
        }
        return successfulTransmission;

    }

    /**
     * Move to method that supports moving a file from a local file directory to a remote directory using SFTP.
     *
     * @param localFileDirectory
     *            the local file directory
     * @param fileInfo
     *            the file info
     * @return true, if successful
     * @throws IOException
     *             Signals that an I/O exception has occurred.
     */
    public static final boolean moveTo(String localFileDirectory, FileInfo fileInfo) throws IOException {
        StandardFileSystemManager localFsm = new StandardFileSystemManager();
        StandardFileSystemManager remoteFsm = new StandardFileSystemManager();
        FileObject source = null;
        FileObject destination = null;
        Boolean successfulTransmission;
        try {

            localFsm.init();
            remoteFsm.init();

            source = localFsm.resolveFile(localFileDirectory + fileInfo.getFileName());

            destination = remoteFsm.resolveFile(fileInfo.getEncodedFilePath(), createFileSystemOptions());
            moveTo(source, destination);

            successfulTransmission = true;
        } catch (FileSystemException fse) {
            LOG.error("Exception caught in TransferFile.moveTo() " + fse.getMessage(), fse);
            throw fse;
        } finally {
        	safeClose(source);
        	safeClose(destination);
            safeClose(localFsm);
            safeClose(remoteFsm);
        }
        return successfulTransmission;

    }

    

    /**
     * Method that obtains the String representation of the size in bytes of the file represented 
     * by the FileInfo instance coming in.
     *
     * @param fileInfo
     *            the file info
     * @return String value
     */
    public static String getSizeFor(FileInfo fileInfo){
        StandardFileSystemManager sfsm = new StandardFileSystemManager();
        String stringSize = "";
        
        FileObject fileObject = null;
        try {
            sfsm.init();
            fileObject = sfsm.resolveFile(fileInfo.getEncodedFilePath(), createFileSystemOptions());          
            long size = fileObject.getContent().getSize();
            stringSize = new Long(size).toString();
        } catch (FileSystemException fse) {
            LOG.error("Exception caught in TransferFile.getSizeFor() " + fse.getMessage(), fse);
            stringSize = "\'Unable to obtain the file size\'";
        } finally {
        	safeClose(fileObject);
            safeClose(sfsm);
        }
        return stringSize;

    }
    
    /**
     * Safe close.
     *
     * @param fileObject the file object
     */
    private static void safeClose(FileObject fileObject) {
    	 try {
             if (fileObject != null) {
            	 fileObject.close();
             }
         } catch (Exception exception) {
             LOG.error("Exception closing FileObject " + exception.toString());
         }
		
	}

	/**
     * Safe close.
     *
     * @param fileSystemManager
     *            the file system manager
     */
    private static void safeClose(DefaultFileSystemManager fileSystemManager) {
        try {
            if (fileSystemManager != null) {
                fileSystemManager.close();
            }
        } catch (Exception exception) {
            LOG.error("Exception closing fileSystemManager " + exception.toString());
        }
    }
}
