/**
 * Source file created in 2011 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.service.common.utility;


import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.List;

import gov.va.med.pharmacy.peps.common.exception.CsvFileException;
import gov.va.med.pharmacy.peps.common.vo.UpdateSiteVo;
import gov.va.med.pharmacy.peps.common.vo.ReportDataUpdateVo;
import gov.va.med.pharmacy.peps.common.vo.ReportType;


/**
 * Generates CSV File for Warning Label Report.
 *
 */
public class ReportDataUpdateCsvFile extends CsvFile {
   
    private static final String HEADER_FILE = "FILE:";
    private static final String HEADER_REL_DT = "RELEASE DATE:";
    private static final String HEADER_COMPLIANCE_DT = "COMPLIANCE DATE:";
    private static final String HEADER_AVG_DAYS_TO_INSTALL = "Average # of Days to Install";
    private static final String[] IN_COMPLIANCE_HEADER = 
            new String[] { "Sites in Compliance", "Install Date", "# Days to install" };
    private static final String[] OUT_OF_COMPLIANCE_HEADER = 
            new String[] { "Sites out of Compliance", "Last file successfully installed", 
                            "Install date", "# days out of compliance" };
    
    private static final String PATH = "./tmp/";
    private static final String FILENAME = ReportType.DATA_UPDATE_COMPLIANCE_REPORT.getView() + ".csv";

    private DateFormat dtf = new SimpleDateFormat("dd-MMM-yy");

    /**
     * Creates file without column headers.
     * @throws Exception 
     */
    public void createFile() throws Exception {
        // open file
        this.openWithoutHeader(PATH + FILENAME);
    }
    
    /**
     * Adds the Compliance Report data to the file contained in ReportDataUpdateVo.
     * @param reportDataUpdateVo
     * @throws Exception
     */
    public void printDataUpdateComplianceReport(ReportDataUpdateVo reportDataUpdateVo) throws Exception {
        putNextHeaderValuePairRow(HEADER_FILE, reportDataUpdateVo.getFilename());
        putNextHeaderValuePairRow(HEADER_REL_DT, dtf.format(reportDataUpdateVo.getReleaseDt()));
        putNextHeaderValuePairRow(HEADER_COMPLIANCE_DT, dtf.format(reportDataUpdateVo.getComplianceDt()));
        putSpace();
        
        putSitesInComplianceHeader();
        putSpace();
        int totalDays = 0;
        for (UpdateSiteVo site : reportDataUpdateVo.getSitesInCompliance()) {
            int daysOfInstall = site.getDaysOfInstall();
            putNextSiteInComplianceRow(addQuotesToStringWithComma(site), dtf.format(site.getInstallDt()), 
                    Integer.toString(daysOfInstall));
            totalDays += daysOfInstall;
        }
        putSpace();
        String avg = "";
        if (reportDataUpdateVo.getSitesInCompliance().size() > 0) {
            avg = String.format("%.1f", ((float)totalDays/reportDataUpdateVo.getSitesInCompliance().size()));
        }
        putSitesInComplianceFooter(avg);
        putSpace();
        
        putSitesOutOfComplianceHeader();
        putSpace();
        for (UpdateSiteVo site : reportDataUpdateVo.getSitesOuOfCompliance()) {
            putNextSiteOutOfComplianceRow(site);
        }
    }
    
    private void openWithoutHeader(String filename) throws CsvFileException {
        try {
            //open file for write (will create a new file or overwrite existing file)
            exportFile = new BufferedWriter(new FileWriter(filename));
        } catch (IOException e) {
            throw new CsvFileException("IO Exception opening file.");
        }
    }
    
    private void putNextHeaderValuePairRow(String header, String value) throws CsvFileException {
        putNextRowWithAnyNbrOfCols(header, value);
    }
    
    private void putSitesInComplianceHeader() throws CsvFileException {
        putNextRowWithAnyNbrOfCols(IN_COMPLIANCE_HEADER);
    }
    
    private void putSitesInComplianceFooter(String avg) throws CsvFileException {
        putNextRowWithAnyNbrOfCols(HEADER_AVG_DAYS_TO_INSTALL, "", avg);
    }
    
    private void putNextSiteInComplianceRow(String site, String installDt, String daysToInstall) throws CsvFileException {
        putNextRowWithAnyNbrOfCols(site, installDt, daysToInstall);
    }
    
    private void putSitesOutOfComplianceHeader() throws CsvFileException {
        putNextRowWithAnyNbrOfCols(OUT_OF_COMPLIANCE_HEADER);
    }
    
    private void putNextSiteOutOfComplianceRow(UpdateSiteVo site) throws CsvFileException {
        
        putNextRowWithAnyNbrOfCols(addQuotesToStringWithComma(site), 
                site.getFileName() == null ? "" : site.getFileName(), 
                site.getInstallDt() == null ? "" : dtf.format(site.getInstallDt()), 
                site.getDaysOutOfCompliance() == 0 ? "" : Integer.toString(site.getDaysOutOfCompliance()));
    }

    /**
     * Adds double quotes around the site name.
     * @param site UpdateSiteVo with the site name to quote.
     * @return String containing the site name with quotes around it.
     */
    private String addQuotesToStringWithComma(UpdateSiteVo site) {
        String siteName = site.getSiteName();
        siteName =  "\"" + siteName  + "\"";
        return siteName;
    }
    
    private void putNextRowWithAnyNbrOfCols(String... strings) throws CsvFileException {
        List<String> listOfStrings = Arrays.asList(strings);
        this.putNextRow(listOfStrings.toArray(new String[listOfStrings.size()]));
    }
    
    private void putSpace() throws CsvFileException {
        putNextRowWithAnyNbrOfCols(" ");
    }
}
