package gov.va.med.pharmacy.peps.service.common.utility;

import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.capability.NdfUpdateFileMgtDomainCapability;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 *  Represents the physical NDF Update File.
 *  Provides logic to Create the physical file, provide the channel to write to, and drives the transmission of the file to 
 *  the SFTP server.
 */

public class NdfUpdateProcessFile {
       
    /** File Extension for the NDF Update File. */
    private static final String FILE_EXTENSION = ".DAT";

    /**
     * Instantiates a new ndf update process file.
     */
    public NdfUpdateProcessFile() {

    }

    /** The Constant LOG. */
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager
        .getLogger(NdfUpdateProcessFile.class);

    /** The national setting domain capability. */
    private NationalSettingDomainCapability nationalSettingDomainCapability;
    
    /** The ndf update file mgt domain capability. */
    private NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> ndfUpdateFileMgtDomainCapability;
       
       
    /** channel*. */
    private SeekableByteChannel channel;
    
    //file name
    /** The file name. */
    private String fileName = null;
    
  //new sequence number
    /** The new seq number. */
    private String newSeqNumber = null;
    
    //file full path
    /** The full path. */
    private Path fullPath = null;
    
    //file creation options
    /** The options. */
    private final OpenOption[] options = {
        StandardOpenOption.WRITE,
        StandardOpenOption.APPEND,
        StandardOpenOption.CREATE_NEW};

    /** The new line. */
    private final String newLine = System.getProperty("line.separator");
    
    /**
     * Creates the file in the location specified in {@code NationalSetting.NDF_OUTPUT_FILE_DIRECTORY}, with
     * a filename that is derived by finding the last file transmitted to production and incrementing the NEW
     * sequence by one.
     *
     * @param newFileSeq the new file seq
     * @return the path
     * @throws IOException Signals that an I/O exception has occurred.
     */
    public final Path createFile(String newFileSeq) throws IOException  {
        
        LOG.info("Creating local file.");
        String localFileDirectory = getLocalFileDirectory();
        
        NdfUpdateFileVo lastCompletedNdfUpdateFile = 
                ndfUpdateFileMgtDomainCapability.retrieveLastTranmittedToProduction();      
        String prevFileSeq = "0";
        // We obtain the file name based on the previously successful file name not the file id
        if (lastCompletedNdfUpdateFile != null){             
            List<String> list = new ArrayList<String>();
            Pattern p = Pattern.compile("\\d+");
            Matcher m = p.matcher(lastCompletedNdfUpdateFile.getFileName()); 
            while (m.find()) {
                list.add(m.group());
             }            
            prevFileSeq = list.get(1); // we want the second number
        }
        
        int newSequenceNumber = new Integer(prevFileSeq).intValue() + 1;
        setNewSeqNumber(new Integer(newSequenceNumber).toString());
        setFileName("PPS_"+ prevFileSeq + "PRV" + "_" + getNewSeqNumber() + "NEW.DAT");
        setPath(Paths.get(localFileDirectory + getFileName()));        
        
        LOG.info("Local file: " + getPath().toString());
        try {
            setChannel(Files.newByteChannel(getPath(), options));            
        } catch (IOException e) {
            LOG.error("Exception caught in NdfUpdateProcessFile.createFile()" + e.getMessage(), e);
        }
        return getPath();
    }

    /**
     * Delete the local file, if it exists.
     *
     * @param ndfUpdateFile the metadata for the file to delete
     * @return boolean true = file existed and was deleted; otherwise false 
     */
    public static final boolean deleteFile(NdfUpdateFileVo ndfUpdateFile) {                    
        boolean wasFileDeleted = false;
        if (ndfUpdateFile != null){
            String fqn = ndfUpdateFile.getDirectoryPath();
            try {
                Files.delete(Paths.get(fqn));
                LOG.info("Deleted local file: " + fqn );
                wasFileDeleted = true;
            } catch (NoSuchFileException e1) {
                LOG.info("No local file found: " + fqn );                
            } catch (Exception e1) {
                //file permission or other problem.
                LOG.error("Error removing local file: " + fqn, e1);                
            }            
        }
        return wasFileDeleted;
    }

    /**
     * Gets the local file directory.
     *
     * @return the local file directory
     */
    private String getLocalFileDirectory() {
        NationalSettingVo returnedSetting = nationalSettingDomainCapability.
                retrieve(NationalSetting.NDF_OUTPUT_FILE_DIRECTORY.toString());
        String localFileDirectory = returnedSetting.getStringValue();
        return localFileDirectory;
    }    
    
    /**
     * send the file via sftp.
     *
     * @param environment String
     * @return boolean - true if successful
     * @throws IOException Signals that an I/O exception has occurred.
     */
    public final boolean transmitFile(String environment) throws IOException {
        
        boolean fileTransmitted = false;  

        if ("PROD".equals(environment)){

            if (TransferFile.moveTo(getPendingFileInfo(), getProdFileInfo())){
                fileTransmitted = true;
            }
            
        } else if ("TEST".equals(environment)){
            String localFileDirectory = nationalSettingDomainCapability.
                    retrieveString(NationalSetting.NDF_OUTPUT_FILE_DIRECTORY.toString());
            if(TransferFile.moveTo(localFileDirectory, getPendingFileInfo())){
                fileTransmitted = true;
            }
        }
        else if ("REJECT".equals(environment)){
            if (TransferFile.moveTo(getPendingFileInfo(), getRejectedFileInfo())){
                fileTransmitted = true;
            }
        }

        return fileTransmitted;
    }
    
    
    /**
     * Gets the pending file info.
     *
     * @return the pending file info
     */
    protected final FileInfo getPendingFileInfo() {
        FileInfo fileInfo = new FileInfo();
        
        fileInfo.setHostName(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_HOSTNAME.toString()));
        fileInfo.setPort(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_PORT.toString()));
        fileInfo.setUserName(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_USERNAME.toString()));
        fileInfo.setAuthorization(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_PASSWORD.toString()));
        fileInfo.setDirectoryPath(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_DIRECTORY.toString()));
        fileInfo.setFileName(getFileName());
        return fileInfo;
    }

    /**
     * Gets the prod file info.
     *
     * @return the prod file info
     */
    protected final  FileInfo getProdFileInfo() {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setHostName(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_PROD_HOSTNAME.toString()));
        fileInfo.setPort(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_PROD_PORT.toString()));
        fileInfo.setUserName(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_PROD_USERNAME.toString()));
        fileInfo.setAuthorization(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_PROD_PASSWORD.toString()));
        fileInfo.setDirectoryPath(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_PROD_DIRECTORY.toString()));
        fileInfo.setFileName(getFileName());
        return fileInfo;
    }
   
    /**
     * Gets the rejected file info.
     *
     * @return the rejected file info
     */
    protected final FileInfo getRejectedFileInfo() {
        FileInfo fileInfo = new FileInfo();
        
        fileInfo.setHostName(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_REJECTED_HOSTNAME.toString()));
        fileInfo.setPort(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_REJECTED_PORT.toString()));
        fileInfo.setUserName(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_REJECTED_USERNAME.toString()));
        fileInfo.setAuthorization(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_REJECTED_PASSWORD.toString()));
        fileInfo.setDirectoryPath(nationalSettingDomainCapability.
                retrieveString(NationalSetting.NDF_OUTPUT_TST_REJECTED_DIRECTORY.toString()));
        // when we reject a file we change its name                 
        fileInfo.setFileName(getRejectedFileName());
        return fileInfo;
    }
    
    
    /**
     * Returns the rejected name for the current file name.  The rejected file appends _REJECTED_YYYYMMdd_HHmm to the filename. 
     *
     * @return the rejected file name
     */
    public String getRejectedFileName(){
        SimpleDateFormat formater = new SimpleDateFormat("YYYYMMdd_HHmm");        
        String noExtensionName = getFileName().substring(0, getFileName().length() - FILE_EXTENSION.length()); 
        return  noExtensionName + "_REJECTED_" + (formater.format(Calendar.getInstance().getTime()) + FILE_EXTENSION);
    }
    
    /**
     * Write the next data row.
     *
     * @param dataString This is a data string.
     * @throws IOException Signals that an I/O exception has occurred.
     */
    public void putNextRow(String dataString) throws IOException {
       
      //Convert the string to a ByteBuffer. 
        byte data[] = (dataString+newLine).getBytes();
        ByteBuffer bb = ByteBuffer.wrap(data);
        
        try  {
            getChannel().write(bb);
            bb.clear();
        } catch (IOException e) {
            LOG.error("Exception caught in NdfUpdateProcessFile.putNextRow()" + e.getMessage(), e);
            throw e;
        }
    }


    /**
     * Close output file.
     *
     * @throws IOException Signals that an I/O exception has occurred.
     */
    public final void closeOutputFile() throws IOException {
        LOG.info("---------------closing file-------------------");
        try {
            channel.close();
        } catch (IOException e) {
            LOG.error("Exception caught in NdfUpdateProcessFile.closeOutputFile()" + e.getMessage(), e);
        }
    }
    

    /**
     * Gets the file name.
     *
     * @return the file name
     */
    public final String getFileName() {
        return fileName;
    }
    
    /**
     * Sets the file name.
     *
     * @param fileName the fileName to set
     */
    public final void setFileName(String fileName) {
        this.fileName = fileName;
    }


    /**
     * Gets the channel.
     *
     * @return the channel
     */
    public final SeekableByteChannel getChannel() {
        return channel;
    }


    /**
     * Sets the channel.
     *
     * @param channel the new channel
     */
    public final void setChannel(SeekableByteChannel channel) {
        this.channel = channel;
    }
    
    /**
     * Gets the path.
     *
     * @return the path
     */
    public final Path getPath() {
        return fullPath;
    }
    
    /**
     * Sets the path.
     *
     * @param path the path to set
     */
    public final void setPath(Path path) {
        this.fullPath = path;
    }

    /**
     * Gets the ndf update file mgt domain capability.
     *
     * @return the ndfUpdateFileMgtDomainCapabilityImpl
     */
    public final NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> getNdfUpdateFileMgtDomainCapability() {
        return ndfUpdateFileMgtDomainCapability;
    }
    
    /**
     * Sets the ndf update file mgt domain capability.
     *
     * @param ndfUpdateFileMgtDomainCapability the new ndf update file mgt domain capability
     */
    public void setNdfUpdateFileMgtDomainCapability(
            NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> ndfUpdateFileMgtDomainCapability) {
        this.ndfUpdateFileMgtDomainCapability = ndfUpdateFileMgtDomainCapability;
    }
    
    /**
     * Gets the national setting domain capability.
     *
     * @return the nationalSettingDomainCapability
     */
    public final NationalSettingDomainCapability getNationalSettingDomainCapability() {
        return nationalSettingDomainCapability;
    }
    
    /**
     * Sets the national setting domain capability.
     *
     * @param nationalSettingDomainCapability the nationalSettingDomainCapability to set
     */
    public final void setNationalSettingDomainCapability(NationalSettingDomainCapability nationalSettingDomainCapability) {
        this.nationalSettingDomainCapability = nationalSettingDomainCapability;
    }

    /**
     * Gets the new seq number.
     *
     * @return the newseqNumber
     */
    public final String getNewSeqNumber() {
        return newSeqNumber;
    }

    
    /**
     * Sets the new seq number.
     *
     * @param newSeqNumber the new new seq number
     */
    public final void setNewSeqNumber(String newSeqNumber) {
        this.newSeqNumber = newSeqNumber;
    }
    
    
    /**
     * Gets the pending file size.
     *
     * @return the pending file size
     */
    public String getPendingFileSize(){
         return TransferFile.getSizeFor(getPendingFileInfo());
    }
   
    /**
     * Gets the production file size.
     *
     * @return the production file size
     */
    public String getProductionFileSize(){
        return TransferFile.getSizeFor(getProdFileInfo());       
    }    
    
    /**
     * Gets the rejected file size.
     *
     * @return the rejected file size
     */    
    public String getRejectedFileSize(){
        return TransferFile.getSizeFor(getRejectedFileInfo());
    }
    
}
