/**
 * Source file created in 2008 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.service.common.session.impl;


import gov.va.med.pharmacy.peps.common.vo.DataUpdateFileVo;
import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.common.vo.ReportDataUpdateVo;
import gov.va.med.pharmacy.peps.common.vo.Role;
import gov.va.med.pharmacy.peps.common.vo.UpdateSiteVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.capability.ReportDomainCapability;
import gov.va.med.pharmacy.peps.service.common.reports.ReportExportState;
import gov.va.med.pharmacy.peps.service.common.reports.ReportObserver;
import gov.va.med.pharmacy.peps.service.common.reports.ReportProcessStatus;
import gov.va.med.pharmacy.peps.service.common.reports.ReportSubject;
import gov.va.med.pharmacy.peps.service.common.session.DomainService;
import gov.va.med.pharmacy.peps.service.common.session.ReportDataUpdateProcess;
import gov.va.med.pharmacy.peps.service.common.session.VistaUpdateSitesService;
import gov.va.med.pharmacy.peps.service.common.utility.ReportDataUpdateCsvFile;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import org.apache.logging.log4j.Logger; 
import org.apache.logging.log4j.LogManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidationType.LOG_FORGING;
import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidator.validateStringInput;


/** Generates the Ingredient list report in a background process. */

@Service
public class ReportDataUpdateProcessImpl implements ReportDataUpdateProcess, ReportSubject {

    private static final Logger LOG = org.apache.logging.log4j.LogManager.getLogger(ReportDataUpdateProcessImpl.class);
    private static final String ERROR = "Cound not open Data Update Compliance file for export: ";
    private static final int FILE_SIZE = 300;
    private static final Long USER_ID = 42L;
    private static final int TIMEOUT = 600;
    private static final int SLEEP = 5000;
    private static final int DAYS_IN_COMPLIANCE = 7;
    
    private volatile boolean running;
    private volatile ReportExportState exportState;
    private List<ReportObserver> observers;

    private ReportDomainCapability reportDomainCapability;
    private NationalSettingDomainCapability nationalSettingDomainCapability;
    private DomainService domainService;
    private VistaUpdateSitesService vistaUpdateSitesService;
    private PlatformTransactionManager transactionManager;

//    private volatile boolean completed;
//    private boolean suspended;

    @Override
    @Transactional
    public void run() {
        LOG.debug("=============> Starting run() in  ReportDataUpdateProcessImpl <====================");
        initialize();
        runProcesses();

        notifyObservers();
        LOG.debug("=============> Exiting run() in ReportDataUpdateProcessImpl <=======================");

    }

    /** Method runs process */

    private synchronized void runProcesses() {

        while (running) {
            try {
                LOG.debug("Export process started...");
                resetValues();
                runTask();

                // if we are done with all exports
                if (this.getExportState().isExportComplete()) {
                    LOG.debug("all file Exports are completed!");
                }

            } catch (InterruptedException e) {
                LOG.debug("Report Generate Process Interupted" + e);
                this.stopProcess();
            }

            notifyObservers();

        }

        this.stopProcess();

    }

    /** Reset state values */

    private void resetValues() {
        ReportExportState state = this.getExportState();
        state.setExportComplete(false);
        state.setRunning(true);
        state.setRecordCounter(0);
        state.setRecordTotal(0);
        state.setReportProcessStatus(ReportProcessStatus.RUNNING);

//        completed = false;
    }

    /** Initialize process */

    private void initialize() {

        LOG.debug("initializing report export process...");
        this.setExportState(new ReportExportState());
        running = true;
        getExportState().setReportProcessStatus(ReportProcessStatus.RUNNING);
        getExportState().setRunning(running);
        getExportState().setExportComplete(false);
        getExportState().setRecordCounter(0);
        getExportState().setRecordTotal(0);
        
        observers = new Vector<ReportObserver>();

//        suspended = false;

    }

    /** Stop process */

    public synchronized void stopProcess() {
        LOG.debug("==>>>Report Export process stopped...");
        running = false;
        getExportState().setReportProcessStatus(ReportProcessStatus.STOPPED);
        getExportState().setRunning(false);

//        suspended = false;
    }

    /** Run tasks within process 
     * @throws InterruptedException thrown if process interrupted 
     */
    private void runTask() throws InterruptedException {
        LOG.debug("starting runTask()...");

        if (Thread.interrupted()) {
            throw new InterruptedException();
        }

        doDataUpdateComplianceExport();

        this.getExportState().setExportComplete(true);
        running = false;

    }

    /** 
     * Generates Ingredient list Report
     * @throws InterruptedException 
     */
    private void doDataUpdateComplianceExport() throws InterruptedException {
        LOG.debug("Exporting Data Update Compliance Report");

        ReportExportState state = this.getExportState();
        
        try {

            //Set start time
            setDateValue(NationalSetting.REPORT_DATA_UPDATE_START, getDateTime());

            ReportDataUpdateCsvFile reportDataUpdateCsvFile = new ReportDataUpdateCsvFile();

            try {
                reportDataUpdateCsvFile.createFile();
            } catch (Exception e) {
                LOG.error("Cound create Data Update Compliance file for export: " + e);
            }
            
            ReportDataUpdateVo dataUpdateReport = new ReportDataUpdateVo();
            DataUpdateFileVo updateFile = reportDomainCapability.getLatestUpdateFileInfo(); 
            dataUpdateReport.setFilename(updateFile.getFileName());
            dataUpdateReport.setReleaseDt(updateFile.getReleaseDt());
            Calendar cal = new GregorianCalendar();
            cal.setTime(updateFile.getReleaseDt());
            cal.add(Calendar.DAY_OF_MONTH, DAYS_IN_COMPLIANCE);
            Date complianceDt = cal.getTime();
            dataUpdateReport.setComplianceDt(complianceDt);
            
            // find / set sites in compliance
            List<UpdateSiteVo> sitesInCompliance = reportDomainCapability.getSitesInCompliance(updateFile.getId(), complianceDt);
            List<Long> siteNbrsInCompliance = new ArrayList<Long>();
            for (UpdateSiteVo siteVo : sitesInCompliance) {
                siteNbrsInCompliance.add(siteVo.getSiteNbr());
                siteVo.setSiteName(vistaUpdateSitesService.getNameBySiteNumber(siteVo.getSiteNbr()));
                siteVo.setDaysOfInstall(daysBetween(siteVo.getInstallDt(), updateFile.getReleaseDt()));
            }
            Collections.sort(sitesInCompliance);    // sort site names alphabetically
            dataUpdateReport.setSitesInCompliance(sitesInCompliance);
            
            // find / set sites out of compliance
            List<UpdateSiteVo> sitesOutOfCompliance = vistaUpdateSitesService.findSitesOutOfCompliances(siteNbrsInCompliance);
            for (UpdateSiteVo oocSite : sitesOutOfCompliance) {
                UpdateSiteVo siteVo = reportDomainCapability.getSiteOutOfComplianceInfo(oocSite.getSiteNbr());
                oocSite.setFileName(siteVo == null ? "" : siteVo.getFileName());
                oocSite.setInstallDt(siteVo == null ? null : siteVo.getInstallDt());
                oocSite.setDaysOutOfCompliance(siteVo == null ? 0 : daysBetween(siteVo.getInstallDt(), complianceDt));
            }
            Collections.sort(sitesOutOfCompliance);    // sort site names alphabetically
            dataUpdateReport.setSitesOuOfCompliance(sitesOutOfCompliance);
            
            // print report
            reportDataUpdateCsvFile.printDataUpdateComplianceReport(dataUpdateReport);
            
            reportDataUpdateCsvFile.closeExport();

            //Set end time
            setDateValue(NationalSetting.REPORT_DATA_UPDATE_COMPLETE, getDateTime());

        } 
        catch (Exception e) {
            LOG.error("FAILED TO generate Data Update Compliance Report. Error Message is ", e);
            this.stopProcess();
        }

    }
    
    private int daysBetween(Date d1, Date d2) {
        if ( d1 == null ) {
            LOG.warn("The d1 parameter passed to daysBetween() is null.");
            return 0;
        }
        if ( d2 == null ) {
            LOG.warn("The d2 parameter passed to daysBetween() is null.");
            return 0;
        }
        
        long diff = d2.getTime() - d1.getTime();
        return (int)(Math.abs(diff / (24 * 60 * 60 * 1000)));
    }

    /**
     * getUser for ReportIngredientProcessImpl
     * @return UserVo
     */
    private UserVo getUser() {
        UserVo user = new UserVo();
        user.setFirstName("FDBAuto");
        user.setLastName("AddProcess");
        user.setStationNumber("999");
        user.setUsername("FdbAutoAddProcess");
        user.setLocation("National");
        user.setId(USER_ID);

        /*NATIONAL_SERVICE_MANAGER*/
        user.addRole(Role.PSS_PPSN_MANAGER);

        return user;

    }

    /**
     * Obtains the current date/time to set the date start or stop date value.
     * @return dateValue
     */
    private Date getDateTime() {

        DateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss", Locale.US);
        Date dateValue = new Date();

        try {

            LOG.debug("Date/Time: " + dateFormat.format(dateValue));

        } catch (Exception e) {
            LOG.error("Exception parsing Start Date/Time: ", e);
        }

//        transactionManager.commit(status);

        return dateValue;
    }

    /**
     * setStartDate.
     * @param type type
     * @param date date
     */
    private void setDateValue(NationalSetting type, Date date) {

        try {
//            DefaultTransactionDefinition def = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
//            def.setTimeout(TIMEOUT);
//            TransactionStatus status = transactionManager.getTransaction(def);

//            List<NationalSettingVo> settingsList = nationalSettingDomainCapability.retrieve();
//
//            for (NationalSettingVo vo : settingsList) {
//                if (type.toString().equals(vo.getKeyName())) {
//                    vo.setDateValue(date);
//                    nationalSettingDomainCapability.update(vo, getUser());
//                }
//            }
//            NationalSettingVo vo = nationalSettingDomainCapability.retrieve(type.toString());
//            if ( vo != null ) {
//                vo.setDateValue(date);
//                nationalSettingDomainCapability.update(vo,  getUser());
//                }
//            transactionManager.commit(status);

//        } catch (TransactionException e) {
//            LOG.error("TransactionException, in setStartDate()  ", e);
//
//            if (e.getCause() != null) {
//                LOG.error("The Underlying cause is " + validateStringInput(e.getCause().getMessage(), LOG_FORGING));
//            }
            nationalSettingDomainCapability.update(type.toString(), date, getUser());
        } catch (Exception e) {
            LOG.error("Exception getting National Settings for " + type + ":", e);
        }

    }

    @Override
    public void registerObserver(ReportObserver pObserver) {
        observers.add(pObserver);

    }

    @Override
    public void removeObserver(ReportObserver pObserver) {
        observers.remove(pObserver);

    }

    @Override
    public void notifyObservers() {

        for (int loopIndex = 0; loopIndex < observers.size(); loopIndex++) {
            ReportObserver observer = (ReportObserver) observers.get(loopIndex);
            observer.update(this.getExportState());
        }

    }

    /**
     * Get export state
     * @return the exportState
     */
    public ReportExportState getExportState() {
        return exportState;
    }

    /**
     * Set export state
     * @param exportState the exportState to set
     */
    public void setExportState(ReportExportState exportState) {
        this.exportState = exportState;
    }

    @Override
    public int getRecordCounter() {
        return 0;
    }

    @Override
    public void setReportProcessStatus(ReportProcessStatus reportProcessStatus) {
    }

    /**
     * getReportDomainCapability
     * @return reportDomainCapability ReportDomainCapability
     */
    public ReportDomainCapability getReportDomainCapability() {
        return reportDomainCapability;
    }

    /**
     * Sets reportDomainCapability
     * @param reportDomainCapability reportDomainCapability
     */
    public void setReportDomainCapability(ReportDomainCapability reportDomainCapability) {
        this.reportDomainCapability = reportDomainCapability;
    }

    /**
     * getNationalSettingDomainCapability
     * @return nationalSettingDomainCapability 
     */
    public NationalSettingDomainCapability getNationalSettingDomainCapability() {
        return nationalSettingDomainCapability;
    }

    /**
     * setNationalSettingDomainCapability
     * @param nationalSettingDomainCapability nationalSettingDomainCapability
     */
    public void setNationalSettingDomainCapability(NationalSettingDomainCapability nationalSettingDomainCapability) {
        this.nationalSettingDomainCapability = nationalSettingDomainCapability;
    }

    /**
     * getDomainService
     * @return domainService
     */
    public DomainService getDomainService() {
        return domainService;
    }

    /**
     * setDomainService
     * @param domainService domainService
     */
    public void setDomainService(DomainService domainService) {
        this.domainService = domainService;
    }
    
    public VistaUpdateSitesService getVistaUpdateSitesService() {
        return vistaUpdateSitesService;
    }
    
    public void setVistaUpdateSitesService(VistaUpdateSitesService vistaUpdateSitesService) {
        this.vistaUpdateSitesService = vistaUpdateSitesService;
    }

    /**
     * getTransactionManager
     * @return transactionManager
     */
    public PlatformTransactionManager getTransactionManager() {
        return transactionManager;
    }

    /**
     * This method is used by Spring Injection to set the TransactionManager.
     * @param transactionManager transactionManager
     */
    public void setTransactionManager(PlatformTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }

}
