package gov.va.med.pharmacy.peps.service.common.session.impl;

import gov.va.med.pharmacy.peps.common.exception.ValidationException;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.common.vo.PaginatedList;
import gov.va.med.pharmacy.peps.common.vo.StatusVo;
import gov.va.med.pharmacy.peps.common.vo.UpdateSearchCriteria;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NdfUpdateFileMgtDomainCapability;
import gov.va.med.pharmacy.peps.service.common.capability.NdfUpdateCapability;
import gov.va.med.pharmacy.peps.service.common.capability.NdfUpdateProcessCapability;
import gov.va.med.pharmacy.peps.service.common.session.NDFUpdateFileService;

import java.util.List;

import javax.annotation.Resource;

import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * NDF Update File service to create, view, and update NdfUpdateFileVo.
 */
@Service
public class NdfUpdateFileServiceImpl implements NDFUpdateFileService{
    
    /** The ndf update capability. */
    @Resource
    private NdfUpdateCapability ndfUpdateCapability;
    
    /** The ndf update process capability. */
    @Resource
    private NdfUpdateProcessCapability ndfUpdateProcessCapability;
    /** The ndf update file mgt domain capability. */
    @Resource
    private NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> ndfUpdateFileMgtDomainCapability;    
    
    /**
     * Retrieve file updates.
     *
     * @return List<FileUpdateVo> all UpdateFiles
     */
    @Override
    @Transactional(readOnly=true)
    public List<NdfUpdateFileVo> retrieveFileUpdates() {
        return ndfUpdateCapability.retrieveFileUpdates();
    }
    
    /**
     * Retrieve file updates.
     *
     * @param searchCriteria the search criteria
     * @return List<FileUpdateVo> paged UpdateFiles
     */
    @Transactional(readOnly=true)
    public PaginatedList<NdfUpdateFileVo> retrievePagedFileUpdates(UpdateSearchCriteria searchCriteria) {
        return ndfUpdateCapability.retrievePagedFileUpdates(searchCriteria);
    }
    
    /**
     * Retrieve all file statuses.
     *
     * @return List<StatusVo> all File Update Statuses
     */
    @Override
    @Transactional(readOnly=true)
    public List<StatusVo> retrieveAllFileStatuses() {
        return ndfUpdateCapability.retrieveAllFileStatuses();
    }    
    
    /**
     * Retrieve NdfUpdateFileVo ready for approval/rejection.
     *
     * @return List<FileUpdateVo> pending UpdateFiles
     */
    @Override
    @Transactional(readOnly=true)
    public List<NdfUpdateFileVo> retrievePendingNdfUpdate() {
        return ndfUpdateCapability.retrievePendingUpdate();
    }
    
    /**
     * Update the specified NdfUpdateFileVo in the database.
     * @param updatedFile The file    
     * @param user the user logged in
     */
   @Override
   @Transactional
   public void updateNdfUpdateFile(NdfUpdateFileVo updatedFile, UserVo user) {
       ndfUpdateCapability.updateNdfUpdateFile(updatedFile, user);
   }  

   
   /**
    * Retrieve NdfUpdateFileVo by id.
    *
    * @param id the id
    * @return FileUpdateVo pending UpdateFile to be updated
    */
   @Override   
   @Transactional
   public NdfUpdateFileVo retrieveById(Long id) {
       return ndfUpdateCapability.retrieveById(id);
   }
   
   /**
    * Retrieve a list of FileUpdateVo with initiated status. Realistically, there should only ever be one.
    * 
    * @return List<FileUpdateVo> initiated UpdateFiles
    */
   @Override
   @Transactional(readOnly=true)
   public List<NdfUpdateFileVo> retrieveInitiatedNdfUpdate() {
       return ndfUpdateCapability.retrieveInitiatedUpdate();
   }
   
   /**
    * Insert NdfUpdateFileVo.
    *
    * @param insertFile the insert file
    * @param user the user logged in
    */
   @Override
   @Transactional
   public void insertNdfUpdateFile(NdfUpdateFileVo insertFile, UserVo user) {
       ndfUpdateCapability.insertNdfUpdateFile(insertFile, user);
   } 
   
   /**
    * Process file creation, which includes creating the file and sending it to the test SFTP server.
    *
    * @param user the user logged in
    */
   @Override 
   @Transactional
   public void processFileCreation(UserVo user) {
       ndfUpdateCapability.process(user);
   }  
   
   /**
    * Remove the specified NdfUpdateFileVo.
    * 
    * @return remove referenced file id
    */
   @Override   
   @Transactional
   public void removeNdfUpdateFileFk(Long id) {
       ndfUpdateProcessCapability.removeAllFileAssociations(id);
   }
   
   @Override
   @Transactional
   public void transmitToProd(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException {
       
       ndfUpdateCapability.transmitUpdateFileToProd(updateFile, user);
   } 
   
   @Override
   @Transactional
   public void transmitToReject(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException {
       
       ndfUpdateCapability.transmitUpdateFileToReject(updateFile, user);
   } 
   
   @Override
   @Transactional
   public NdfUpdateFileVo retrieveCurrentStatus() {
       
       return ndfUpdateCapability.retrieveCurrentStatus();
   } 

    /**
     * Creates and stores an NdfUpdateFileVo with {@code StatusVo.INITIATED} status.
     * @param user
     * @return NdfUpdateFileVo the saved NdfUpdateFileVo with status {@code StatusVo.INITIATED}
     */
    @Override
    @Transactional
    public NdfUpdateFileVo createInitiatedFile(UserVo user) {
        return ndfUpdateCapability.createInitiatedFile(user);
    }   

    /**
     * Approve.  Updates the status and transmits the file to the production folder.
     *
     * @param toUpdate the to update
     * @param user the user
     * @throws ValidationException the validation exception
     */
    @Transactional(rollbackForClassName="gov.va.med.pharmacy.peps.common.exception.ValidationException")
    public void approve(NdfUpdateFileVo toUpdate, UserVo user) 
            throws ValidationException{
        ndfUpdateCapability.approve(toUpdate, user);
    }

    /**
     * Reject.  Updates the status, moves the file to the reject folder, and remove the associations between the File 
     * and the data.
     *
     * @param toUpdate the to update
     * @param user the user
     * @throws ValidationException the validation exception
     */
    @Transactional(rollbackForClassName="gov.va.med.pharmacy.peps.common.exception.ValidationException")
    public void reject(NdfUpdateFileVo toUpdate, UserVo user)
            throws ValidationException{
        ndfUpdateCapability.reject(toUpdate, user);
    }

    public NdfUpdateCapability getNdfUpdateCapability() {
        return ndfUpdateCapability;
    }


    public void setNdfUpdateCapability(NdfUpdateCapability ndfUpdateCapability) {
        this.ndfUpdateCapability = ndfUpdateCapability;
    }

    
    public NdfUpdateProcessCapability getNdfUpdateProcessCapability() {
        return ndfUpdateProcessCapability;
    }

    
    public void setNdfUpdateProcessCapability(NdfUpdateProcessCapability ndfUpdateProcessCapability) {
        this.ndfUpdateProcessCapability = ndfUpdateProcessCapability;
    }       
}
