package gov.va.med.pharmacy.peps.service.common.capability.impl;

import gov.va.med.pharmacy.peps.common.utility.VistaFileNumber;
import gov.va.med.pharmacy.peps.domain.common.model.EplNdfOutgoingDifferences;
import gov.va.med.pharmacy.peps.service.common.capability.WordProcessingFieldHelper;
import gov.va.med.pharmacy.peps.service.common.update.impl.VADrugClassNdfFileSyncProcessor;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateNdfDataMapping;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile;

import java.io.IOException;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

/**
 * This is the implementation class for the WordProcessingFieldHelper interface.
 * 
 * @author vhaisaduffg
 *
 */
@Service
public class WordProcessingFieldHelperImpl implements WordProcessingFieldHelper {

    /**
     * The maximum length of a String chunk.
     */
    private static final Integer MAX_VISTA_FIELD_LEN = 250;
    
    /**
     * The line.separator system property.
     */
    private static final String LINE_SEPARATOR = System.lineSeparator();
    
    /**
     * A character to indicate a blank line.
     */
    private static final Character BLANK_LINE_INDICATOR = new Character(
            (char) 0xFF);

    /**
     * The NdfUpdateNdfDataMapping object.
     */
    private NdfUpdateNdfDataMapping ndfUpdateNdfDataMapping = new NdfUpdateNdfDataMapping();

    /**
     * Check the VistA File Number and Field Number to see if the value is a
     * Word-Processing field in VistA.
     * 
     * @return boolean true if the field is a Word-Processing field
     */
    @Override
    public final boolean isWordProcessingField(String fileNumber, String fieldNumber) {
        if (VistaFileNumber.FIFTY_605.getFileNumber().equals(fileNumber)
                && VADrugClassNdfFileSyncProcessor.DESCRIPTION_FIELD_NUMBER
                        .equals(fieldNumber)) {
            return true;
        }

        return false;
    }

    /**
     * Breaks up longer Strings into multiple Strings that are a maximum length
     * of MAX_VISTA_FIELD_LEN. These multiple Strings are then formatted and
     * written to the update file.
     * 
     * In order to preserve blank lines, the method replaces all occurrences of
     * LINE.SEPARATOR in the String with the BLANK_LINE_INDICATOR +
     * LINE.SEPARATOR. This is to prevent the removal of lines with just a
     * LINE.SEPARATOR on it which could equate to a NULL lines when splitting by
     * LINE.SEPARATOR. This BLANK_LINE_INDICATOR is removed before the string
     * value is written to the file.s
     * 
     * @param ndfUpdateFile
     *            The NdfUpdateProcessFile being used.
     * @param difference
     *            The current difference being processed.
     * @param fileNumber
     *            The VistA File Number
     * @param fieldNumber
     *            The VistA Field Number
     * @throws IOException
     *             If an error occurs writing to the NdfUpdateProcessFile.
     */
    public final void formatWordProcessingField(NdfUpdateProcessFile ndfUpdateFile,
            EplNdfOutgoingDifferences difference, String fileNumber,
            String fieldNumber) throws IOException {
        final String blankLinesIndicated = difference.getNewValue().replaceAll(
                LINE_SEPARATOR, BLANK_LINE_INDICATOR + LINE_SEPARATOR);
        String[] splitByLineSeparator = splitByLineSeparator(blankLinesIndicated);
        int count = 1;

        ndfUpdateFile.putNextRow(ndfUpdateNdfDataMapping
                .formatHeaderRowForDATAN(fileNumber, count));
        ndfUpdateFile.putNextRow(ndfUpdateNdfDataMapping
                .formatIenRowForDATAN(difference));

        for (int i = 0; i < splitByLineSeparator.length; i++) {
            String[] paraSplit = split(splitByLineSeparator[i],
                    MAX_VISTA_FIELD_LEN);
            for (int j = 0; j < paraSplit.length; j++) {
                ndfUpdateFile.putNextRow(ndfUpdateNdfDataMapping
                        .formatHeaderRowForDATAN(
                                difference.getVistaFileNumber(), ++count));
                ndfUpdateFile.putNextRow(paraSplit[j].replaceAll(
                        BLANK_LINE_INDICATOR + "", ""));
            }
        }
    }

    /**
     * Splits the supplied text in to chunks by the line.separator value.
     * 
     * @param text
     *            the text to split
     * @return an array of Strings split by the line.separator
     */
    private String[] splitByLineSeparator(String text) {
        return StringUtils.split(text, LINE_SEPARATOR);
    }

    /**
     * Splits a String into equal chunkSize Strings.
     * 
     * @param text
     *            the String to split
     * @param chunkSize
     *            the size of each split out substring
     * @return String[]
     */
    private String[] split(String text, int chunkSize) {
        char[] data = text.toCharArray();
        int len = Math.min(data.length, text.length());
        String[] result = new String[(len + chunkSize - 1) / chunkSize];
        int linha = 0;
        for (int i = 0; i < len; i += chunkSize) {
            result[linha] = new String(data, i, Math.min(chunkSize, len - i));
            linha++;
        }
        return result;
    }
}
