package gov.va.med.pharmacy.peps.service.common.capability.impl;

import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidationType.LOG_FORGING;
import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidator.validateStringInput;
import gov.va.med.pharmacy.peps.common.email.Email;
import gov.va.med.pharmacy.peps.common.email.EmailService;
import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.model.EplProductDo;
import gov.va.med.pharmacy.peps.domain.common.utility.RxNormCapability;
import gov.va.med.pharmacy.peps.service.common.capability.RxNormUpdateCapability;
import gov.va.med.pharmacy.peps.service.common.scheduler.ProcessStatus;

import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;

/**
 * This class is the main implementation that manages the RxNorm update process
 * 
 * @author VHAISHEdgecK
 *
 */
public class RxNormUpdateCapabilityImpl implements RxNormUpdateCapability {
    
    private static final Logger LOG = LogManager.getLogger(RxNormUpdateCapabilityImpl.class);
    private NationalSettingDomainCapability nationalSettingDomainCapability;
    private RxNormCapability rxNormCapability;
    private static final int TIMEOUT = 200;
    private static final int LONG_TIMEOUT = 1200;
    
    @Autowired
    private PlatformTransactionManager transactionManager;    
    
    @Autowired
    private EmailService emailService;


    /* (non-Javadoc)
     * @see gov.va.med.pharmacy.peps.service.common.capability.RxNormUpdateCapability#process(gov.va.med.pharmacy.peps.common.vo.UserVo)
     */
    public void process(UserVo user) {
       
        try {
            updateJobStatusToRunning(user);
            
            List<EplProductDo> products = fetchRxNormUpdateProducts();
            
            processEachProduct(user, products);
            
            updateJobStatusToCompleted(user);
            
        } catch (Exception e) {
    		e.printStackTrace();
    		LOG.error("Exception in RxNormUpdateCapabilityImpl.process()  " + validateStringInput(e.getMessage(), LOG_FORGING), e);
    		if (e.getCause() != null) {
    		    LOG.error("The Underlying cause is " + e.getCause().getMessage(), e);
    		}
            updateJobStatusToError(user);
        }
    }

    
	/**
	 * Calls the RxNorm domain capability to fetch and return a list of EPL Product objects that are enabled
	 * for rxNorm updates
	 * 
	 * @return the list of EPL Product objects to process
	 */
    @Transactional(noRollbackForClassName="java.lang.Exception")   
	private List<EplProductDo> fetchRxNormUpdateProducts() {
		DefaultTransactionDefinition def;
		TransactionStatus status;
		def = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
		def.setTimeout(LONG_TIMEOUT);
		status = transactionManager.getTransaction(def);
		
		List<EplProductDo> products = rxNormCapability.fetchRxNormUpdateProducts();
		transactionManager.commit(status);
		
		return products;
	}

	/**
	 * Updates the RxNorm job status to Error
	 * 
	 * @param user
	 * @param e
	 */
    @Transactional(noRollbackForClassName="java.lang.Exception")   
	private void updateJobStatusToError(UserVo user) {
		DefaultTransactionDefinition def;
		TransactionStatus status;
		def = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
		def.setTimeout(TIMEOUT);
		status = transactionManager.getTransaction(def);
		
		updateStatus(ProcessStatus.ERROR, user);
		
		transactionManager.commit(status);
	}

	/**
	 * This method iterates thru a list of products, creates a new transaction for each product, and passes processing 
	 * down into the populate and persist method of the RxNorm domain capability object.
	 * @param user
	 * @param products the list of products to send to rxnorm
	 */
    @Transactional(noRollbackForClassName="java.lang.Exception")   
	private void processEachProduct(UserVo user, List<EplProductDo> products) {
		DefaultTransactionDefinition def;
		TransactionStatus status;

		Date runDate = new Date();
		for (EplProductDo product : products) {
            def = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            def.setTimeout(TIMEOUT);
            status = transactionManager.getTransaction(def);

            rxNormCapability.populateAndPersist(user, product, runDate);
            
            // Job completes normally
            transactionManager.commit(status);

            // Delay for 50 milliseconds so that we never send more than 20 per second (NLM web service req)
            try {
            	TimeUnit.MILLISECONDS.sleep(50);
            } catch(InterruptedException ie) {
            	//ignore
            }
        }
	}

	/**
	 * Updates the RxNorm job status to Completed
	 * 
	 * @param user
	 */
    @Transactional(noRollbackForClassName="java.lang.Exception")   
	private void updateJobStatusToCompleted(UserVo user) {
		DefaultTransactionDefinition def;
		TransactionStatus status;
		def = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
		def.setTimeout(TIMEOUT);
		status = transactionManager.getTransaction(def);
		    
		updateStatus(ProcessStatus.COMPLETED, user);
		updateRuntime(user);
   
		transactionManager.commit(status);
	}

	/**
	 * Updates the RxNorm job status to Running
	 * 
	 * @param user
	 */
    @Transactional(noRollbackForClassName="java.lang.Exception")   
	private void updateJobStatusToRunning(UserVo user) {
		DefaultTransactionDefinition def;
		TransactionStatus status;
		def = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
		def.setTimeout(TIMEOUT);
		status = transactionManager.getTransaction(def);

		LOG.info("---------------processing RXNORM Update-------------------");
		
		updateStatus(ProcessStatus.RUNNING, user);
		
		transactionManager.commit(status);
	}
 
    /**
     * The updateRuntime method will update the RxNorm Update run status. 
     * @param processStatus ProcessStatus
     */
    private void updateStatus(ProcessStatus processStatus, UserVo user) {

        NationalSettingVo returnedSetting = nationalSettingDomainCapability.retrieve(NationalSetting.RXNORM_UPDATE_RUN_STATE.toString());

        returnedSetting.setStringValue(processStatus.toString());
        nationalSettingDomainCapability.update(returnedSetting, user);
    }

    /**
     * The updateRuntime method will update the RxNorm Update runtime date. 
     */
    private void updateRuntime(UserVo user) {

        NationalSettingVo returnedSetting = nationalSettingDomainCapability.retrieve(NationalSetting.RXNORM_UPDATE_LAST_RUN.toString());

        returnedSetting.setDateValue(new Date());
        nationalSettingDomainCapability.update(returnedSetting, user);
    }
    
    
    /**
     * gets the TransactionManager for RxNormUpdateCapabilityImpl.
     * @return the transactionManager
     */
    public PlatformTransactionManager getTransactionManager() {
        return transactionManager;
    }

    /**
     * sets the TransactionManager for RxNormUpdateCapabilityImpl.
     * @param transactionManager the transactionManager to set
     */
    public void setTransactionManager(PlatformTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }
    
    /**
     * setNationalSettingDomainCapability in ProposedInactivateCapabilityImpl.
     * @param nationalSettingDomainCapability the nationalSettingDomainCapability to set
     */
    public void setNationalSettingDomainCapability(NationalSettingDomainCapability
                                                   nationalSettingDomainCapability) {
        this.nationalSettingDomainCapability = nationalSettingDomainCapability;
    }
    
    /**
     * Gets the rx norm capability.
     * @return the rx norm capability
     */
    public RxNormCapability getRxNormCapability() {
        return rxNormCapability;
    }
    /**
     * Sets the rx norm capability.
     * @param rxNormCapability the new rx norm capability
     */
    public void setRxNormCapability(RxNormCapability rxNormCapability) {
        this.rxNormCapability = rxNormCapability;
    }
    
    
    /**
     * @return the emailService
     */
    public EmailService getEmailService() {
        return emailService;
    }

    
    /**
     * @param emailService the emailService to set
     */
    public void setEmailService(EmailService emailService) {
        this.emailService = emailService;
    }
    
    
    /**
     * Sends text email notification.
     * 
     * @param toEmailIds
     * @param emailSubject
     * @param emailMessage
     */
    private void sendEmailNotification(Long status, String filename) {
        String toEmailIds = null;  
        String emailSubject = null;
        String emailMessage = null;
        
        if(status == 3  ){                      
            // For SQA
            toEmailIds ="     .     @domain,              .Gopi@domain,David.        @domain,Shahzad.Butt@domain"; 
            emailSubject="Update File Transmitted to Test FTP - " + filename;
            emailMessage="\n\n Transmission of the Created file " + filename + " to the test FTP location has been completed.";
            
         }        
        else if(status == 8){
            toEmailIds ="pbmndf@domain,Shahzad.Butt@domain";
            emailSubject="Update File Transmitted to Production FTP - " + filename;
            emailMessage="\n\n Transmission of the approved file " + filename + " to the production FTP location has been completed.";
        }
        
        try {
                      
            Email emailBean = new Email();
            
            /*
             * Need to set to in another way  by converting to an array as setting addresses like emailid1@domain ,emailid2@domain
             * Gives error by javax mail Address Parser.
             */
            String[] toEmailId = new String[]{""};
            
            // if toEmailIdStr string is a comma separated string then split it. 
            if(null!=toEmailIds && toEmailIds.length()>0)
            {
                toEmailId = toEmailIds.split(",");
            }               
                                
            emailBean.setTo(toEmailId);        
            emailBean.setSubject(emailSubject);        
            emailBean.setEmailMessage(emailMessage);
            
            emailService.sendEmail(emailBean);
               
        } catch (Exception e) {
            LOG.error("Error in sendEmailNotification ", e);
        }
    }    
      
}