/*
 * 
 */
package gov.va.med.pharmacy.peps.service.common.capability.impl;

import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidationType.LOG_FORGING;
import static gov.va.med.pharmacy.peps.common.utility.ESAPIValidator.validateStringInput;
import gov.va.med.pharmacy.peps.common.email.Email;
import gov.va.med.pharmacy.peps.common.email.EmailService;
import gov.va.med.pharmacy.peps.common.exception.ValidationException;
import gov.va.med.pharmacy.peps.common.exception.ValueObjectValidationException;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.common.vo.PaginatedList;
import gov.va.med.pharmacy.peps.common.vo.StatusVo;
import gov.va.med.pharmacy.peps.common.vo.UpdateSearchCriteria;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.common.vo.validator.ErrorKey;
import gov.va.med.pharmacy.peps.common.vo.validator.Errors;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.capability.NdfUpdateFileMgtDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.capability.NdfUpdateFileStatusDomainCapability;
import gov.va.med.pharmacy.peps.service.common.capability.NdfUpdateCapability;
import gov.va.med.pharmacy.peps.service.common.capability.NdfUpdateProcessCapability;
import gov.va.med.pharmacy.peps.service.common.scheduler.ProcessStatus;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile;

import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.PropertySource;
import org.springframework.core.env.Environment;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * Contains the main logic to create the NDF Update File and send email notifications.
 */
@Configuration
@PropertySource("classpath:gov.va.med.pharmacy.peps.emailservice.properties")
public class NdfUpdateCapabilityImpl implements NdfUpdateCapability {

    /** The Logger. */
    private static final Logger LOG = LogManager
            .getLogger(NdfUpdateCapabilityImpl.class);
    
    /** The national setting domain capability. */
    private NationalSettingDomainCapability nationalSettingDomainCapability;
    
    /** The NdfUpdateProcessCapability. */
    private NdfUpdateProcessCapability ndfUpdateProcessCapability;
    
    /** The NdfUpdateFileMgtDomainCapability. */
    private NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> ndfUpdateFileMgtDomainCapability;
    
    /** The NdfUpdateFileStatusDomainCapability. */
    private NdfUpdateFileStatusDomainCapability ndfUpdateFileStatusDomainCapability;
    
    /** The retrieved ndf update file. */
    private NdfUpdateFileVo retrievedNdfUpdateFile;
    
    /** The NdfUpdateProcessFile. */
    private NdfUpdateProcessFile ndfUpdateProcessFile;

    /** The transaction manager. */
    @Resource
    private PlatformTransactionManager transactionManager;

    /** The email service. */
    @Resource
    private EmailService emailService;

    /** The environment. */
    @Resource
    private Environment environment;

    /**
     * Main process of the NDF Update Job that creates the update file and associated metadata.
     *
     * @param user the user
     */
    @Transactional
    public void process(UserVo user) {
        ProcessStatus processStatus = null;    
        NdfUpdateFileVo newFileVo = null;

        try {
            LOG.info("---------------processing NDF Update-------------------");
            updateStatus(ProcessStatus.RUNNING, user);            
            Calendar gc = GregorianCalendar.getInstance();
            Date startDTM = gc.getTime();
            boolean shouldGenerateFile = true;
            
            NdfUpdateFileVo latestFileVo = retrieveCurrentStatus();
            if (latestFileVo == null){
                //very first time creating a file
                newFileVo = createInitiatedFile(user);
            }                  
            else if (StatusVo.INITIATED == latestFileVo.getStatus().getStatusId()){
                //on-demand request via the UI
                LOG.info("This is an on-demand NDF Update file creation.");
                newFileVo = latestFileVo;
            }
            else if (StatusVo.NON_FINAL_STATE_STATUSES.contains(latestFileVo.getStatus().getStatusId())){
                LOG.warn("Cannot create new file, as the previous file is not in a final state.");
                //update runtime status (in finally)
                shouldGenerateFile = false;
            }
            else if (StatusVo.PPSN_ERROR == latestFileVo.getStatus().getStatusId()){
                //on-demand would have changed error to initiated already, so only scheduled would get here.
                LOG.info("This is a scheduled NDF Update file creation.");
                //reuse the NdfUpdateFileVo                
                newFileVo = latestFileVo;
                newFileVo.setComments(null);
                newFileVo.setDirectoryPath(null);
                newFileVo.setFileName(null);                
                newFileVo.getStatus().setStatusId(StatusVo.INITIATED);
                //update status in a separate Txn so we prevent others from initiating the job.
                updateNdfUpdateFileInSeparateTxn(newFileVo, user);
            }                
            else{
                LOG.info("This is a scheduled NDF Update file creation.");
                newFileVo = createInitiatedFile(user);
            }
                    
            if (shouldGenerateFile){
        
                createAndAttachPhysicalFile(user, newFileVo);
    
                processData(newFileVo, startDTM);
    
                transmitUpdateFileToTest(newFileVo, user);
            }
                        
        } catch (Exception e) {
            processStatus = handleExceptionDuringCreateAndSendToTestProcess(user, newFileVo, e);
            cleanupFromExceptionDuringCreateAndSendToTestProcess(newFileVo);            
        }
        finally
        {
            processStatus = ProcessStatus.COMPLETED;
            updateJobStatusAndRunTime(processStatus, user);
        }
    }

    /**
     * Adds the data to the file and associates the data to the file id of the NdfUpdateFileVo.
     *
     * @param newFileVo the new NdfUpdateFileVo
     * @param startDTM the start date of the process
     * @throws IOException Signals that an I/O exception has occurred.
     */
    private void processData(NdfUpdateFileVo newFileVo, Date startDTM) throws IOException {
   
        addDataToFile(startDTM);  
        updateFileFKForDataIncludedInFile(newFileVo, startDTM);
    }

    /**
     * Create the local physical file and attach it to specified NdfUpdateFileVo.
     * @param user User
     * @param newFileVo NdfUpdateFileVo to associate to the new physical file
     * @throws IOException Thrown if errors encountered during physical file creation
     */ 
    private void createAndAttachPhysicalFile(UserVo user, NdfUpdateFileVo newFileVo) throws IOException {

        ndfUpdateProcessFile.createFile(newFileVo.getFileId());
   
        newFileVo.setFileName(ndfUpdateProcessFile.getFileName());
        newFileVo.setDirectoryPath(ndfUpdateProcessFile.getPath().toString());
        newFileVo.setStatus(new StatusVo(StatusVo.CREATED));
        updateNdfUpdateFile(newFileVo, user);
    }

    /**
     * Handles an exception during the file creation process.
     * @param user User who invoked the file creation
     * @param newFileVo the NdfUpdateFileVo being processed
     * @param e the exception encountered
     * @return ProcessStatus.ERROR
     */
    private ProcessStatus handleExceptionDuringCreateAndSendToTestProcess(UserVo user, NdfUpdateFileVo newFileVo, Exception e) {
        try{
            LOG.error("Exception in NdfUpdateCapabilityImpl.process() - "
                    + validateStringInput(e.getMessage(), LOG_FORGING), e);
            if (newFileVo != null){
                newFileVo.setStatus(new StatusVo(StatusVo.PPSN_ERROR));
                newFileVo.setComments("PPS-N/NDF Update Test File Creation Failed. Log a CA ticket");
                updateNdfUpdateFileInSeparateTxn(newFileVo, user);
            }
        }
        catch (Exception e1){
            LOG.error("Secondary exception encountered during cleanup after a primary exception.", e1);
        }
        return ProcessStatus.ERROR;
    }

    /**
     * Updates the NDF Update job status to Completed and the runtime completion to now.
     *
     * @param processStatus the process status
     * @param user the user
     */  
    @Transactional(propagation=Propagation.REQUIRES_NEW)
    public void updateJobStatusAndRunTime(ProcessStatus processStatus, UserVo user) {

        LOG.info("Updating Job Status and Runtime.");
            
        updateStatus(processStatus, user);
        updateRuntime(user);
    }    
    
    /**
     * Update file fk for data included in file.
     *
     * @param updateFile the NdfUpdateFileVo
     * @param startDTM the start date of the process
     */
    private void updateFileFKForDataIncludedInFile(NdfUpdateFileVo updateFile, Date startDTM) {
        LOG.info("Associating data to file.");
        ndfUpdateProcessCapability.updateNdfUpdateFileFk(updateFile);
        ndfUpdateProcessCapability.updateRematchNdfUpdateFileFk(startDTM,
                updateFile);
    }

    /**
     * Writes all the data to the physical file.
     * @param startDTM the start date of the process
     * @throws IOException Thrown if errors encountered writing data to the file
     */
    private void addDataToFile(Date startDTM) throws IOException {
        LOG.info("Writing data to file.");
        //for local development testing, it is handy to comment out the PEM, Label, and Map Data.
        ndfUpdateProcessCapability.processPemData(ndfUpdateProcessFile);
        ndfUpdateProcessCapability
                .processWarningLabelData(ndfUpdateProcessFile);
        ndfUpdateProcessCapability
                .processVaProcuctMapData(ndfUpdateProcessFile);
        ndfUpdateProcessCapability.processNdfData(startDTM,
                ndfUpdateProcessFile);
        ndfUpdateProcessCapability.processMiscellaneousNdfDataItems(
                startDTM, ndfUpdateProcessFile);
        ndfUpdateProcessCapability.processTextItems(ndfUpdateProcessFile);
        ndfUpdateProcessFile.closeOutputFile();
        
    }

    /**
     * Creates an NdfUpdateFileVo with initiated status. 
     * @param user User initiating the file
     * @return NdfUpdateFileVo with initiated status
     */
    public NdfUpdateFileVo createInitiatedFile(UserVo user) {
        NdfUpdateFileVo initiatedFile = new NdfUpdateFileVo();
        initiatedFile.setStatus(new StatusVo(StatusVo.INITIATED));
        initiatedFile.setFileName("");
        NdfUpdateFileVo updateFile = insertNdfUpdateFile(initiatedFile, user);
        return updateFile;
    }

    /**
     * Approve.  Updates the status and transmits the file to the production folder.
     *
     * @param toUpdate the to update
     * @param user the user
     * @throws ValidationException the value object validation exception
     */
    public void approve(NdfUpdateFileVo toUpdate, UserVo user) throws ValidationException {
        try {
            transmitUpdateFileToProd(toUpdate, user);
        } catch (ValidationException e) {
            throw e;
        } catch (Exception e) {
            throw new ValidationException(ValidationException.FILE_APPROVAL_FAILED);
        }
    }
    
    /**
     * Reject.  Updates the status, moves the file to the reject folder, and remove the associations between the File 
     * and the data.
     *
     * @param toUpdate the to update
     * @param user the user
     * @throws ValidationException the value object validation exception
     */
    public void reject(NdfUpdateFileVo toUpdate, UserVo user)
            throws ValidationException {
        
        try {
            if (StringUtils.isBlank(toUpdate.getComments())){
                Errors errors = new Errors();
                errors.addError(FieldKey.UPDATE_FILE, ErrorKey.FILE_REJECTION_COMMENTS_REQUIRED);                        
                throw new ValueObjectValidationException(errors);
            }   
            
            transmitUpdateFileToReject(toUpdate, user);
        } catch (ValidationException e) {
            throw e;            
        } catch (Exception e) {
            throw new ValidationException(ValidationException.FILE_REJECTION_FAILED);
        }
    }       
    
    /**
     * Transmit update file to production - Moves the file to the production folder, sends an email notification, and stores 
     * the NdfUpdateFileVo with an updated status.  
     *
     * @param updateFile the update file
     * @param user the user
     * @throws ValidationException Thrown if File transfer is turned off in the National Settings, if the file transmission 
     *      failed, or some other error was encountered.
     */
    public void transmitUpdateFileToProd(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException {
        LOG.info("Preparing to transmit " + updateFile.getFileName() + " file to Prod.");
        final String transmission = "PROD";
        
        if (!isTransmitFileTurnedOn()) {         
            throw new ValidationException(ValidationException.FTP_TURNED_OFF, transmission);
        }
        boolean transmissionWasSuccess  = false;
        
        try {
            if(ndfUpdateProcessFile.getFileName() == null){
                ndfUpdateProcessFile.setFileName(updateFile.getFileName());
            }
            transmissionWasSuccess = ndfUpdateProcessFile.transmitFile(transmission);
            if (transmissionWasSuccess) {
                Calendar gc = GregorianCalendar.getInstance();
                updateFile.setProdTransmissionDtm(gc.getTime());
                updateFile.setStatus(new StatusVo(StatusVo.TRANSMITTED_TO_PRODUCTION));
                sendEmailNotification(updateFile.getStatus(), updateFile.getFileName(), 
                        ndfUpdateProcessFile.getProductionFileSize());
            } else{ 
                throw new ValidationException(ValidationException.FILE_APPROVAL_FAILED);
            }
        } 
        catch (ValidationException e) {
            throw e;
        }
        catch (Exception e) {            
            LOG.error("Exception in NdfUpdateCapabilityImpl.transmitUpdateFileToProd()", e);
            if (transmissionWasSuccess){
                updateFile.setComments("PPS-N/NDF Update Production File Transmission Successful, but subsequent errors encountered sending email. Log a CA ticket.");
                LOG.error("File was sent, but subsequent error occurred sending email.", e);
                //the file was sent, but the email or status updates failed.
            }
            else{               
                throw new ValidationException(ValidationException.FILE_APPROVAL_FAILED);
            }
        }
        finally{
            if (transmissionWasSuccess){
                updateNdfUpdateFileInSeparateTxn(updateFile, user);      
            }
        }        
    }
    
    /**
     * Transmit update file to test - Moves the file to the test folder, sends an email notification, and stores 
     * the NdfUpdateFileVo with an updated status.  
     *
     * @param updateFile the update file
     * @param user the user
     * @throws ValidationException Thrown if File transfer is turned off in the National Settings, if the file transmission 
     *      failed, or some other error was encountered.
     */    
    public void transmitUpdateFileToTest(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException {
        LOG.info("Preparing to transmit " + updateFile.getFileName() + " file to Test.");

        final String transmission = "TEST";
        
        if (!isTransmitFileTurnedOn()) {
            throw new ValidationException(ValidationException.FTP_TURNED_OFF, transmission);
        }

        boolean transmissionWasSuccess = false;

        try {
            if(ndfUpdateProcessFile.getFileName() == null){
                ndfUpdateProcessFile.setFileName(updateFile.getFileName());
            }

            transmissionWasSuccess = ndfUpdateProcessFile.transmitFile(transmission);
            if (transmissionWasSuccess) {
                Calendar gc = GregorianCalendar.getInstance();
                updateFile.setTestTransmissionDtm(gc.getTime());
                updateFile.setStatus(new StatusVo(StatusVo.READY_FOR_TESTING));
                sendEmailNotification(updateFile.getStatus(), updateFile.getFileName(), 
                        ndfUpdateProcessFile.getPendingFileSize());
            }
        } 
        catch (Exception e) {
            LOG.error("Exception in NdfUpdateCapabilityImpl.transmitUpdateFileToTest()", e);

            if (transmissionWasSuccess){
                LOG.error("File was sent, but subsequent error occurred sending email.", e);
                updateFile.setComments(
                        "PPS-N/NDF Update Test File Transmission Successful, but subsequent errors encountered sending email. "
                                + "Log a CA ticket.");                
                //the file was sent, but the email or status updates failed.
            }
            else{
                updateFile.setStatus(new StatusVo(StatusVo.PPSN_ERROR));
                throw new ValidationException(ValidationException.FILE_CREATION_PROCESS_FAILED);
            }
        } 
        finally{
            if (transmissionWasSuccess){
                updateNdfUpdateFileInSeparateTxn(updateFile, user);      
            }
        }            
    }    
    
    /**
     * Transmit update file to reject - Renames and moves the file to the reject folder, sends an email notification, 
     * and stores the NdfUpdateFileVo with an updated status.  
     *
     * @param updateFile the update file
     * @param user the user
     * @throws ValidationException Thrown if File transfer is turned off in the National Settings, if the file transmission 
     *      failed, or some other error was encountered.
     */       
    public void transmitUpdateFileToReject(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException {
        LOG.info("Preparing to rename and transmit " + updateFile.getFileName() + " file to the Rejected folder.");

        final String transmission = "REJECT";
        if (!isTransmitFileTurnedOn()) {
            throw new ValidationException(ValidationException.FTP_TURNED_OFF, transmission);
        }        

        boolean transmissionWasSuccess = false;
        
        try {
            if(ndfUpdateProcessFile.getFileName() == null){
                ndfUpdateProcessFile.setFileName(updateFile.getFileName());
            }
            
            transmissionWasSuccess = ndfUpdateProcessFile.transmitFile(transmission);
            if (transmissionWasSuccess) {
                ndfUpdateProcessCapability.removeAllFileAssociations(Long.valueOf(updateFile.getFileId()));
                
                //If the caller specified VISTA_ERROR, store VISTA_ERROR, otherwise, set the status to REJECTED_BY_QA 
                if (updateFile.getStatus().getStatusId() != StatusVo.VISTA_ERROR)
                {
                    updateFile.setStatus(new StatusVo(StatusVo.REJECTED_BY_QA));
                }
                
                // we rename the file name in the database also
                updateFile.setFileName(ndfUpdateProcessFile.getRejectedFileName());
                sendEmailNotification(updateFile.getStatus(), updateFile.getFileName(), 
                        ndfUpdateProcessFile.getRejectedFileSize());
            }
        }        
        catch (Exception e) {            
            LOG.error("Exception in NdfUpdateCapabilityImpl.transmitUpdateFileToReject()", e);
            if (transmissionWasSuccess){
                //the file was sent, but the email or status updates failed.
                updateFile.setComments(
                        "PPS-N/NDF Update Reject File Rename/Move Successful, but subsequent errors encountered sending email. "
                        + "Log a CA ticket.");
                LOG.error("File was sent, but subsequent error occurred sending email or removing file associations from the database.", e);
            }
            else{               
                throw new ValidationException(ValidationException.FILE_REJECTION_FAILED);
            }
        }
        finally{
            if (transmissionWasSuccess){
                updateNdfUpdateFileInSeparateTxn(updateFile, user);      
            }
        }             
        
    }       
    
    /**
     * Update the Inactivation Check runtime date.
     *
     * @param processStatus            ProcessStatus
     * @param user the user
     */
    private void updateStatus(ProcessStatus processStatus, UserVo user) {
        if (processStatus == null){
            return;            
        }
        NationalSettingVo returnedSetting = nationalSettingDomainCapability
                .retrieve(NationalSetting.NDF_UPDATE_RUN_STATE.toString());

        returnedSetting.setStringValue(processStatus.toString());
        nationalSettingDomainCapability.update(returnedSetting, user);
    }

    /**
     * Update the NDF Update runtime date.
     *
     * @param user the user
     */
    private void updateRuntime(UserVo user) {

        NationalSettingVo returnedSetting = nationalSettingDomainCapability
                .retrieve(NationalSetting.NDF_UPDATE_LAST_RUN.toString());

        returnedSetting.setDateValue(new Date());
        nationalSettingDomainCapability.update(returnedSetting, user);
    }

    /**
     * Update the given NdfUpdateFileVo in the database and send an e-mail
     * notification to the appropriate recipients.
     *
     * @param updatedFile ManagedItemVo to update
     */
    public void ndfUpdateFileNotification(NdfUpdateFileVo updatedFile) {
        ndfUpdateProcessFile.setFileName(updatedFile.getFileName());
        String fileSize = ndfUpdateProcessFile.getPendingFileSize();
        sendEmailNotification(updatedFile.getStatus(),
                updatedFile.getFileName(), fileSize);
    }

    /**
     * Update the given NdfUpdateFileVo in the database.
     * 
     * @param updatedFile
     *            ManagedItemVo to update
     * @param user
     *            UserVo updating item
     */
    @Override
    public void updateNdfUpdateFile(NdfUpdateFileVo updatedFile, UserVo user) {
        LOG.info("Updating NDFUpdateFile Info.");
        Calendar gc = GregorianCalendar.getInstance();

        try {
            updatedFile.setModifiedDate(gc.getTime());
            updatedFile.setStatusModifiedDtm(gc.getTime());
            updatedFile.setModifiedBy(user.getUsername());

            ndfUpdateFileMgtDomainCapability.update(updatedFile, user);

        } catch (TransactionException e) {
            LOG.error(
                    "TransactionException, in NdfUpdateCapabilityImpl.updateNdfUpdateFile()  "
                            + e.getMessage(), e);
            throw e;
        }
    }

    /**
     * Update the given NdfUpdateFileVo in the database, in a separate transaction.  Useful for updating status and comments 
     * during error scenarios.
     * 
     * @param updatedFile
     *            ManagedItemVo to update
     * @param user
     *            UserVo updating item
     */   
    @Transactional(propagation=Propagation.REQUIRES_NEW)
    public void updateNdfUpdateFileInSeparateTxn(NdfUpdateFileVo updatedFile, UserVo user)
    {
        updateNdfUpdateFile(updatedFile, user);       
    }

    /**
     * Insert the given NdfUpdateFileVo in the database.
     *
     * @param insertFile the insert file
     * @param user            UserVo updating item
     * @return the ndf update file vo
     */
    @Override
    public NdfUpdateFileVo insertNdfUpdateFile(NdfUpdateFileVo insertFile,
            UserVo user) {
        NdfUpdateFileVo insertedFile = null;
        Calendar gc = GregorianCalendar.getInstance();

        try {
            insertFile.setModifiedBy(user.getUsername());
            insertFile.setModifiedDate(gc.getTime());
            insertFile.setStatusModifiedDtm(gc.getTime());

            insertedFile = ndfUpdateFileMgtDomainCapability.insert(insertFile,
                    user);
        } catch (TransactionException e) {
            LOG.error(
                    "TransactionException, in NdfUpdateCapabilityImpl.insertNdfUpdateFile()  "
                            + e.getMessage(), e);

            if (e.getCause() != null) {
                LOG.error(
                        "The Underlying cause is "
                                + validateStringInput(
                                        e.getCause().getMessage(), LOG_FORGING),
                        e);
            }
            LOG.error(
                    "TransactionException, in NdfUpdateCapabilityImpl.insertNdfUpdateFile()",
                    e);
            throw e;
        }
        return insertedFile;
    }

    /**
     * Retrieve all NdfUpdateFileVo.
     * @return List<FileUpdateVo> updateFiles
     */
    @Override
    public List<NdfUpdateFileVo> retrieveFileUpdates() {
        return ndfUpdateFileMgtDomainCapability.retrieveAll();
    }

    /**
     * Retrieve paged NdfUpdateFileVo.
     *
     * @param searchCriteria
     *            the search criteria
     * @return List<FileUpdateVo> updateFiles
     */
    @Override
    public PaginatedList<NdfUpdateFileVo> retrievePagedFileUpdates(
            UpdateSearchCriteria searchCriteria) {
        return ndfUpdateFileMgtDomainCapability.retrievePaged(searchCriteria);
    }

    /**
     * Retrieve all the status values (EPL_STATUS).
     * @return List<StatusVo> all update file statuses
     */
    @Override
    public List<StatusVo> retrieveAllFileStatuses() {
        return ndfUpdateFileStatusDomainCapability.retrieveAll();
    }

    /**
     * Retrieve NdfUpdateFileVo ready for approval/rejection.
     * @return List<NdfUpdateFileVo> pendingUpdateFiles
     */
    @Override
    public List<NdfUpdateFileVo> retrievePendingUpdate() {
        return ndfUpdateFileMgtDomainCapability.retrievePending();
    }

    /**
     * Retrieve NdfUpdateFileVo by NDFUpdateFileId.
     * @param id NDFUpdateFileId 
     * @return NdfUpdateFileVo The found file row, or null if the id is null or no row found with that id.
     */
    @Override
    public NdfUpdateFileVo retrieveById(Long id) {
        return ndfUpdateFileMgtDomainCapability.retrieveById(id);
    }

    /**
     * Retrieves the latest NdfUpdateFileVo, meaning the one with the max file id.
     *
     * @return the NdfUpdateFileVo with the highest file id
     */
    @Override
    public NdfUpdateFileVo retrieveCurrentStatus() {
        return ndfUpdateFileMgtDomainCapability.retrieveCurrentStatus();
    }

    /**
     * Retrieve a list of FileUpdateVo with initiated status.  Realistically, there should only ever be one.
     *
     * @return List<FileUpdateVo> initiatedUpdateFiles
     */
    @Override
    public List<NdfUpdateFileVo> retrieveInitiatedUpdate() {
        return ndfUpdateFileMgtDomainCapability.retrieveInitiated();
    }

    /**
     * gets the TransactionManager for NdfUpdateCapabilityImpl.
     * 
     * @return the transactionManager
     */
    public PlatformTransactionManager getTransactionManager() {
        return transactionManager;
    }

    /**
     * sets the TransactionManager for NdfUpdateCapabilityImpl.
     * 
     * @param transactionManager
     *            the transactionManager to set
     */
    public void setTransactionManager(
            PlatformTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }

    /**
     * setNationalSettingDomainCapability in ProposedInactivateCapabilityImpl.
     * 
     * @param nationalSettingDomainCapability
     *            the nationalSettingDomainCapability to set
     */
    public void setNationalSettingDomainCapability(
            NationalSettingDomainCapability nationalSettingDomainCapability) {
        this.nationalSettingDomainCapability = nationalSettingDomainCapability;
    }

    /**
     * setNdfUpdateProcessCapability in NdfUpdateProcessCapabilityImpl.
     *
     * @param ndfUpdateProcessCapability the new NdfUpdateProcessCapability
     */
    public void setNdfUpdateProcessCapability(NdfUpdateProcessCapability ndfUpdateProcessCapability) {
        this.ndfUpdateProcessCapability = ndfUpdateProcessCapability;
    }

    /**
     * Gets the NdfUpdateFileMgtDomainCapability.
     *
     * @return the NdfUpdateFileMgtDomainCapability
     */
    public NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> getNdfUpdateFileMgtDomainCapability() {
        return ndfUpdateFileMgtDomainCapability;
    }

    /**
     * Sets the NdfUpdateFileMgtDomainCapability.
     *
     * @param ndfUpdateFileMgtDomainCapability the new NdfUpdateFileMgtDomainCapability
     */
    public void setNdfUpdateFileMgtDomainCapability(
            NdfUpdateFileMgtDomainCapability<NdfUpdateFileVo> ndfUpdateFileMgtDomainCapability) {
        this.ndfUpdateFileMgtDomainCapability = ndfUpdateFileMgtDomainCapability;
    }

    /**
     * Gets the retrieved ndf update file.
     *
     * @return the setRetrievedNdfUpdateFile
     */
    public NdfUpdateFileVo getRetrievedNdfUpdateFile() {
        return retrievedNdfUpdateFile;
    }

    /**
     * Sets the retrieved ndf update file.
     *
     * @param retrievedNdfUpdateFile the new retrieved ndf update file
     */
    public void setRetrievedNdfUpdateFile(NdfUpdateFileVo retrievedNdfUpdateFile) {
        this.retrievedNdfUpdateFile = retrievedNdfUpdateFile;
    }

    /**
     * Get the reference to the NdfUpdateProcessFile, which represents a physical file and its actions.
     *
     * @return the NdfUpdateProcessFile
     */
    public NdfUpdateProcessFile getNdfUpdateProcessFile() {
        return ndfUpdateProcessFile;
    }

    /**
     * Set the reference to the NdfUpdateProcessFile, which represents a physical file and its actions.
     *
     * @param ndfUpdateProcessFile the new NdfUpdateProcessFile
     */
    public void setNdfUpdateProcessFile(NdfUpdateProcessFile ndfUpdateProcessFile) {
        this.ndfUpdateProcessFile = ndfUpdateProcessFile;
    }

    /**
     * Gets the email service.
     *
     * @return the emailService
     */
    public EmailService getEmailService() {
        return emailService;
    }

    /**
     * Sets the email service.
     *
     * @param emailService the emailService to set
     */
    public void setEmailService(EmailService emailService) {
        this.emailService = emailService;
    }

    /**
     * Sends text email notification.
     *
     * @param status the status
     * @param filename the filename
     * @param fileSize the file size
     */
    private void sendEmailNotification(StatusVo status, String filename, String fileSize)
    {
        String toEmailIds = null;  
        String emailSubject = null;
        String emailMessage = null;
        
                
        if(status.isReadyForTesting()){                      
            // File ready for SQA            
            emailSubject="PPS-N/NDF File ["+ filename +"] CREATED FOR QA";
            emailMessage="\n\nThe PPS-N/NDF File " + filename + " (Size " + fileSize  + " bytes) "
                    + "has been CREATED and TRANSMITTED to the test sFTP.  It is now ready to "
                    + "be DOWNLOADED and INSTALLED.";
            toEmailIds = environment.getRequiredProperty("update_file_on_test_site_to_email");
         }  // File has been Approved/Released      
        else if(status.isTransmittedToProduction()){
            emailSubject="PPS-N/NDF File " + filename + " RELEASED NATIONALLY";
            emailMessage="\n\nThe PPS-N/NDF Update File " + filename + " (Size " + fileSize  + " bytes) "
                    + "has been Nationally Released to the Product sFTP and is available to be "
                    + "DOWNLOADED and INSTALLED at the local sites.";
            toEmailIds = environment.getRequiredProperty("update_file_approved_to_email");
        }  // File has been Rejected
        else if (status.isRejectedByQA()){
            emailSubject="PPS-N/NDF File " + filename + " REJECTED";
            emailMessage="\n\nThe PPS-N/NDF Update Test File " + filename + " (Size " + fileSize  + " bytes) "
                    + "has been REJECTED by the PPS-N Manager.";
            toEmailIds = environment.getRequiredProperty("update_file_rejected_to_email");
        } else if ( status.isVistaError()) {
            emailSubject="PPS-N/NDF File " + filename + " AUTO REJECTED";
            emailMessage="\n\nThe PPS-N/NDF File " + filename + " (Size " + fileSize + " bytes) has been "
                    + "AUTO REJECTED by QA.";
           toEmailIds = environment.getRequiredProperty("update_file_rejected_to_email");
        }

        try {

            Email emailBean = new Email();

            /*
             * Need to set to in another way by converting to an array as
             * setting addresses like emailid1@domain ,emailid2@domain Gives
             * error by javax mail Address Parser.
             */
            String[] toEmailId = new String[] { "" };

            // if toEmailIdStr string is a comma separated string then split it.
            if (null != toEmailIds && toEmailIds.length() > 0) {
                toEmailId = toEmailIds.split(",");
            }

            emailBean.setTo(toEmailId);
            emailBean.setSubject(emailSubject);
            emailBean.setEmailMessage(emailMessage);

            emailService.sendEmailMutlipleTimes(emailBean);

        } catch (Exception e) {
            LOG.error("Error in sendEmailNotification ", e);
        }
    }  
    
    /**
     * Cleanup from a PPSN error during file creation or sending to test.
     *
     * @param ndfUpdateFileVo the NdfUpdateFileVo with {@code StatusVo.PPSN_ERROR}status.
     * @return true, if successful
     */   
    private boolean cleanupFromExceptionDuringCreateAndSendToTestProcess(NdfUpdateFileVo ndfUpdateFileVo) {
        try{            
            //cleanup the physical file on the local drive, if it exists.
            if (ndfUpdateFileVo != null){
                NdfUpdateProcessFile.deleteFile(ndfUpdateFileVo);
            }          
        }
        catch (Exception e1){
            LOG.error("Secondary exception encountered during cleanup after a primary exception.", e1);
        }        
        return true;
    }  
    
    /**
     * Gets the ndf update file status domain capability.
     *
     * @return the ndf update file status domain capability
     */
    public NdfUpdateFileStatusDomainCapability getNdfUpdateFileStatusDomainCapability() {
        return ndfUpdateFileStatusDomainCapability;
    }

    /**
     * Sets the ndf update file status domain capability.
     *
     * @param ndfUpdateFileStatusDomainCapability the new ndf update file status domain capability
     */
    public void setNdfUpdateFileStatusDomainCapability(
            NdfUpdateFileStatusDomainCapability ndfUpdateFileStatusDomainCapability) {
        this.ndfUpdateFileStatusDomainCapability = ndfUpdateFileStatusDomainCapability;
    }

    /**
     * Checks if is transmit file turned on.
     *
     * @return true, if is transmit file turned on
     */
    private boolean isTransmitFileTurnedOn(){
        NationalSettingVo returnedTransmitSetting = nationalSettingDomainCapability
                .retrieve(NationalSetting.NDF_OUTPUT_FTP_ON.toString());
        return returnedTransmitSetting.getBooleanValue();
    }
    
    
}
