package gov.va.med.pharmacy.peps.service.common.capability.impl;

import gov.va.med.pharmacy.peps.common.utility.VistaFileNumber;
import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.dao.EplNdfOutgoingDifferencesDao;
import gov.va.med.pharmacy.peps.domain.common.dao.EplRematchSuggestionDao;
import gov.va.med.pharmacy.peps.service.common.capability.MessageDataCapability;
import gov.va.med.pharmacy.peps.service.common.utility.MessageFormatter;
import gov.va.med.pharmacy.peps.service.common.utility.NDFUpdateFileMapping;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateMessageUtility;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Implementation of MessageDataCapability to build the reports sent via email message.
 */
public class MessageDataCapabilityImpl implements MessageDataCapability {

    private static final String END_SQUARE = "]";
	private static final String OPENING_SQUARE = "[";
	public static final String NEWLINE = System.getProperty("line.separator");
	private static final String EMPTYLINE = "";
    /* consolidated here from NdfUpdateMessageUtility */
    public static final String SPACER = "     ";
    public static final String SPACER2 = "          ";
    public static final String SPACER3 = "                 ";
    public static final String DOUBLEDSPACE = "    ";
    public static final String SPACE = "  ";
    public static final String ONE_SPACE = " ";
    /* consolidated here from NDFUpdateFileMapping */
    public static final String OLD = "      Old Value: ";
    public static final String NEW = "      New Value: ";

    /** Logger for this class. */
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager
            .getLogger(MessageDataCapabilityImpl.class);

    /** The EplNdfOutgoingDifferencesDao. */
    private EplNdfOutgoingDifferencesDao eplNdfOutgoingDifferencesDao;

    /** The EplRematchSuggestionDao. */
    private EplRematchSuggestionDao eplRematchSuggestionDao;

    /** The NationalSettingDomainCapability. */
    private NationalSettingDomainCapability nationalSettingDomainCapability;

    /** Title for Formulary Section of report. */
    public static final String FORMULARY = "FORMULARY ITEMS";

    /** Title for Non-Formulary Section of report. */
    public static final String NONFORMULARY = "NON-FORMULARY ITEMS";

    /** Verbiage used when no records found for a section. */
    public static final String NONE = "NONE";

    /** Ending parentheses to match the starting parentheses in some of the verbiage constants below. */
    public static final String ENDROUNDBRAC = ")";

    /** The Constant CMOP. */
    public static final String CMOP = "(CMOP - ";

    /** The Constant DISPENSE. */
    public static final String DISPENSE = "(DISPENSE UNIT - ";

    /** The Constant AUTOCREATEQUEST. */
    public static final String AUTOCREATEQUEST = "  Auto-Create Default Possible Dosage? ";

    /** The Constant AUTOCREATENEG. */
    public static final String AUTOCREATENEG = "  Auto-Create Default Possible Dosage? No ";

    /** The Constant POSSIBLEDOSAGESTMT. */
    public static final String POSSIBLEDOSAGESTMT = "  Possible Dosages To Auto-Create: ";

    /** The Constant PACKAGE. */
    public static final String PACKAGE = "  Package: ";

    /** The Constant ADD. */
    public static final String ADD = "ADDED";

    /** The Constant EDIT. */
    public static final String EDIT = "EDITED";

    /** The Constant DELETE. */
    public static final String DELETE = "DELETED";

    /** The Constant ADDINT. */
    public static final String ADDINT = "ADDED INTERACTION";

    /** The Constant EDITINT. */
    public static final String EDITINT = "EDITED INTERACTIONS";

    /** The Constant INACTIVATEINT. */
    public static final String INACTIVATEINT = "INACTIVATED INTERACTIONS";

    /** The Constant USE. */
    public static final String USE = "  USE";

    /*
     * (non-Javadoc)
     *
     * @see gov.va.med.pharmacy.peps.service.common.capability.MessageDataCapability#messageReportData(java.util.Date,
     * gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile)
     */
    @Override
    public void messageReportData(Date startDTM, NdfUpdateProcessFile ndfUpdateFile) throws IOException {
      
    	String report = makeReport(startDTM, new Date()).stream().collect(Collectors.joining(NEWLINE));
    	
        ndfUpdateFile.putNextRow(report);

    }

    private List<String> getReportData(Date startDate) {
        List<String> result = new ArrayList<String>();
        
        result.addAll(addedProductReport(startDate));
        result.addAll(inactivatedCMOPReport(startDate));
        result.addAll(futureInactiveRematchReport());
        result.addAll(futureInactivationReport());
        result.addAll(inactivatedRematchReport(startDate));
        result.addAll(inactivatedRematchReportSugg(startDate));
        result.addAll(inactivatedProductsReport(startDate));
        result.addAll(reactivatedProductReport(startDate));
        result.addAll(nationalFormularyChangeReport(startDate));
        result.addAll(genNameAllChangeReport(startDate));
        result.addAll(printNameChangeReport(startDate));
        result.addAll(cmopChangeReport(startDate));
        result.addAll(strengthChangeReport(startDate));
        result.addAll(drugUnitsChangeReport(startDate));
        result.addAll(dispenseUnitsChangeReport(startDate));
        result.addAll(scheduleAllChangeReport(startDate));
        result.addAll(possibleDosageReport(startDate));
        result.addAll(overrideDFChckExclReport(startDate));
        result.addAll(dosageFormChangeReport(startDate));
        result.addAll(otherRematchedReport(startDate));
        result.addAll(otherRematchSuggReport(startDate));
        result.addAll(otherRematchNoSuggReport(startDate));
        result.addAll(classChangeReport(startDate));
        result.addAll(newClassReport(startDate));
       
        result.add(EMPTYLINE);
        
        return result;
    }

    private List<String> makeReport(Date startDTM, Date today) {
        List<String> reportData = getReportData(startDTM);
        MessageFormatter formatter = new MessageFormatter(today, reportData);
        return formatter.apply(NDFUpdateFileMapping.MESSAGE);
    }

    

    /*
     * (non-Javadoc)
     *
     * @see gov.va.med.pharmacy.peps.service.common.capability.MessageDataCapability#message2ReportData(java.util.Date,
     * gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile)
     */
    @Override

    public void message2ReportData(Date startDTM, NdfUpdateProcessFile ndfUpdateFile) throws IOException {
            try {
                   	
        	List<String> result = message2ReportData(startDTM);
        	 
        	MessageFormatter formatter = new MessageFormatter(new Date(), result);
        	 
        	List<String> formattedResult =  formatter.apply(NDFUpdateFileMapping.MESSAGE2);
        	
        	String formattedString = formattedResult.stream().collect(Collectors.joining(NEWLINE));
        	
            ndfUpdateFile.putNextRow(formattedString);
            
        } catch (Exception e) {
            LOG.error("error processing row for message data ");
        }

    }

    
    /**
     * Adds the strength change report to the message for changes to File 50.68 Field 2.
     * 
     * @param startDTM
     * @return list
     */
    protected List<String> strengthChangeReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> strengthChangeProducts =
                eplNdfOutgoingDifferencesDao.messageGenericChange(VistaFileNumber.FIFTY_68.getFileNumber(), "2", startDTM);
        if (strengthChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_STRENGTH_TEXT.toString()));
        for (Object[] strengthChangeProduct : strengthChangeProducts) {
            StrengthChangeProductMapper mapper = new StrengthChangeProductMapper(strengthChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(OLD + mapper.getOldValue());
            result.add(NEW + mapper.getNewValue());
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class StrengthChangeProductMapper {
        //These index values must match the order of fields in the messageGenericChangeReport query in the DAO 
        final int NEW_VALUE = 0;
        final int OLD_VALUE = 1;
        final int PRODUCT_NAME = 2;
        final int CMOP_ID = 3;
        Object[] data;
        StrengthChangeProductMapper(Object[] row) {
            this.data = row;
        }
        String getOldValue() {
            return (String) data[OLD_VALUE];
        }
        String getNewValue() {
            return (String) data[NEW_VALUE];
        }
        String getProductName() {
            return (String) data[PRODUCT_NAME];
        }
        String getCmopId() {
            return (String) data[CMOP_ID];
        }
    }

        
    /**
     * Adds the override dosage form checks report to the message for changes to File 50.68 Field 31.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> overrideDFChckExclReport(Date startDTM) {
    	
        List<String> result = new ArrayList<String>();
        List<Object[]> overrideDFCheckkExcls =
                eplNdfOutgoingDifferencesDao.findOverrideDFChkExcl(VistaFileNumber.FIFTY_68.getFileNumber(), "31", startDTM);
        if (overrideDFCheckkExcls.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_OVERRIDEDOSECHECK_TEXT.toString()));
        for (Object[] overrideDFCheckkExcl : overrideDFCheckkExcls) {
            OverrideDfCheckExclsMapper mapper = new OverrideDfCheckExclsMapper(overrideDFCheckkExcl);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(getChangeString(mapper.getNewValue()));
            result.add(EMPTYLINE);
        }
      
        return result;
    }

    static class OverrideDfCheckExclsMapper {
        Object[] data;
        OverrideDfCheckExclsMapper(Object[] row) {
            this.data = row;
        }
        String getNewValue() {
            return (String) data[0];
        }
        String getProductName() {
            return (String) data[1];
        }
        String getCmopId() {
            return (String) data[2];
        }
    }

   
    /**
     * Adds the National Formulary Change Report to the message for changes to 50.68 Field 17.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> nationalFormularyChangeReport(
            Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> activeNationalFormularyProds =
                eplNdfOutgoingDifferencesDao.findActiveNatFormularyProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "17",
                        startDTM);
        List<Object[]> inactiveNationalFormularyProds =
                eplNdfOutgoingDifferencesDao.findInactiveNatFormularyProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "17",
                        startDTM);

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_NATIONALFORMULARY_TEXT.toString()));

        result.add(SPACE + FORMULARY);
        if (!activeNationalFormularyProds.isEmpty()) {
            for (Object[] activeNationalFormularyProd : activeNationalFormularyProds) {
                NationalFormularyProdMapper mapper = new NationalFormularyProdMapper(activeNationalFormularyProd);
                result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
                result.add(EMPTYLINE);
            }
        } else {
            result.add(doubleIndent(NONE));
            result.add(EMPTYLINE);
        }

        result.add(SPACE + NONFORMULARY);
        if (!inactiveNationalFormularyProds.isEmpty()) {
            for (Object[] inactiveNationalFormularyProd : inactiveNationalFormularyProds) {
                NationalFormularyProdMapper mapper = new NationalFormularyProdMapper(inactiveNationalFormularyProd);
                result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
                result.add(EMPTYLINE);
            }
        } else {
            result.add(doubleIndent(NONE));
            result.add(EMPTYLINE);
        }

        return result;
    }

    static class NationalFormularyProdMapper {
        Object[] data;
        NationalFormularyProdMapper(Object[] row) {
            this.data = row;
        }
        String getNewValue() {
            return (String) data[0];
        }
        String getProductName() {
            return (String) data[1];
        }
        String getCmopId() {
            return (String) data[2];
        }
    }

    
    /**
     * Adds the inactivated CMOP report to the message for changes to File 50.68 Field 7.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> inactivatedCMOPReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> inactiveCMOPProds = eplNdfOutgoingDifferencesDao
                .findInactivatedCMOPProducts(
                        VistaFileNumber.FIFTY_68.getFileNumber(), "7", startDTM);
        if (inactiveCMOPProds.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_INACTIVATEDCMOP_TEXT
                .toString()));
        for (Object[] inactiveCMOPProd : inactiveCMOPProds) {
            result.add(formatPairWithSquareBraces(
                    (String) inactiveCMOPProd[1],
                    (String) inactiveCMOPProd[2]));
            result.add(EMPTYLINE);
        }
        return result;
    }

    
    
    /**
     * Adds the added product report to the message for changes to File 50.68 Field .01.
     * 
     * @param startDTM
     * @return
     */
    protected List<String> addedProductReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> addedProds =
                eplNdfOutgoingDifferencesDao.findAddedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), ".01", startDTM);
        boolean initialData = true;

        if ((addedProds != null) && !addedProds.isEmpty()) {
            result.addAll(processMessageText_2(NationalSetting.MESSAGE_ADDEDPRODUCT_TEXT.toString()));

            String processedParent = "None";
            int maxDataLimit = 4;
            int currentDataIndex = 0;

            for (Object[] addedProd : addedProds) {
                AddedProductMapper mapper = new AddedProductMapper(addedProd);
                String currentParent = mapper.getProductName();
                if (processedParent.equalsIgnoreCase(currentParent)) {
                    if (currentDataIndex < maxDataLimit) {
                        if ((mapper.getNdcNumber() != null) && !mapper.getNdcNumber().isEmpty()) {
                            result.add(SPACER + mapper.getNdcNumber());
                            currentDataIndex++;
                        }
                    } else {
                        if ((mapper.getNdcNumber() != null) && !mapper.getNdcNumber().isEmpty()) {
                            currentDataIndex = 0;
                            result.add(EMPTYLINE);
                            result.add(doubleIndent(mapper.getNdcNumber()));
                            currentDataIndex++;
                        }
                    }
                    processedParent = currentParent;
                } else {
                    currentDataIndex = 0;
//                    if (!initialData) result.add(EMPTY_STRING); //
                    result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));

                    String data2 = DISPENSE + mapper.getDispenseUnitName() + ENDROUNDBRAC;
                    result.add(doubleIndent(data2));

                    if ((mapper.getNdcNumber() != null) && !mapper.getNdcNumber().isEmpty()) {
                        result.add(doubleIndent(mapper.getNdcNumber()));
//                        result.add(EMPTY_STRING); //
                        currentDataIndex++;
                    }

                    processedParent = currentParent;
                    initialData = false;
                }
                result.add(EMPTYLINE);
            }
        }

        return result;
    }

    static class AddedProductMapper {
        Object[] data;
        AddedProductMapper(Object[] row) {
            this.data = row;
        }
        String getNewValue() {
            return (String) data[0];
        }
        String getProductName() {
            return (String) data[1];
        }
        String getCmopId() {
            return (String) data[2];
        }
        String getDispenseUnitName() {
            return (String) data[3];
        }
        String getNdcNumber() {
            return (String) data[4];
        }
    }

    static class AddedProductMapper2 {
        Object[] data;
        AddedProductMapper2(Object[] row) {
            this.data = row;
        }
        String getNewValue() {
            return (String) data[0];
        }
        String getProductName() {
            return (String) data[1];
        }
        String getCmopId() {
            return (String) data[2];
        }
        String getClassification() {
            return (String) data[3];
        }
        // just an alias..
        String getCode() {
            return getNewValue();
        }
    }

    
    
    /**
     * Possible dosage report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> possibleDosageReport(Date startDTM) {
       
    	List<String> result = new ArrayList<String>();
        List<Object[]> yesPossibleDosageProds =
                eplNdfOutgoingDifferencesDao.findYesPossibleDosageProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "40",
                        startDTM);
        List<Object[]> noPossibleDosageProds =
                eplNdfOutgoingDifferencesDao.findNoPossibleDosageProducts(VistaFileNumber.FIFTY_68.getFileNumber(), startDTM);

        if (yesPossibleDosageProds.isEmpty() && noPossibleDosageProds.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_POSSIBLEDOSAGE_TEXT.toString()));

        for (Object[] yesPossibleDosageProd : yesPossibleDosageProds) {
            YesPossibleDosageMapper mapper = new YesPossibleDosageMapper(yesPossibleDosageProd);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(doubleIndent(AUTOCREATEQUEST + getChangeStringPossibleDosage(mapper.getNewValue())));
            result.add(EMPTYLINE);
        }

        for (Object[] noPossibleDosageProd : noPossibleDosageProds) {
            NoPossibleDosageMapper mapper = new NoPossibleDosageMapper(noPossibleDosageProd);
            if ("41".equals(mapper.getVistaFieldNumber())) {
                result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
                result.add(doubleIndent(AUTOCREATENEG));
                result.add(doubleIndent(POSSIBLEDOSAGESTMT + getChangeStringDosageAutoCreate(mapper.getNewValue())));
            } else if ("42".equals(mapper.getVistaFieldNumber())) {
                result.add(doubleIndent(PACKAGE + getChangeStringPackage(mapper.getNewValue())));
            }
            result.add(EMPTYLINE);
        }

        return result;
    }

    static class YesPossibleDosageMapper {
        Object[] data;
        YesPossibleDosageMapper(Object[] row) {
            this.data = row;
        }
        String getNewValue() {
            return (String) data[0];
        }
        String getProductName() {
            return (String) data[1];
        }
        String getCmopId() {
            return (String) data[2];
        }
    }

    static class NoPossibleDosageMapper {
        Object[] data;
        NoPossibleDosageMapper(Object[] row) {
            this.data = row;
        }
        String getVistaFieldNumber() {
            return (String) data[0];
        }
        String getNewValue() {
            return (String) data[1];
        }
        String getVistaIen() {
            return (String) data[2];
        }
        String getProductName() {
            return (String) data[3];
        }
        String getCmopId() {
            return (String) data[4];
        }
    }

   

    /**
     * Prints the name change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> printNameChangeReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> printNameChangeProducts =
                eplNdfOutgoingDifferencesDao.messageGenericChange(VistaFileNumber.FIFTY_68.getFileNumber(), "5", startDTM);
        if (printNameChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_PRINTNAMECHANGE_TEXT.toString()));
        for (Object[] printNameChangeProduct : printNameChangeProducts) {
            ChangeProductMapper mapper = new ChangeProductMapper(printNameChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(OLD + mapper.getOldValue());
            result.add(NEW + mapper.getNewValue());
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class ChangeProductMapper {
        Object[] data;
        ChangeProductMapper(Object[] row) {
            this.data = row;
        }
        String getNewValue() {
            return (String) data[0];
        }
        String getOldValue() {
            return (String) data[1];
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
    }

    
    
    /**
     * Drug units change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> drugUnitsChangeReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> unitsChangeProducts =
                eplNdfOutgoingDifferencesDao.drugUnitsChange(VistaFileNumber.FIFTY_68.getFileNumber(), "3", startDTM);
        if (unitsChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_DRUGUNITCHANGE_TEXT.toString()));
        for (Object[] unitsChangeProduct : unitsChangeProducts) {
            GenNameChangeProductMapper mapper = new GenNameChangeProductMapper(unitsChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(OLD + mapper.getOldName());
            result.add(NEW + mapper.getNewName());
            result.add(EMPTYLINE);
        }
        return result;
    }

    
    /**
     * Dispense units change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> dispenseUnitsChangeReport(Date startDTM) {
        
    	List<String> result = new ArrayList<String>();
        List<Object[]> unitsChangeProducts =
                eplNdfOutgoingDifferencesDao.dispenseUnitsChange(VistaFileNumber.FIFTY_68.getFileNumber(), "8", startDTM);
        if (unitsChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_DISPENSEUNITCHANGE_TEXT.toString()));
        for (Object[] unitsChangeProduct : unitsChangeProducts) {
            GenNameChangeProductMapper mapper = new GenNameChangeProductMapper(unitsChangeProduct);
            result.add(doubleIndent(mapper.getProductName()));
            result.add(OLD + mapper.getOldName());
            result.add(NEW + mapper.getNewName());
            result.add(EMPTYLINE);
        }
        return result;
    }

    
    /**
     * Class change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> classChangeReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> classChangeProducts =
                eplNdfOutgoingDifferencesDao.classChange(VistaFileNumber.FIFTY_68.getFileNumber(), "15", startDTM);
        if (classChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_VACLASSCHANGE_TEXT.toString()));
        for (Object[] classChangeProduct : classChangeProducts) {
            ClassChangeProductMapper mapper = new ClassChangeProductMapper(classChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(OLD + mapper.getOldClass());
            result.add(NEW + mapper.getNewClass());
            result.add(EMPTYLINE);
        }
        return result;
    }

    /**
     * Class change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> newClassReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> newClassProducts =
                eplNdfOutgoingDifferencesDao.findNewClassProducts(VistaFileNumber.FIFTY_605.getFileNumber(), ".01", startDTM);
        if (newClassProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_NEWVADRUGCLASS_TEXT.toString()));
        for (Object[] newClassProduct : newClassProducts) {
            AddedProductMapper2 mapper = new AddedProductMapper2(newClassProduct);
            result.add(formatPairWithSquareBraces(mapper.getCode(), mapper.getClassification()));
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class ClassChangeProductMapper {
        Object[] data;
        ClassChangeProductMapper(Object[] row) {
            this.data = row;
        }
        String getOldClass() {
            String result = (String) data[5];
            if (result == null) return ONE_SPACE;
            return result;
        }
        String getNewClass() {
            String result = (String) data[4];
            if (result == null) return ONE_SPACE;
            return result;
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
    }

       
    /**
     * Cmop change report.
     * @param startDTM
     * @return
     */
    private List<String> cmopChangeReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> cmopChangeProducts =
                eplNdfOutgoingDifferencesDao.messageCMOPChange(VistaFileNumber.FIFTY_68.getFileNumber(), "6", startDTM);
        if (cmopChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_CMOPCHANGE_TEXT.toString()));
        for (Object[] cmopChangeProduct : cmopChangeProducts) {
            ChangeProductMapper mapper = new ChangeProductMapper(cmopChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(OLD + mapper.getOldValue());
            result.add(NEW + mapper.getNewValue());
            result.add(EMPTYLINE);
        }
        return result;
    }

    
    /**
     * Future inactive rematch report.
     * @return list
     */
    private List<String> futureInactiveRematchReport() {
        List<String> result = new ArrayList<String>();
        List<Object[]> futureInactivatedProducts = eplNdfOutgoingDifferencesDao.futureInactiveRematched();
        if (futureInactivatedProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_PROPOSEDINACTIVATEDWSUGG_TEXT.toString()));
        for (Object[] futureInactivatedProduct : futureInactivatedProducts) {
            FutureInactivatedProductsMapper mapper = new FutureInactivatedProductsMapper(futureInactivatedProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(doubleIndent(formatDateString(mapper.getFutureDate())));
            result.add(doubleIndent(USE));
            result.add(formatPairWithSquareBraces(mapper.getNewProductName(), mapper.getNewCmopId()));
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class FutureInactivatedProductsMapper {
        Object[] data;
        FutureInactivatedProductsMapper(Object[] row) {
            this.data = row;
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
        String getFutureDate() {
            return (String) data[4];
        }
        String getNewProductName() {
            return (String) data[6];
        }
        String getNewCmopId() {
            return (String) data[7];
        }
    }

    

    /**
     * Future inactivation report.
     * @return list
     */
    private List<String> futureInactivationReport() {
        List<String> result = new ArrayList<String>();

        List<Object[]> futureInactivatedProducts = eplNdfOutgoingDifferencesDao
                .futureInactiveChange();
        if (futureInactivatedProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_FUTUREINACTIVE_TEXT
                .toString()));
        for (Object[] futureInactivatedProduct : futureInactivatedProducts) {
            FutureInactiveChangeMapper mapper = new FutureInactiveChangeMapper(
                    futureInactivatedProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(doubleIndent(formatDateString(mapper.getFutureDate())));
            result.add(EMPTYLINE);
        }
        return result;
    }

    private String formatDateString(String date) {
        long inactivationDate = Long.parseLong(date);
        return new SimpleDateFormat("MM/dd/yyyy").format(new java.util.Date(inactivationDate));
    }

    private String doubleIndent(String data) {
        return DOUBLEDSPACE + data;
    }

    private String formatPairWithSquareBraces(String item1, String item2) {
    	
    	StringBuffer buf = new StringBuffer();
    	
    	buf.append(DOUBLEDSPACE);
    	buf.append(item1);
    	buf.append(SPACER);
    	buf.append(OPENING_SQUARE);
    	buf.append(item2);
    	buf.append(END_SQUARE);
    	
        return buf.toString();
    }
    
    private String formatWithDoubleSpace(String item1, String item2) {
    	
    	StringBuffer buf = new StringBuffer();
    	buf.append(DOUBLEDSPACE);
    	buf.append(item1);
    	buf.append(SPACER3);    	
    	buf.append(item2);
    	
        return buf.toString(); 	
    	
        
    }

    private String inSquareBraces(String item) {
        return OPENING_SQUARE + item + END_SQUARE;
    }

    static class FutureInactiveChangeMapper {
        Object[] data;
        FutureInactiveChangeMapper(Object[] row) {
            this.data = row;
        }
        String getProductName() {
            return (String) data[1];
        }
        String getCmopId() {
            return (String) data[2];
        }
        String getFutureDate() {
            return (String) data[3];
        }
    }

    
    /**
     * Schedule all change report.
     * @param startDTM
     * @return list
     */
    private List<String> scheduleAllChangeReport(Date startDTM) {
      
    	List<String> result = new ArrayList<String>();
        List<Object[]> scheduleChangeProducts =
                eplNdfOutgoingDifferencesDao.scheduleAllChange(VistaFileNumber.FIFTY_68.getFileNumber(), "19", startDTM);
        if (scheduleChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_SCHEDULECHANGEALL_TEXT.toString()));
       
        for (Object[] scheduleChangeProduct : scheduleChangeProducts) {
            ScheduleChangeProductMapper mapper = new ScheduleChangeProductMapper(scheduleChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(OLD + mapper.getOldSchedule());
            result.add(NEW + mapper.getNewSchedule());
            result.add(EMPTYLINE);
        }
      
        return result;
    }

    static class ScheduleChangeProductMapper {
        Object[] data;
        ScheduleChangeProductMapper(Object[] row) {
            this.data = row;
        }
        String getOldSchedule() {
            return (String) data[5];
        }
        String getNewSchedule() {
            return (String) data[4];
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
    }

        
    /**
     * Gen name all change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> genNameAllChangeReport(Date startDTM) {
       
    	List<String> result = new ArrayList<String>();
        List<Object[]> genNameChangeProducts =
                eplNdfOutgoingDifferencesDao.genNameAllChange(VistaFileNumber.FIFTY_68.getFileNumber(), ".05", startDTM);
        if (genNameChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_GENERICNAMECHANGE_TEXT
                .toString()));
        for (Object[] genNameChangeProduct : genNameChangeProducts) {
            GenNameChangeProductMapper mapper = new GenNameChangeProductMapper(genNameChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(OLD + mapper.getOldName());
            result.add(NEW + mapper.getNewName());
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class GenNameChangeProductMapper {
        Object[] data;
        GenNameChangeProductMapper(Object[] row) {
            this.data = row;
        }
        String getOldName() {
            String result = (String) data[5];
            if (result == null) return ONE_SPACE;
            return result;
        }
        String getNewName() {
            String result = (String) data[4];
            if (result == null) return ONE_SPACE;
            return result;
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
    }

   
    /**
     * Dosage form change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> dosageFormChangeReport(Date startDTM) {
        
    	List<String> result = new ArrayList<String>();
        List<Object[]> dosageFormChangeProducts =
                eplNdfOutgoingDifferencesDao.dosageFormChange(VistaFileNumber.FIFTY_68.getFileNumber(), "1", startDTM);
        if (dosageFormChangeProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_DOSAGEFORMCHANGE_TEXT.toString()));
        
        for (Object[] dosageFormChangeProduct : dosageFormChangeProducts) {
            GenNameChangeProductMapper mapper = new GenNameChangeProductMapper(dosageFormChangeProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(OLD + mapper.getOldName());
            result.add(NEW + mapper.getNewName());
            result.add(EMPTYLINE);
        }
       
        return result;
    }

    
    /**
     * Inactivated rematch report sugg.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> inactivatedRematchReportSugg(
            Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> inactiveProducts =
                eplNdfOutgoingDifferencesDao.findInactiveRematchReportSugg(VistaFileNumber.FIFTY_68.getFileNumber(), "21",
                        startDTM);
        if (inactiveProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_INACTIVATEDWSUGG_TEXT.toString()));
        for (Object[] inactiveProduct : inactiveProducts) {
            InactiveProductMapper mapper = new InactiveProductMapper(inactiveProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(doubleIndent(USE));
            result.add(formatPairWithSquareBraces(mapper.getNewProductName(), mapper.getNewCmopId()));
            result.add(EMPTYLINE);
        }
        return result;
    }

    

    /**
     * Inactivated rematch report.
     * @param startDTM
     * @return list
     */
    private List<String> inactivatedRematchReport(
            Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> inactiveProducts =
                eplNdfOutgoingDifferencesDao
                        .findInactiveRematchReport(VistaFileNumber.FIFTY_68.getFileNumber(), "21", startDTM);
        if (inactiveProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_INACTIVATEDREMATCHED_TEXT.toString()));
        for (Object[] inactiveProduct : inactiveProducts) {
            InactiveProductMapper mapper = new InactiveProductMapper(inactiveProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(doubleIndent(USE));
            result.add(formatPairWithSquareBraces(mapper.getNewProductName(),
                    mapper.getNewCmopId()));
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class InactiveProductMapper {
        Object[] data;
        InactiveProductMapper(Object[] row) {
            this.data = row;
        }
        String getProductName() {
            return (String) data[3];
        }
        String getCmopId() {
            return (String) data[4];
        }
        String getNewCmopId() {
            return (String) data[7];
        }
        String getNewProductName() {
            return (String) data[6];
        }
    }

    
    /**
     * Other rematched report.
     * @param startDTM
     * @return list
     */
    private List<String> otherRematchedReport(Date startDTM) {
       
    	List<String> result = new ArrayList<String>();
        List<Object[]> rematchProducts = eplRematchSuggestionDao.findRematchAuto(startDTM);
        if ((rematchProducts != null) && rematchProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_OTHERREMATCHED_TEXT.toString()));
      
        for (Object[] rematchProduct : rematchProducts) {
            RematchProductMapper mapper = new RematchProductMapper(rematchProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(), mapper.getCmopId()));
            result.add(doubleIndent(USE));
            result.add(formatPairWithSquareBraces(mapper.getRematchedProductName(), mapper.getRematchedCmopId()));
            result.add(EMPTYLINE);
        }
       
        return result;
    }

    static class RematchProductMapper {
        Object[] data;
        RematchProductMapper(Object[] row) {
            this.data = row;
        }
        String getRematchedProductName() {
            return (String) data[4];
        }
        String getRematchedCmopId() {
            return (String) data[5];
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
    }

   
    /**
     * Other rematch sugg report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> otherRematchSuggReport(Date startDTM) {
        
    	List<String> result = new ArrayList<String>();
        List<Object[]> rematchProducts = eplRematchSuggestionDao.findRematchSugg(startDTM);
        if (rematchProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_OTHERREMATCHSUGG_TEXT.toString()));
       
        for (Object[] rematchProduct : rematchProducts) {
            RematchProductMapper mapper = new RematchProductMapper(rematchProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(doubleIndent(USE));
            result.add(formatPairWithSquareBraces(mapper.getRematchedProductName(),
                    mapper.getRematchedCmopId()));
            result.add(EMPTYLINE);
        }
       
        return result;
    }

    /**
     * Other rematch no sugg report.
     *
     * @param counter
     *            the counter
     * @param message
     *            the message
     * @param startDTM
     *            the start dtm
     * @return the int
     */
    @Deprecated
    public int otherRematchNoSuggReport(int counter, StringBuffer message, Date startDTM) {
        // final int OLD_IEN = 0;
        // final int REMATCHED_IEN = 1;
        final int PRODUCT_NAME = 2;
        final int CMOP_ID = 3;

        List<Object[]> rematchProducts = eplRematchSuggestionDao.findRematchNoSugg(startDTM);

        if (!rematchProducts.isEmpty()) {
            counter = processMessageText(counter, message, NationalSetting.MESSAGE_OTHERNOREMATCH_TEXT.toString());
            counter++;
            NdfUpdateMessageUtility.formatHeaderRowForMESSAGEStart(counter, message);

            for (Object[] rematchProduct : rematchProducts) {

                counter++;
                NdfUpdateMessageUtility.formatParallelDataSqr((String) rematchProduct[PRODUCT_NAME],
                        (String) rematchProduct[CMOP_ID], message);
                NdfUpdateMessageUtility.formatHeaderRowForMESSAGEEnd(counter, message);
            }
        }

        return counter;

    }

    /**
     * Other rematch no sugg report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> otherRematchNoSuggReport(Date startDTM) {
        
    	List<String> result = new ArrayList<String>();
        List<Object[]> rematchProducts = eplRematchSuggestionDao.findRematchNoSugg(startDTM);
        
        if (rematchProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_OTHERNOREMATCH_TEXT.toString()));
        
        for (Object[] rematchProduct : rematchProducts) {
            RematchProductMapper mapper = new RematchProductMapper(rematchProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(EMPTYLINE);
        }
        
        return result;
    }

   

    /**
     * Inactivated products report.
     * @param startDTM
     * @return list
     */
    private List<String> inactivatedProductsReport(
            Date startDTM) {
        List<String> result = new ArrayList<String>();
        List<Object[]> inactiveProducts =
                eplNdfOutgoingDifferencesDao.findInactivatedProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "21", startDTM);
        if (inactiveProducts.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_INACTIVATEDPRODUCT_TEXT.toString()));
        for (Object[] inactiveProduct : inactiveProducts) {
            InactiveProductMapper mapper = new InactiveProductMapper(inactiveProduct);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(EMPTYLINE);
        }
        return result;
    }


        
    /**
     * Reactivated product report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> reactivatedProductReport(Date startDTM) {
        List<String> result = new ArrayList<String>();
        final int PRODUCT_NAME = 2;
        final int CMOP_ID = 3;
        List<Object[]> reactivatedProds = eplNdfOutgoingDifferencesDao
                .findReactivatedProducts(
                        VistaFileNumber.FIFTY_68.getFileNumber(), "21",
                        startDTM);
        if (reactivatedProds.isEmpty()) return result;

        result.addAll(processMessageText_2(NationalSetting.MESSAGE_REACTIVATEDPRODUCT_TEXT
                .toString()));
        for (Object[] reactivatedProd : reactivatedProds) {
            ReactivatedProductMapper mapper = new ReactivatedProductMapper(
                    reactivatedProd);
            result.add(formatPairWithSquareBraces(mapper.getProductName(),
                    mapper.getCmopId()));
            result.add(EMPTYLINE);
        }
        return result;
    }

    static class ReactivatedProductMapper {
        Object[] data;
        ReactivatedProductMapper(Object[] row) {
            this.data = row;
        }
        String getProductName() {
            return (String) data[2];
        }
        String getCmopId() {
            return (String) data[3];
        }
    }

       
    /**
     *  Med guide change report.
     *  
     * @param startDTM
     * @return list
     */
    private List<String> medGuideChangeReport( Date startDTM) {

    	 List<String> result = new ArrayList<String>();
    	 
        List<Object[]> addedMedGuideProds =
                eplNdfOutgoingDifferencesDao.findAddedMedGuideProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "100",
                        startDTM);
        List<Object[]> editedMedGuideProds =
                eplNdfOutgoingDifferencesDao.findEditedMedGuideProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "100",
                        startDTM);
        List<Object[]> deletedMedGuideProds =
                eplNdfOutgoingDifferencesDao.findDeletedMedGuideProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "100",
                        startDTM);

        if (((addedMedGuideProds != null) && !addedMedGuideProds.isEmpty())
                || ((editedMedGuideProds != null) && !editedMedGuideProds.isEmpty())
                || ((deletedMedGuideProds != null) && !deletedMedGuideProds.isEmpty())) {

        	result.addAll(processMessageText_2( NationalSetting.MESSAGE2_MEDGUIDE_TEXT.toString()) );

            if (!addedMedGuideProds.isEmpty()) {

                result.add(SPACE + ADD);

               

                for (Object[] addedMedGuideProd : addedMedGuideProds) {
                    result.add(formatPairWithSquareBraces((String) addedMedGuideProd[1], (String) addedMedGuideProd[2]));
                    result.add(EMPTYLINE);                    
                }
            }

            if (!editedMedGuideProds.isEmpty()) {
            	
            	 result.add(SPACE + EDIT);

                for (Object[] editedMedGuideProd : editedMedGuideProds) {                    
                    result.add(formatPairWithSquareBraces((String) editedMedGuideProd[1], (String) editedMedGuideProd[2]));
                    result.add(EMPTYLINE);
                }
            }

            if (!deletedMedGuideProds.isEmpty()) {
               
            	result.add(SPACE + DELETE);
            	
                for (Object[] deletedMedGuideProd : deletedMedGuideProds) {
                    result.add(formatPairWithSquareBraces((String) deletedMedGuideProd[1], (String) deletedMedGuideProd[2]));
                    result.add(EMPTYLINE);    
                }
            }
        }
        return result;
    }

        
    
    /**
     * Ddi change report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> ddiChangeReport(Date startDTM) {

        List<String> result = new ArrayList<String>();
    	
    	List<Object[]> addedDDIs =
                eplNdfOutgoingDifferencesDao.findAddedDDIs(VistaFileNumber.FIFTY_SIX.getFileNumber(), ".01", startDTM);
        List<Object[]> reactivatedDDIs =
                eplNdfOutgoingDifferencesDao.findReactivatedDDIs(VistaFileNumber.FIFTY_SIX.getFileNumber(), "7", startDTM);
        List<Object[]> editedDDIs =
                eplNdfOutgoingDifferencesDao.findEditedDDIs(VistaFileNumber.FIFTY_SIX.getFileNumber(), "3", startDTM);
        List<Object[]> inactivatedDDIs =
                eplNdfOutgoingDifferencesDao.findInactivatedDDIs(VistaFileNumber.FIFTY_SIX.getFileNumber(), "7", startDTM);

       
        result.addAll(processMessageText_2(NationalSetting.MESSAGE2_DDI_TEXT.toString()));
        
        
        result.add(SPACE + ADDINT);


        if (((addedDDIs != null) && !addedDDIs.isEmpty()) || ((reactivatedDDIs != null) && !reactivatedDDIs.isEmpty())) {

            if (!addedDDIs.isEmpty()) {

                for (Object[] addedDDI : addedDDIs) {

                    result.add(formatWithDoubleSpace( (String) addedDDI[0], (String) addedDDI[1])  );
                    result.add(EMPTYLINE);
                }
            }

            if (!reactivatedDDIs.isEmpty()) {
                for (Object[] reactivatedDDI : reactivatedDDIs) {

                    result.add(formatWithDoubleSpace( (String) reactivatedDDI[3], (String) reactivatedDDI[4])  );
                    result.add(EMPTYLINE);                    
                }
            }

        } else {

        	result.add(doubleIndent(NONE));
        	result.add(EMPTYLINE);
        }


        result.add(SPACE + EDITINT );

        if ((editedDDIs != null) && !editedDDIs.isEmpty()) {

            for (Object[] editedDDI : editedDDIs) {
                result.add(formatWithDoubleSpace( (String) editedDDI[1], (String) editedDDI[2] ));
                result.add(EMPTYLINE);
            }
        } else {
           
            result.add(doubleIndent(NONE));
            result.add(EMPTYLINE);
        }

        result.add(SPACE + INACTIVATEINT);

        if (((inactivatedDDIs != null) && !inactivatedDDIs.isEmpty())) {
        	for (Object[] inactivatedDDI : inactivatedDDIs) {               
                result.add( formatWithDoubleSpace( (String) inactivatedDDI[3],  (String) inactivatedDDI[4] )  );
                result.add(EMPTYLINE);                
            }
        } else {
        	 result.add(doubleIndent(NONE));
        	 result.add(EMPTYLINE);        	 
        }

        return result;
    }

        
    /**
     * Excluded ddi report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> excludedDDIReport(Date startDTM) {

    	 List<String> result = new ArrayList<String>();
    	 
        List<Object[]> excludedDDIProds =
                eplNdfOutgoingDifferencesDao.findExcludedDDIProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "23", startDTM);

        if (((excludedDDIProds != null) && !excludedDDIProds.isEmpty())) {

            result.addAll(processMessageText_2(NationalSetting.MESSAGE2_EXCLUEDEDDDI_TEXT.toString()));            

            for (Object[] excludedDDIProd : excludedDDIProds) {

                result.add(formatPairWithSquareBraces((String) excludedDDIProd[1], (String) excludedDDIProd[2]));
                result.add(EMPTYLINE);
            }
        }

        return result;
    }

       
    /**
     * Previously excluded ddi report.
     * 
     * @param startDTM
     * @return list
     */
    private List<String> previouslyExcludedDDIReport(Date startDTM) {

    	List<String> result = new ArrayList<String>();
    	
        List<Object[]> prevExcludedDDIProds =
                eplNdfOutgoingDifferencesDao.findPrevExcludedDDIProducts(VistaFileNumber.FIFTY_68.getFileNumber(), "23",
                        startDTM);

        if (((prevExcludedDDIProds != null) && !prevExcludedDDIProds.isEmpty())) {

            result.addAll(processMessageText_2(NationalSetting.MESSAGE2_PREVEXCLUEDEDDDI_TEXT.toString()));
            
            for (Object[] prevExcludedDDIProd : prevExcludedDDIProds) {
                result.add(formatPairWithSquareBraces((String) prevExcludedDDIProd[1], (String)  prevExcludedDDIProd[2]));
                result.add(EMPTYLINE);
            }
        }

        return result;
    }

    /**
     * Process message text.
     *
     * @param counter
     *            the counter
     * @param message
     *            the message
     * @param textType
     *            the text type
     * @return counter
     */
    public int processMessageText(int counter, StringBuffer message, String textType) {

        NationalSettingVo returnedNationalSetting = nationalSettingDomainCapability.retrieve(textType);

        String messageText =
                returnedNationalSetting.getStringValue() != null ? returnedNationalSetting.getStringValue() : "missing text";

        String[] splits = messageText.split("\\|");

        if (splits.length > 0) {
            counter++;
            NdfUpdateMessageUtility.formatHeaderRowForMESSAGEStart(counter, message);
            for (String text : splits) {
                counter++;
                NdfUpdateMessageUtility.formatTitleDataRowForMESSAGE(text, message);
                NdfUpdateMessageUtility.formatHeaderRowForMESSAGEEnd(counter, message);
            }

        }

        return counter;

    }

    /**
     * Adds a blank line following basic processing of vertical bar-delimited
     * text retrieved from database--note the vertical bar delimited text is a
     * clear sign of UI implementation creeping inappropriately into the
     * backend as the responsibility to format the text (line size) rightly
     * rests on a presentation layer component.
     * 
     * @param textType
     * @return List<String> parsed out of national setting domain db value
     *         for textType, plus one blank line at the end of the list.
     */
    // method name is terrible.. what are we really doing here??
    private List<String> processMessageText_2(String textType) {
        List<String> result = new ArrayList<String>();
        List<String> processedLines = doBasicProcessingForMessage(textType);
        if (processedLines.size() > 0) {
            result.add(EMPTYLINE);
            result.addAll(processedLines);
            result.add(EMPTYLINE);
        }
        return result;
    }

    /**
     * Does a lookup of textType in national setting domain capability and 
     * returns a list of strings parsed out by vertical bar delimiters or a
     * list containing only the string "missing text" if national setting value
     * not populated in database.
     * 
     * @param textType
     * @return List<String> parsed out of national setting domain db value
     *         for textType.
     */
    private List<String> doBasicProcessingForMessage(String textType) {
        NationalSettingVo returnedNationalSetting = nationalSettingDomainCapability
                .retrieve(textType);
        String messageText = returnedNationalSetting.getStringValue() != null ? returnedNationalSetting
                .getStringValue() : "missing text";
        return Arrays.asList(messageText.split("\\|"));
    }

    /**
     * process default TEXT data.
     *
     * @param counter
     *            the counter
     * @param message
     *            the message
     * @param textType
     *            the text type
     * @return int counter
     */
    public int processMessage2Text(int counter, StringBuffer message, String textType) {

        NationalSettingVo returnedNationalSetting = nationalSettingDomainCapability.retrieve(textType);

        String messageText =
                returnedNationalSetting.getStringValue() != null ? returnedNationalSetting.getStringValue() : "missing text";

        String[] splits = messageText.split("\\|");

        if (splits.length > 0) {
            counter++;
            NdfUpdateMessageUtility.formatHeaderRowForMESSAGE2Start(counter, message);
            for (String text : splits) {
                counter++;
                NdfUpdateMessageUtility.formatTitleDataRowForMESSAGE(text, message);
                NdfUpdateMessageUtility.formatHeaderRowForMESSAGE2End(counter, message);

            }

        }

        return counter;
    }

    /**
     * Gets the change string.
     *
     * @param strConversion
     *            the str conversion
     * @return the change string
     */
    private String getChangeString(String strConversion) {
        String actionString = EMPTYLINE;
        if (strConversion.equals("1")) {
            actionString = "  changed from NO to YES";
        } else {
            actionString = "  changed from YES to NO";
        }
        return actionString;

    }

    /**
     * Gets the change string possible dosage.
     *
     * @param strConversion
     *            the str conversion
     * @return the change string possible dosage
     */
    private String getChangeStringPossibleDosage(String strConversion) {
        String actionString = EMPTYLINE;
        if (strConversion.equals("Y")) {
            actionString = "YES";
        } else {
            actionString = "NO";
        }
        return actionString;
    }

    /**
     * Gets the change string dosage auto create.
     *
     * @param strConversion
     *            the str conversion
     * @return the change string dosage auto create
     */
    private String getChangeStringDosageAutoCreate(String strConversion) {
        String actionString = EMPTYLINE;
        if (strConversion.equals("N")) {
            actionString = "NO POSSIBLE DOSAGES";
        } else if (strConversion.equals("B")) {
            actionString = "1x AND 2x POSSIBLE DOSAGES";
        } else if (strConversion.equals("O")) {
            actionString = "1x POSSIBLE DOSAGES";
        }
        return actionString;
    }

    /**
     * Gets the change string package.
     *
     * @param strConversion
     *            the str conversion
     * @return the change string package
     */
    private String getChangeStringPackage(String strConversion) {
        String actionString = EMPTYLINE;
        if (strConversion.equals("I")) {
            actionString = "Inpatient";
        } else if (strConversion.equals("O")) {
            actionString = "Outpatient";

        } else if (strConversion.equals("IO")) {
            actionString = "Both";
        }
        return actionString;

    }

    /**
     * Gets the epl ndf outgoing differences dao.
     *
     * @return the epl ndf outgoing differences dao
     */
    public EplNdfOutgoingDifferencesDao getEplNdfOutgoingDifferencesDao() {
        return eplNdfOutgoingDifferencesDao;
    }

    /**
     * Sets the epl ndf outgoing differences dao.
     *
     * @param eplNdfOutgoingDifferencesDao
     *            the new epl ndf outgoing differences dao
     */
    public void setEplNdfOutgoingDifferencesDao(EplNdfOutgoingDifferencesDao eplNdfOutgoingDifferencesDao) {
        this.eplNdfOutgoingDifferencesDao = eplNdfOutgoingDifferencesDao;
    }

    /**
     * Gets the national setting domain capability.
     *
     * @return the national setting domain capability
     */
    public NationalSettingDomainCapability getNationalSettingDomainCapability() {
        return nationalSettingDomainCapability;
    }

    /**
     * Sets the national setting domain capability.
     *
     * @param nationalSettingDomainCapability
     *            the new national setting domain capability
     */
    public void setNationalSettingDomainCapability(NationalSettingDomainCapability nationalSettingDomainCapability) {
        this.nationalSettingDomainCapability = nationalSettingDomainCapability;
    }

    /**
     * Gets the epl rematch suggestion dao.
     *
     * @return the epl rematch suggestion dao
     */
    public EplRematchSuggestionDao getEplRematchSuggestionDao() {
        return eplRematchSuggestionDao;
    }

    /**
     * Sets the epl rematch suggestion dao.
     *
     * @param eplRematchSuggestionDao
     *            the new epl rematch suggestion dao
     */
    public void setEplRematchSuggestionDao(EplRematchSuggestionDao eplRematchSuggestionDao) {
        this.eplRematchSuggestionDao = eplRematchSuggestionDao;
    }

    
    
    private  List<String> message2ReportData(Date startDTM) throws Exception {
        
        List<String> result = new ArrayList<String>();

        try {
        	
        	result.addAll(medGuideChangeReport (startDTM) );
        	
        	result.addAll(ddiChangeReport(startDTM) );
            
        	result.addAll(excludedDDIReport(startDTM) );
            
        	result.addAll(previouslyExcludedDDIReport(startDTM) );
            
        } catch (Exception e) {
            LOG.error("error processing row for message2 data.");
            return result;
        }

        return result;
    }
    
    @Override
    public List<String>  generateUpdateFilePreviewReport(Date startDTM) {
    	
    	// message report data
    	List<String> reportList = getReportData(startDTM);
    	
    	try {
    		// append message2 report data.
			reportList.addAll(message2ReportData(startDTM));
			
		} catch (Exception e) {
			
			LOG.error("error processing row for message data for update file preview report in message2ReportDataForUpdateFilePreview ");
		}
      
        return reportList;
    }
}
