/**
 * Source file created in 2011 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.service.common.capability.impl;


import gov.va.med.pharmacy.peps.common.vo.EntityType;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.NationalSetting;
import gov.va.med.pharmacy.peps.common.vo.NationalSettingVo;
import gov.va.med.pharmacy.peps.common.vo.NdfSynchQueueVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.common.vo.diff.Difference;
import gov.va.med.pharmacy.peps.domain.common.capability.NationalSettingDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.capability.NdfSynchQueueDomainCapability;
import gov.va.med.pharmacy.peps.external.common.preencapsulation.outbound.capability.VistaFileSynchCapability;
import gov.va.med.pharmacy.peps.external.common.vo.outbound.common.ItemAction;
import gov.va.med.pharmacy.peps.service.common.capability.FdbSchedulerProcessCapability;
import gov.va.med.pharmacy.peps.service.common.capability.ManagedItemCapability;
import gov.va.med.pharmacy.peps.service.common.scheduler.EplNationalInfo;
import gov.va.med.pharmacy.peps.service.common.scheduler.FdbJobNames;
import gov.va.med.pharmacy.peps.service.common.scheduler.FdbSchedulerControlBean;
import gov.va.med.pharmacy.peps.service.common.scheduler.JobCommands;
import gov.va.med.pharmacy.peps.service.common.scheduler.JobStatus;
import gov.va.med.pharmacy.peps.service.common.scheduler.JobStatusInfo;
import gov.va.med.pharmacy.peps.service.common.scheduler.ProcessStatus;
import gov.va.med.pharmacy.peps.service.common.scheduler.SchedulerState;
import gov.va.med.pharmacy.peps.service.common.scheduler.SchedulerStatus;
import gov.va.med.pharmacy.peps.service.common.utility.ManagedItemCapabilityFactory;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.Logger; 
import org.apache.logging.log4j.LogManager;
import org.quartz.JobKey;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.quartz.impl.JobDetailImpl;
import org.quartz.impl.StdScheduler;
import org.quartz.impl.matchers.GroupMatcher;
import org.quartz.impl.triggers.CronTriggerImpl;
import org.springframework.scheduling.quartz.SpringBeanJobFactory;
import org.springframework.transaction.annotation.Transactional;


/**
 * 
 * FdbSchedulerProcessCapabiltyImpl
 * 
 */
public class FdbSchedulerProcessCapabiltyImpl implements FdbSchedulerProcessCapability {

    private static final Logger LOG = LogManager.getLogger(FdbSchedulerProcessCapabiltyImpl.class);
    
    private static final String FDBGROUP = "FDBGroup";
    
    @Resource
    private JobDetailImpl fdbAddJob;
    @Resource
    private JobDetailImpl fdbUpdateJob;
    @Resource
    private JobDetailImpl stsJob;
    @Resource
    private JobDetailImpl fssJob;
    @Resource
    private JobDetailImpl inactivationCheckJob;
    @Resource
    private JobDetailImpl ndfUpdateJob;
    @Resource
    private JobDetailImpl rxNormUpdateJob;

    @Resource
    private SpringBeanJobFactory jobFactory;
    @Resource
    private StdScheduler scheduler;
    @Resource
    private CronTriggerImpl fdbAddTrigger;
    @Resource
    private SchedulerState schedulerState;
    private boolean intialized;
    private NationalSettingDomainCapability nationalSettingDomainCapability;
    private NdfSynchQueueDomainCapability ndfSynchQueueDomainCapability;
    private ManagedItemCapability managedItemCapability;
    private VistaFileSynchCapability vistaFileSynchCapability;

    private ManagedItemCapabilityFactory managedItemCapabilityFactory;

    /**
     * This method initilizes the schedulerState
     */
    private void init() {
        try {

            LOG.debug("init()->>");

            getNationalSettingsData();

            // Make sure the time entered here match the times in the ServiceContextTemplate.xml
            // Spring configuration file.
            int fdbAddJobHour = 6;
            int fdbAddJobMins = 0;
            int fdbUpdateJobHour = 5;
            int fdbUpdateJobMins = 0;
            int stsJobHour = 3;
            int stsJobMins = 0;
            int fssJobHour = 2;
            int fssJobMins = 0;
            int inactivationJobHour = 1;
            int inactivationJobMins = 0;
            int ndfUpdateJobHour = 2;
            int ndfUpdateJobMins = 10;
            int rxNormJobHour = 4;
            int rxNormJobMins = 10;

            FdbSchedulerControlBean schedulerControl1 = new FdbSchedulerControlBean();
            schedulerControl1.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl1.setJobName(FdbJobNames.FDB_ADD_JOB.getJobName());
            schedulerControl1.setFdbJobNames(FdbJobNames.FDB_ADD_JOB);
            schedulerControl1.setHour(Integer.valueOf(fdbAddJobHour));
            schedulerControl1.setMins(Integer.valueOf(fdbAddJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.FDB_ADD_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.FDB_ADD_JOB, FDBGROUP, JobStatus.RUNNING, fdbAddJobHour, fdbAddJobMins, true);
            String cronExpr1 = generateCronExpression(schedulerControl1);
            rescheduleJob(schedulerControl1.getJobName(), cronExpr1);

            FdbSchedulerControlBean schedulerControl2 = new FdbSchedulerControlBean();
            schedulerControl2.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl2.setJobName(FdbJobNames.FDB_UPDATE_JOB.getJobName());
            schedulerControl2.setFdbJobNames(FdbJobNames.FDB_UPDATE_JOB);
            schedulerControl2.setHour(Integer.valueOf(fdbUpdateJobHour));
            schedulerControl2.setMins(Integer.valueOf(fdbUpdateJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.FDB_UPDATE_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.FDB_UPDATE_JOB, FDBGROUP, JobStatus.RUNNING, fdbUpdateJobHour, fdbUpdateJobMins, true);
            String cronExpr2 = generateCronExpression(schedulerControl2);
            rescheduleJob(schedulerControl2.getJobName(), cronExpr2);

            FdbSchedulerControlBean schedulerControl3 = new FdbSchedulerControlBean();
            schedulerControl3.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl3.setJobName(FdbJobNames.STS_JOB.getJobName());
            schedulerControl3.setFdbJobNames(FdbJobNames.STS_JOB);
            schedulerControl3.setHour(Integer.valueOf(stsJobHour));
            schedulerControl3.setMins(Integer.valueOf(stsJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.STS_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.STS_JOB, FDBGROUP, JobStatus.RUNNING, stsJobHour, stsJobMins, true);
            String cronExpr3 = generateCronExpression(schedulerControl3);
            rescheduleJob(schedulerControl3.getJobName(), cronExpr3);

            FdbSchedulerControlBean schedulerControl4 = new FdbSchedulerControlBean();
            schedulerControl4.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl4.setJobName(FdbJobNames.FSS_JOB.getJobName());
            schedulerControl4.setFdbJobNames(FdbJobNames.FSS_JOB);
            schedulerControl4.setHour(Integer.valueOf(fssJobHour));
            schedulerControl4.setMins(Integer.valueOf(fssJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.FSS_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.FSS_JOB, FDBGROUP, JobStatus.RUNNING, fssJobHour, fssJobMins, true);
            String cronExpr4 = generateCronExpression(schedulerControl4);
            rescheduleJob(schedulerControl4.getJobName(), cronExpr4);

            FdbSchedulerControlBean schedulerControl5 = new FdbSchedulerControlBean();
            schedulerControl5.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl5.setJobName(FdbJobNames.INACTIVATION_CHECK_JOB.getJobName());
            schedulerControl5.setFdbJobNames(FdbJobNames.INACTIVATION_CHECK_JOB);
            schedulerControl5.setHour(Integer.valueOf(inactivationJobHour));
            schedulerControl5.setMins(Integer.valueOf(inactivationJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.INACTIVATION_CHECK_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.INACTIVATION_CHECK_JOB, FDBGROUP, JobStatus.RUNNING, inactivationJobHour, inactivationJobMins, true);
            String cronExpr5 = generateCronExpression(schedulerControl5);
            rescheduleJob(schedulerControl5.getJobName(), cronExpr5);

            /** Story 33495 add batch job for NDF file sharing with VistA */
            FdbSchedulerControlBean schedulerControl6 = new FdbSchedulerControlBean();
            schedulerControl6.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl6.setJobName(FdbJobNames.NDF_UPDATE_JOB.getJobName());
            schedulerControl6.setFdbJobNames(FdbJobNames.NDF_UPDATE_JOB);
            schedulerControl6.setHour(Integer.valueOf(ndfUpdateJobHour));
            schedulerControl6.setMins(Integer.valueOf(ndfUpdateJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.NDF_UPDATE_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.NDF_UPDATE_JOB, FDBGROUP, JobStatus.RUNNING, 7, 2, 10, true);
            String cronExpr6 = generateCronExpression(schedulerControl6);
            rescheduleJob(schedulerControl6.getJobName(), cronExpr6);

            /** Story 165367 add batch job for RxNorm file update */
            FdbSchedulerControlBean schedulerControl7 = new FdbSchedulerControlBean();
            schedulerControl7.setJobCommands(JobCommands.SCHEDULE);
            schedulerControl7.setJobName(FdbJobNames.RXNORM_UPDATE_JOB.getJobName());
            schedulerControl7.setFdbJobNames(FdbJobNames.RXNORM_UPDATE_JOB);
            schedulerControl7.setHour(Integer.valueOf(rxNormJobHour));
            schedulerControl7.setMins(Integer.valueOf(rxNormJobMins));
            scheduler.pauseJob(new JobKey(FdbJobNames.RXNORM_UPDATE_JOB.getJobName(), FDBGROUP));
            this.updateJobStatus(FdbJobNames.RXNORM_UPDATE_JOB, FDBGROUP, JobStatus.RUNNING, 4, 10, true);
            String cronExpr7 = generateCronExpression(schedulerControl7);
            rescheduleJob(schedulerControl7.getJobName(), cronExpr7);


            List<String> jobNames = getJobNamesFromArray();
            this.getSchedulerState().setJobNames(jobNames);

            this.setIntialized(true);

            if (scheduler.isShutdown() || scheduler.isInStandbyMode()) {
                scheduler.start();
            }

        } catch (SchedulerException e) {
            LOG.error("Error: init() -> SchedulerException: " + e.getMessage());
        } catch (ParseException e) {
            LOG.error("Error: init() -> ParseException: " + e.getMessage());
        }

    }

    /**
     * execute Job Command
     * @param schedulerControl schedulerControl
     * @param pRunMode the run mode
     * @return Scheduler state.
     */
    @Override
    public SchedulerState executeJobCommand(FdbSchedulerControlBean schedulerControl, Boolean pRunMode) {
        initialize();

        return executeCommand(schedulerControl);
    }

    /**
     * initialize scheuduler
    */
    private void initialize() {
        if (!this.isIntialized()) {
            init();
        }
    }

    /**
     * Executes Job commands
     * @param schedulerControl schedulerControl
     * @return scheduler state
    */
    private SchedulerState executeCommand(FdbSchedulerControlBean schedulerControl) {

        FdbJobNames fdbJobName = schedulerControl.getFdbJobNames();
        JobCommands jobCommand = schedulerControl.getJobCommands();
        String cronExpr = generateCronExpression(schedulerControl);

        LOG.debug("executeCommand()-->> groupName: " + FDBGROUP + " fdbJobName: "
                    + fdbJobName + " jobCommand: " + jobCommand.toString());

        try {
            switch (jobCommand) {
                case SCHEDULE:
                    LOG.debug("executeCommand()-->> SCHEDULE");
                    rescheduleJob(schedulerControl.getJobName(), cronExpr);
                    updateJobStatus(fdbJobName, FDBGROUP, JobStatus.RUNNING,
                        schedulerControl.getDayOfWeek(), schedulerControl.getHour(), schedulerControl.getMins(), false);
                    schedulerState = getSchedulerStatus();

                    break;
                case START:
                    if (scheduler.isShutdown() || scheduler.isInStandbyMode()) {
                        scheduler.start();
                        scheduler.resumeAll();
                        schedulerState = getSchedulerStatus();
                        LOG.debug("executeCommand()-->> START");
                    }

                    break;
                case PAUSE:
                    if (!scheduler.isShutdown()) {
                        scheduler.pauseJob(new JobKey(fdbJobName.getJobName(), FDBGROUP));
                        updateJobStatus(fdbJobName, FDBGROUP, JobStatus.PAUSED,
                            schedulerControl.getDayOfWeek(), schedulerControl.getHour(), schedulerControl.getMins(), false);

                        schedulerState = getSchedulerStatus();
                        LOG.debug("executeCommand()-->> PAUSE");
                    }

                    break;
                case PAUSE_ALL:
                    if (!scheduler.isShutdown()) {
                        pauseAllJobs();
                        schedulerState = getSchedulerStatus();
                    }

                    break;
                case RESUME:
                    if (!scheduler.isShutdown()) {
                        scheduler.resumeJob(new JobKey(fdbJobName.getJobName(), FDBGROUP));
                        updateJobStatus(fdbJobName, FDBGROUP, JobStatus.RUNNING,
                            schedulerControl.getDayOfWeek(), schedulerControl.getHour(), schedulerControl.getMins(), false);
                        schedulerState = getSchedulerStatus();
                        LOG.debug("executeCommand()-->> RESUME");
                    }

                    break;

                case RESUME_ALL:
                    if (!scheduler.isShutdown()) {
                        resumeAllJobs();
                        schedulerState = getSchedulerStatus();
                    }

                    break;
                case STOP:
                    if (!scheduler.isShutdown()) {
                        LOG.debug("executeCommand()-->> STOP");
                        scheduler.standby();
                        pauseAllJobs();
                        schedulerState = getSchedulerStatus();
                    }

                    break;
                case SHUTDOWN:
                    scheduler.shutdown();

                    break;
                case STATUS:
                    LOG.debug("executeCommand()-->>STATUS");
                    schedulerState.setServerCurrentTime(Calendar.getInstance().getTime());

                    if (!scheduler.isShutdown()) {
                        LOG.debug("executeCommand()-->> scheduler.isShutDown" + scheduler.isShutdown());
                        schedulerState = getSchedulerStatus();
                    }

                    break;
                default:

                    break;
            }

        } catch (SchedulerException e) {
            LOG.error("Error: SchedulerException:  " + e.getMessage(),e);
        } catch (ParseException e) {
            LOG.error("Error: ParseException: " + e.getMessage(),e);
        }

        return schedulerState;
    }

    /**
     * 
     * Description
     *
     * @param jobName
     * @param groupName
     */
    private void updateTriggerStatus() {

        Trigger[] fdbAddTriggers;
        LOG.debug("updateTriggerStatus -->>");

        try {
            fdbAddTriggers = this.getTriggers("fdbAddJob");
            this.getSchedulerState().addJobTriggersMap("fdbAddJob", fdbAddTriggers);

            Trigger[] fdbUpdateTriggers = this.getTriggers("fdbUpdateJob");
            this.getSchedulerState().addJobTriggersMap("fdbUpdateJob", fdbUpdateTriggers);

            Trigger[] stsTriggers = this.getTriggers("stsJob");
            this.getSchedulerState().addJobTriggersMap("stsJob", stsTriggers);

            Trigger[] fssTriggers = this.getTriggers("fssJob");
            this.getSchedulerState().addJobTriggersMap("fssJob", fssTriggers);

            Trigger[] inactivationTriggers = this.getTriggers("inactivationCheckJob");
            this.getSchedulerState().addJobTriggersMap("inactivationCheckJob", inactivationTriggers);

            Trigger[] ndfUpdateTriggers = this.getTriggers("ndfUpdateJob");            
            this.getSchedulerState().addJobTriggersMap("ndfUpdateJob", ndfUpdateTriggers);

            Trigger[] rxNormUpdateTriggers = this.getTriggers("rxNormUpdateJob");            
            this.getSchedulerState().addJobTriggersMap("rxNormUpdateJob", rxNormUpdateTriggers);

        } catch (SchedulerException e) {
            LOG.error(" Error:  SchedulerException: " + e.getMessage(),e);
        }

    }

    /**
        * resumes all Jobs
        *
        * @throws SchedulerException SchedulerException
        */
    private void resumeAllJobs() throws SchedulerException {

        Map<String, JobStatusInfo> jobStatusInfoMap = getSchedulerState().getJobStatusInfoMap();

        LOG.debug(".................resuming All Jobs..................");
        scheduler.resumeAll();

        for (Map.Entry<String, JobStatusInfo> entry : jobStatusInfoMap.entrySet()) {

            LOG.debug(entry.getKey() + "/" + entry.getValue());
            JobStatusInfo statusInfo = entry.getValue();

            updateJobStatus(statusInfo.getFdbJobNames(), statusInfo.getGroupName(),
                    JobStatus.RUNNING, statusInfo.getHour(), statusInfo.getMins(), false);

            LOG.debug("1.statusInfo: jobName: " + statusInfo.getJobName());
            LOG.debug("1.statusInfo: jobGroupName:  " + statusInfo.getGroupName());
            LOG.debug("1.statusInfo: job status: " + statusInfo.getStatus());

        }

    }

    /**
     * pause All Jobs
     *
     * @throws SchedulerException SchedulerException
     */
    private void pauseAllJobs() throws SchedulerException {

        Map<String, JobStatusInfo> jobStatusInfoMap = getSchedulerState().getJobStatusInfoMap();
        LOG.debug(".................pausing All Jobs..................");
        scheduler.pauseAll();

        for (Map.Entry<String, JobStatusInfo> entry : jobStatusInfoMap.entrySet()) {
            JobStatusInfo statusInfo = entry.getValue();
            updateJobStatus(statusInfo.getFdbJobNames(), statusInfo.getGroupName(),
                    JobStatus.PAUSED, statusInfo.getDayOfWeek(), statusInfo.getHour(), statusInfo.getMins(), false);
        }
    }

    /**
     * 
     * Reshedules a job
     *
     * @param schedulerControl schedulerControl
     * @param cronExpr cron expression
     * @throws ParseException ParseException
     * @throws SchedulerException SchedulerException
     */
    public void rescheduleJob(String jobName, String cronExpr) throws ParseException, SchedulerException {

        Map<String, Trigger[]> triggerMap = this.getSchedulerState().getJobTriggerMap();
        Trigger[] jobTriggers = triggerMap.get(jobName);
        TriggerKey triggerKey = jobTriggers[0].getKey(); // we are only using one trigger per job.
        CronTriggerImpl trigger = (CronTriggerImpl) scheduler.getTrigger(triggerKey);
        trigger.setCronExpression(cronExpr );
        scheduler.rescheduleJob(triggerKey, trigger);
        
        SchedulerState state = getSchedulerState();
        state.addJobTriggersMap(jobName, new Trigger[]{trigger});
        this.getScheduler().rescheduleJob(triggerKey, trigger);

    }

    /**
     * Update the JobStatusInfo and put in map
     * @param pFdbJobName job status
     * @param groupName group name
     * @param pJobStatus pJobStatus
     * @param pDayOfWeek day of week
     * @param pHour hours
     * @param pMins mins
     * @param init init
     */
    private void updateJobStatus(FdbJobNames pFdbJobName, String groupName,
            JobStatus pJobStatus, Integer pDayOfWeek, Integer pHour, Integer pMins, boolean init) {

        JobStatusInfo jobStatusInfo;
        Trigger[] fdbTriggers;
        SchedulerState state = getSchedulerState();

        LOG.debug("updateJobStatus()->> jobName: "
                    + pFdbJobName + "groupName: " + groupName + " JobStatus: " + pJobStatus.toString() + "init: " + init);

        try {

            if (init) {
                fdbTriggers = this.getTriggers(pFdbJobName.getJobName());
                state.addJobTriggersMap(pFdbJobName.getJobName(), fdbTriggers);
            } else {
                Map<String, Trigger[]> triggerMap = state.getJobTriggerMap();
                fdbTriggers = triggerMap.get(pFdbJobName.getJobName()); // we are only using one trigger per job.
            }

            if (state.getJobStatusInfoMap().containsKey(pFdbJobName.getJobName())) {
                jobStatusInfo = state.getJobStatusInfoMap().get(pFdbJobName.getJobName());
            } else {
                jobStatusInfo = new JobStatusInfo();
            }

            // get eplNational data from 
            EplNationalInfo eplNationalInfo = state.getEplNationalInfoMap().get(pFdbJobName);

            if (eplNationalInfo != null) {
                jobStatusInfo.setProcessStatus(eplNationalInfo.getProcessStatus());
                jobStatusInfo.setLastSuccessRunDate(eplNationalInfo.getLastSuccessRunDate());
            }

            jobStatusInfo.setJobName(pFdbJobName.getJobName());
            jobStatusInfo.setGroupName(groupName);
            jobStatusInfo.setStatus(pJobStatus);
            jobStatusInfo.setFdbJobNames(pFdbJobName);
            jobStatusInfo.setNextFireTime(fdbTriggers[0].getNextFireTime());
            jobStatusInfo.setDayOfWeek(pDayOfWeek);
            jobStatusInfo.setHour(pHour);
            jobStatusInfo.setMins(pMins);

            state.addJobStatusInfoToMap(jobStatusInfo);

        } catch (SchedulerException e) {
            LOG.error("Error: SchedulerException: " + e.getMessage(),e);
        }

    }

    /**
     * Update the JobStatusInfo and put in map
     * @param pFdbJobName job status
     * @param groupName group name
     * @param pJobStatus pJobStatus
     * @param pHour hours
     * @param pMins mins
     * @param init init
     */
    private void updateJobStatus(FdbJobNames pFdbJobName, String groupName,
            JobStatus pJobStatus, Integer pHour, Integer pMins, boolean init) {
        updateJobStatus(pFdbJobName, groupName, pJobStatus, null /* dayOfWeek */, pHour, pMins, init);
    }
    
    /**
     * update All JobsStatus
     *
     * @throws SchedulerException SchedulerException
     */
    private void updateAllJobsStatus() throws SchedulerException {

        Map<String, JobStatusInfo> jobStatusInfoMap = getSchedulerState().getJobStatusInfoMap();
        LOG.debug(".................updating All Jobs..................");

        for (Map.Entry<String, JobStatusInfo> entry : jobStatusInfoMap.entrySet()) {
            JobStatusInfo statusInfo = entry.getValue();

            LOG.debug("statusInfo: jobName: " + statusInfo.getJobName());
            LOG.debug("statusInfo: jobGroupName:  " + statusInfo.getGroupName());
            LOG.debug("statusInfo: job status: " + statusInfo.getStatus());
            LOG.debug("statusInfo: next fire time: " + statusInfo.getNextFireTime());
            LOG.debug("statusInfo: Process status: " + statusInfo.getProcessStatus());

            updateJobStatus(statusInfo.getFdbJobNames(), statusInfo.getGroupName(),
                    statusInfo.getStatus(), statusInfo.getDayOfWeek(), statusInfo.getHour(), statusInfo.getMins(), false);
        }
    }

    /**
     * returns the scheduler state 
     * @return scheduler state
     */
    private SchedulerState getSchedulerStatus() {
        SchedulerState state = null;

        try {

            state = getNationalSettingsData();

            if (scheduler.isInStandbyMode()) {
                state.setSchedulerStatus(SchedulerStatus.STANDBY);
            } else if (scheduler.isShutdown()) {
                state.setSchedulerStatus(SchedulerStatus.SHUTDOWN);
            } else {
                state.setSchedulerStatus(SchedulerStatus.RUNNING);
            }

            updateTriggerStatus();
            updateAllJobsStatus();

            state.setScheduler(scheduler);

        } catch (SchedulerException e) {
            LOG.error("Error:getSchedulerStatus(): SchedulerException: " + e.getMessage(),e);
        }

        return state;
    }

    /**
     *  returns triggers for JobName, JobGroup
     * @param pJobName job name
     * @return Triggers
     * @throws SchedulerException SchedulerException
     */
    private Trigger[] getTriggers(String pJobName) throws SchedulerException {

        List<? extends Trigger> triggers = scheduler.getTriggersOfJob(new JobKey(pJobName, FDBGROUP));

        return (Trigger[]) triggers.toArray(new Trigger[triggers.size()]);
    }

    /**
     * creates a list of job names
     * 
     * @return returns list of job names
     * @throws SchedulerException
     *             SchedulerException
     */
    private List<String> getJobNamesFromArray() throws SchedulerException {

        List<String> jobNamesList = new ArrayList<String>();

        Set<JobKey> jobKeys = scheduler.getJobKeys(GroupMatcher.jobGroupEquals(FDBGROUP));

        for (JobKey jobKey: jobKeys) {
            jobNamesList.add(jobKey.getName());
        }

        return jobNamesList;
    }

    /**
     * Returns the GroupName
     * 
     * @param pJobName
     *            the job name
     * @return returns the groupName
     */
//    private String getGroupName(String pJobName) {
//        // just need to use the constant, since all jobs are part of the same group.
//        return FDBGROUP;
//    }

    /**
     * populates the SchedulerState 
     * with data form VistaMessageInfo and EplNationalInfo.
     * @return SchedulerState
     */
    private SchedulerState getNationalSettingsData() {

        LOG.debug("getNationalSettingsData()->>");

        Map<String, Object> nationalSettingsMap = retrieveNationalSettingsMap();
        SchedulerState state = this.getSchedulerState();

        Map<FdbJobNames, EplNationalInfo> nationalInfoMap = createNationalInfoMap(nationalSettingsMap);
        state.setEplNationalInfoMap(nationalInfoMap);

        NationalSettingVo vo1 = (NationalSettingVo) nationalSettingsMap.get(NationalSetting.HOST_NAME.toString());
        state.setFdaHostName(vo1.getStringValue());

        NationalSettingVo vo2 = (NationalSettingVo) nationalSettingsMap.get(NationalSetting.MESSAGE_STATUS.toString());
        state.setMessagingRunning(vo2.getBooleanValue());

        NationalSettingVo vo3 = (NationalSettingVo) nationalSettingsMap.get(NationalSetting.MESSAGE_ERROR.toString());
        state.setErrorMessage(vo3.getStringValue());

        NationalSettingVo vo4 = (NationalSettingVo) nationalSettingsMap.get(NationalSetting.NUM_MSG_QUEUE.toString());
        state.setMessagesOnQueueCount(vo4.getIntegerValue().intValue());

        NationalSettingVo vo5 =
            (NationalSettingVo) nationalSettingsMap.get(NationalSetting.MESSAGE_QUEUE_IN_PROGRESS.toString());
        state.setMessagingQueueInProcess(vo5.getBooleanValue());

        return state;
    }

    /**
     * get Message Status
     *
     * @return messages status
     */
    @Transactional(readOnly=true)
    public boolean getMessageStatus() {

        NationalSettingVo vo = null;
        boolean success = false;

        try {
            vo = nationalSettingDomainCapability.retrieve(NationalSetting.MESSAGE_STATUS.toString());
            if (vo != null) {
                success = vo.getBooleanValue();
            }
        } catch (Exception e) {
            LOG.error("Exception, in getMessageStatus()  " + e.getMessage(),e);

            if (e.getCause() != null) {
                LOG.error("This Underlying cause is " + e.getCause().getMessage(),e);
            }
        }

        return success;
    }

    /**
     * updates the Host Name
     * @param hostName hostName
     * @param pUser pUser
     */
    @Override
    public void updateHostName(String hostName, UserVo pUser) {

        NationalSettingVo vo = new NationalSettingVo();
        vo.setKeyName(NationalSetting.HOST_NAME.toString());
        vo.setStringValue(hostName);
        updateNationalSettings(vo, pUser);

    }

    /**
     * update MessagingState
     * @param messagingState messagingState
     * @param pUser pUser
     */
    @Override
    public void updateMessagingState(Boolean messagingState, UserVo pUser) {

        SchedulerState state = null;

        LOG.debug("getSchedulerStatus()");

        state = getNationalSettingsData();

        if (state.isMessagingQueueInProgress()) {

            // No changes to Message Status while Message Queuing is in Process
            return;
        }

        NationalSettingVo vo = new NationalSettingVo();

        if (messagingState) {

            // Set Message Queuing to be in process
            NationalSettingVo messageQueuingVo = new NationalSettingVo();
            messageQueuingVo.setKeyName(NationalSetting.MESSAGE_QUEUE_IN_PROGRESS.toString());
            messageQueuingVo.setBooleanValue(true);
            updateNationalSettings(messageQueuingVo, pUser);
            
            vo.setKeyName(NationalSetting.MESSAGE_STATUS.toString());
            vo.setBooleanValue(messagingState);
            vo.setDateValue(new Date());
            updateNationalSettings(vo, pUser);

            try {
                // Empty the VistA queue
                emptyQueue(pUser);
            } catch (Exception e) {
                vo.setKeyName(NationalSetting.MESSAGE_STATUS.toString());
                vo.setBooleanValue(!messagingState);
                vo.setDateValue(new Date());
                updateNationalSettings(vo, pUser);
                LOG.error("Excpetion occured while procesing messages so putting the state back in false.",e);
                
            }

            // Set Message Queuing in Process to false
            messageQueuingVo.setKeyName(NationalSetting.MESSAGE_QUEUE_IN_PROGRESS.toString());
            messageQueuingVo.setBooleanValue(false);
            updateNationalSettings(messageQueuingVo, pUser);

        }

        vo.setKeyName(NationalSetting.MESSAGE_STATUS.toString());
        vo.setBooleanValue(messagingState);

        // also setting the Date as it is used in other settings.
        vo.setDateValue(new Date());
        updateNationalSettings(vo, pUser);

    }

    /**
     * Generate a CRON expression is a string comprising 5 or 6 fields separated by white space.
     * minutes mandatory = yes. allowed values = {@code  0-59    * / , -}
     * hours mandatory = yes. allowed values = {@code 0-23   * / , -}
     * dayOfMonth mandatory = yes. allowed values = {@code 1-31  * / , - ? L W}
     * 
     * m month mandatory = yes. allowed values = {@code 1-12 or JAN-DEC    * / , -}
     * dayOfWeek mandatory = yes. allowed values = {@code 0-6 or SUN-SAT * / , - ? L #}
     * year mandatory = no. allowed values = {@code 1970-2099    * / , -}
     * @return a CRON Formatted String.
     * 
     * 
        * * * * * ?
        - - - - - -
        | | | | | |
        | | | | | +----- day of week (MON-SUN)
        | | | | +------- month (1 - 12)
        | | | +--------- day of month (1 - 31)
        | | +----------- hour (0 - 23)
        | +------------- min (0 - 59)
        +------------- sec (0 - 59)
     */

    /**
     * Generates cron expresion based on hours/minutes
     * @param schedulerControl schedulerControl 
     * @return the cron expresssion
     */
    private String generateCronExpression(FdbSchedulerControlBean schedulerControl) {
       
        /* if job is a weekly job, the Day of Month needs to be set to ? (no value) in that case. 
         * Vice versa for the Day of Week */ 
        String dayOfMonth = "*";
        Integer dayOfWeek = schedulerControl.getDayOfWeek(); 
        String cronDayOfWeek = null;
        if (dayOfWeek != null && dayOfWeek.intValue() > 0) {
            cronDayOfWeek = dayOfWeek.toString();
            dayOfMonth = "?";
        } else {
            cronDayOfWeek = "?";
        }
        
        return String.format("%1$s %2$s %3$s %4$s %5$s %6$s", "0", schedulerControl.getMins(),
            schedulerControl.getHour(), dayOfMonth, "*", cronDayOfWeek);
    }

    /**
     * retrieveNationalSettingsMap
     *
     * @return Map
     */
    public Map<String, Object> retrieveNationalSettingsMap() {
        Map<String, Object> nationalMap = new HashMap<String, Object>();
        List<NationalSettingVo> nationalSettingList = nationalSettingDomainCapability.retrieve();

        for (NationalSettingVo nationalSettingVo2 : nationalSettingList) {
            nationalMap.put(nationalSettingVo2.getKeyName(), nationalSettingVo2);
        }

        return nationalMap;
    }

    /**
     * updates NationalSettings Vo by seting the vo with the key/value
     * @param vo NationalSettingVo
     * @param pUser pUser
     */
    @Transactional
    public void updateNationalSettings(NationalSettingVo vo, UserVo pUser) {
       
        try {
            Map<String, Object> nsMap = retrieveNationalSettingsMap();
    
            if (nsMap != null) {
                NationalSettingVo pVo = (NationalSettingVo) nsMap.get(vo.getKeyName());
    
                pVo.setKeyName(vo.getKeyName());
                pVo.setStringValue(vo.getStringValue());
                pVo.setBooleanValue(vo.getBooleanValue());
                pVo.setDateValue(vo.getDateValue());
                pVo.setDecimalValue(vo.getDecimalValue());
                pVo.setIntegerValue(vo.getIntegerValue());
                pVo.setCreatedBy(vo.getCreatedBy());
                pVo.setCreatedDate(vo.getCreatedDate());
    
                nationalSettingDomainCapability.update(pVo, pUser);
    
            }
        } catch (Exception e) {
            LOG.error("Exception, in getMessageStatus()  " + e.getMessage(),e);

            if (e.getCause() != null) {
                LOG.error("This Underlying cause is " + e.getCause().getMessage(),e);
            }
        }

    }

    /**
     * create National Info map
     * @param natitionalSettingsMap natitionalSettingsMap
     * @return Map
     */
    private Map<FdbJobNames, EplNationalInfo> createNationalInfoMap(Map<String, Object> natitionalSettingsMap) {

        LOG.debug("createNationalInfoMap()-->");

        SchedulerState schedState = this.getSchedulerState();

        Map<FdbJobNames, EplNationalInfo> eplNationalInfoMap = schedState.getEplNationalInfoMap();
        eplNationalInfoMap.clear();

        // fdb_add_job
        EplNationalInfo eplNationalInfo1 = new EplNationalInfo();
        eplNationalInfo1.setFdbJobNames(FdbJobNames.FDB_ADD_JOB);
        NationalSettingVo lastRunVo1 = (NationalSettingVo)
                        natitionalSettingsMap.get(NationalSetting.FDB_ADD_LAST_RUN.toString());
        eplNationalInfo1.setLastSuccessRunDate(lastRunVo1.getDateValue());

        NationalSettingVo runStateVo1 = (NationalSettingVo)
                        natitionalSettingsMap.get(NationalSetting.FDB_ADD_RUN_STATE.toString());
        eplNationalInfo1.setProcessStatus(ProcessStatus.valueOf(runStateVo1.getStringValue()));
        eplNationalInfoMap.put(FdbJobNames.FDB_ADD_JOB, eplNationalInfo1);

        // fdb_update_job
        EplNationalInfo eplNationalInfo2 = new EplNationalInfo();
        eplNationalInfo2.setFdbJobNames(FdbJobNames.FDB_UPDATE_JOB);
        NationalSettingVo lastRunVo2 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.FDB_UPDATE_LAST_RUN.toString());
        eplNationalInfo2.setLastSuccessRunDate(lastRunVo2.getDateValue());

        NationalSettingVo runStateVo2 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.FDB_UPDATE_RUN_STATE.toString());
        eplNationalInfo2.setProcessStatus(ProcessStatus.valueOf(runStateVo2.getStringValue()));
        eplNationalInfoMap.put(FdbJobNames.FDB_UPDATE_JOB, eplNationalInfo2);

        // FSS_job
        EplNationalInfo eplNationalInfo3 = new EplNationalInfo();
        eplNationalInfo3.setFdbJobNames(FdbJobNames.FSS_JOB);
        NationalSettingVo lastRunVo3 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.FSS_UPDATE_LAST_RUN.toString());
        eplNationalInfo3.setLastSuccessRunDate(lastRunVo3.getDateValue());

        NationalSettingVo runStateVo3 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.FSS_RUN_STATE.toString());
        eplNationalInfo3.setProcessStatus(ProcessStatus.valueOf(runStateVo3.getStringValue()));
        eplNationalInfoMap.put(FdbJobNames.FSS_JOB, eplNationalInfo3);

        // STS_job
        EplNationalInfo eplNationalInfo4 = new EplNationalInfo();
        eplNationalInfo4.setFdbJobNames(FdbJobNames.STS_JOB);
        NationalSettingVo lastRunVo4 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.STS_UPDATE_LAST_RUN.toString());
        eplNationalInfo4.setLastSuccessRunDate(lastRunVo4.getDateValue());

        NationalSettingVo runStateVo4 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.STS_RUN_STATE.toString());
        eplNationalInfo4.setProcessStatus(ProcessStatus.valueOf(runStateVo4.getStringValue()));
        eplNationalInfoMap.put(FdbJobNames.STS_JOB, eplNationalInfo4);

        // Inactivation check Job
        EplNationalInfo eplNationalInfo5 = new EplNationalInfo();
        eplNationalInfo5.setFdbJobNames(FdbJobNames.INACTIVATION_CHECK_JOB);
        NationalSettingVo lastRunVo5 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.INACTIVATION_CHECK_LAST_RUN.toString());
        eplNationalInfo5.setLastSuccessRunDate(lastRunVo5.getDateValue());

        NationalSettingVo runStateVo5 =
            (NationalSettingVo) natitionalSettingsMap.get(NationalSetting.INACTIVATION_RUN_STATE.toString());
        eplNationalInfo5.setProcessStatus(ProcessStatus.valueOf(runStateVo5.getStringValue()));
        eplNationalInfoMap.put(FdbJobNames.INACTIVATION_CHECK_JOB, eplNationalInfo5);

        // ndf_update_job
        EplNationalInfo eplNationalInfo6 = new EplNationalInfo();
        eplNationalInfo6.setFdbJobNames(FdbJobNames.NDF_UPDATE_JOB);
        NationalSettingVo lastRunVo6 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.NDF_UPDATE_LAST_RUN.toString());
        if (lastRunVo6 != null) {
            eplNationalInfo6.setLastSuccessRunDate(lastRunVo6.getDateValue());
        }
        
        NationalSettingVo runStateVo6 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.NDF_UPDATE_RUN_STATE.toString());
        if (runStateVo6!=null && !StringUtils.isEmpty(runStateVo6.getStringValue())) {
            eplNationalInfo6.setProcessStatus(ProcessStatus.valueOf(runStateVo6.getStringValue()));
        }

        eplNationalInfoMap.put(FdbJobNames.NDF_UPDATE_JOB, eplNationalInfo6);

        // rxnorm_update_job
        EplNationalInfo eplNationalInfo7 = new EplNationalInfo();
        eplNationalInfo7.setFdbJobNames(FdbJobNames.RXNORM_UPDATE_JOB);
        NationalSettingVo lastRunVo7 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.RXNORM_UPDATE_LAST_RUN.toString());
        if (lastRunVo7 != null) {
            eplNationalInfo7.setLastSuccessRunDate(lastRunVo7.getDateValue());
        }
        
        NationalSettingVo runStateVo7 = (NationalSettingVo)
                    natitionalSettingsMap.get(NationalSetting.RXNORM_UPDATE_RUN_STATE.toString());
        if (runStateVo7!=null && !StringUtils.isEmpty(runStateVo7.getStringValue())) {
            eplNationalInfo7.setProcessStatus(ProcessStatus.valueOf(runStateVo7.getStringValue()));
        }

        eplNationalInfoMap.put(FdbJobNames.RXNORM_UPDATE_JOB, eplNationalInfo7);
        return eplNationalInfoMap;
    }

    /**
     * Get the {@link ManagedItemDomainCapability} for the given {@link EntityType} using the
     * {@link ManagedItemCapabilityFactory}.
     * 
     * @param <T> Type of {@link ManagedItemDomainCapability}
     * @param entityType {@link EntityType}
     * @return {@link ManagedItemDomainCapability}
     */

    /**
     * emptyQueue empties the VistA queue
     * @param user User not really used
     */
    private void emptyQueue(UserVo user) {

        // Get the list of messages to be sent
        List<NdfSynchQueueVo> queueList = ndfSynchQueueDomainCapability.retrieve();

        // Go through the list for each of the Types of messages needed
        if (queueList != null) {

            List<Difference> diffList = new ArrayList<Difference>();
            Difference diff = new Difference(FieldKey.INACTIVATION_DATE, "", "");
            diffList.add(diff);

           
            // Send Drug Units
            EntityType itemType = EntityType.DRUG_UNIT;
            emptyType(queueList, diffList, user, itemType);

            // Send Dispense Units
            itemType = EntityType.DISPENSE_UNIT;
            emptyType(queueList, diffList, user, itemType);
 
            // Send VA Generic Name
            itemType = EntityType.GENERIC_NAME;
            emptyType(queueList, diffList, user, itemType);

            // Send Dosage Form
            itemType = EntityType.DOSAGE_FORM;
            emptyType(queueList, diffList, user, itemType);

            // Send Drug Class
            diff.setFieldKey(FieldKey.PARENT_DRUG_CLASS);
            itemType = EntityType.DRUG_CLASS;
            emptyType(queueList, diffList, user, itemType);


            // Send Drug Ingredient
            diff.setFieldKey(FieldKey.INACTIVATION_DATE);
            itemType = EntityType.INGREDIENT;
            emptyType(queueList, diffList, user, itemType);

            // Send VA Product
            itemType = EntityType.PRODUCT;
            emptyType(queueList, diffList, user, itemType);

            // Send PACKAGE_TYPE
            itemType = EntityType.PACKAGE_TYPE;
            emptyType(queueList, diffList, user, itemType);
            
            // Send MANUFACTURER
            itemType = EntityType.MANUFACTURER;
            emptyType(queueList, diffList, user, itemType);
            
            // Send NDC
            itemType = EntityType.NDC;
            emptyType(queueList, diffList, user, itemType);
            
        }
        
        // reset error message to empty if no items left in queue
        Long errors = nationalSettingDomainCapability.retrieveInteger(NationalSetting.NUM_MSG_QUEUE.toString());
        if ( errors == 0 ) {
            nationalSettingDomainCapability.update(NationalSetting.NUM_MSG_QUEUE.toString(), "", user );
        }
    }


    /**
     * Description
     *
     * @param queueList List of queued messages
     * @param diffList Faked diff list to cause it to be sent
     * @param user User not really used
     * @param itemType Entity Type being sent
     * @return true if no error logged
     */
    private boolean emptyType(List<NdfSynchQueueVo> queueList, List<Difference> diffList, UserVo user, EntityType itemType) {
        
        NationalSettingVo numQueueMessagesSetting = nationalSettingDomainCapability.retrieve(NationalSetting.NUM_MSG_QUEUE.toString());

        if ( numQueueMessagesSetting == null ) {
            LOG.error( "Cannot find National Setting for key " + NationalSetting.NUM_MSG_QUEUE.toString());
        }
        
        for (NdfSynchQueueVo synchQueueVo : queueList) {

            try {

                if (synchQueueVo.getItemType().equalsIgnoreCase(itemType.toString())) {
                    ManagedItemVo item = managedItemCapability.retrieve(synchQueueVo.getIdFk(), itemType);
                    LOG.debug("Found a " + itemType + " and it is a value of " + item.getValue());
               
                    if (synchQueueVo.getActionType().equalsIgnoreCase(ItemAction.ADD.toString())) {
                        LOG.debug("ADD and send");
                        vistaFileSynchCapability.sendNewItemToVista(item, user, false, true);
                        LOG.debug("ADD and update");
                        managedItemCapability.update(item, user);
                        LOG.debug("ADD and done");
                    } else {
                        LOG.debug("Modfy and send");
                        vistaFileSynchCapability.sendModifiedItemToVista(item, diffList, user, false, true);
                        LOG.debug("Modfy and done");
                    }
                    
                    ndfSynchQueueDomainCapability.deleteItemById(synchQueueVo.getId());
                    LOG.debug("after delete");
                    if ( numQueueMessagesSetting != null ) {
                        numQueueMessagesSetting.setIntegerValue(numQueueMessagesSetting.getIntegerValue() - 1);
                    }
                }
                
            } catch (Exception e) {
                String error = "Error while trying to empty queue for ViatA - EPL_ID " + synchQueueVo.getId()
                    + "/n    ItemType - " + synchQueueVo.getItemType()
                    + "/n    Action - " + synchQueueVo.getActionType() + "/n";
                    
                LOG.error(error, e);

                nationalSettingDomainCapability.update(NationalSetting.MESSAGE_ERROR.toString(), error, user);
                nationalSettingDomainCapability.update(numQueueMessagesSetting, user);
                
                return false;

            }
            
        }
        
        nationalSettingDomainCapability.update(numQueueMessagesSetting, user);
        return true;

    }
    
    /**
     * Add Current Request to the VistA Queue
     *
     * @param managedItem Item to be added (by reference) 
     * @param itemType The item's type
     * @param action The action - Add or Modify
     * @param user - The current user
     */
    protected void addToQueue(ManagedItemVo managedItem, EntityType itemType, ItemAction action, UserVo user) {
        NdfSynchQueueVo ndfSynchQueueVo = new NdfSynchQueueVo();
        ndfSynchQueueVo.setIdFk(managedItem.getId());
        ndfSynchQueueVo.setItemType(itemType.toString());
        ndfSynchQueueVo.setActionType(action.value());

        ndfSynchQueueDomainCapability.createWithoutDuplicateCheck(ndfSynchQueueVo, user);

        NationalSettingVo setting = nationalSettingDomainCapability.retrieve(NationalSetting.NUM_MSG_QUEUE.toString());

        if (setting != null ) {
            setting.setIntegerValue(setting.getIntegerValue() + 1);
            nationalSettingDomainCapability.update(setting, user);
        }
    }
    
    /**
     * returns instance of the scheduler    
     * @return the scheduler
     */
    @Override
    public StdScheduler getScheduler() {
        return scheduler;
    }

    /**
     * setter for the scheduler
     * @param pScheduler sets the scheduler
     */
    @Override
    public void setScheduler(StdScheduler pScheduler) {
        this.scheduler = pScheduler;
    }

    /**
     * get Scheduler State
     * @return the scheduler state
     */
    public SchedulerState getSchedulerState() {
        return schedulerState;
    }

    /**
     * setter for the scheduler state
     * @param pSchedulerState sets the scheduler state
     */
    public void setSchedulerState(SchedulerState pSchedulerState) {
        this.schedulerState = pSchedulerState;
    }

    /**
     * returns jobFactory
     * @return jobFactory
     */
    public SpringBeanJobFactory getJobFactory() {
        return jobFactory;
    }

    /**
     * setter for the jobFactory
     * @param pJobFactory sets the jobFactory
     */
    public void setJobFactory(SpringBeanJobFactory pJobFactory) {
        this.jobFactory = pJobFactory;
    }

    /**
     * gets FdbAddJob
     * @return the fdbAddJob
     */
    public JobDetailImpl getFdbAddJob() {
        return fdbAddJob;
    }
    
    /**
     * sets FdbAddJob
     * @param fdbAddJob the fdbAddJob to set
     */
    public void setFdbAddJob(JobDetailImpl fdbAddJob) {
        this.fdbAddJob = fdbAddJob;
    }

    /**
     * sets NationalSettingDomainCapability
     * @param nationalSettingDomainCapability the nationalSettingDomainCapability to set
     */
    public void setNationalSettingDomainCapability(
            NationalSettingDomainCapability nationalSettingDomainCapability) {
        this.nationalSettingDomainCapability = nationalSettingDomainCapability;
    }

    /**
     * gets FdbAddTrigger
     * @return the fdbAddTrigger
     */
    public CronTriggerImpl getFdbAddTrigger() {
        return fdbAddTrigger;
    }

    /**
     * sets FdbAddTrigger
     * @param fdbAddTrigger the fdbAddTrigger to set
     */
    public void setFdbAddTrigger(CronTriggerImpl fdbAddTrigger) {
        this.fdbAddTrigger = fdbAddTrigger;
    }

    /**
     * gets FdbUpdateJob
     * @return the fdbUpdateJob
     */
    public JobDetailImpl getFdbUpdateJob() {
        return fdbUpdateJob;
    }

    /**
     * sets  FdbUpdateJob
     * @param fdbUpdateJob the fdbUpdateJob to set
     */
    public void setFdbUpdateJob(JobDetailImpl fdbUpdateJob) {
        this.fdbUpdateJob = fdbUpdateJob;
    }

    /**
     *  gets StsJob 
     * @return the stsJob
     */
    public JobDetailImpl getStsJob() {
        return stsJob;
    }

    /**
     * sets StsJob
     * @param stsJob the stsJob to set
     */
    public void setStsJob(JobDetailImpl stsJob) {
        this.stsJob = stsJob;
    }

    /**
     * gets FssJob
     * @return the FssJob
     */
    public JobDetailImpl getFssJob() {
        return fssJob;
    }

    /**
     * sets fssJob
     * @param fssJob the fssJob to set
     */
    public void setFssJob(JobDetailImpl fssJob) {
        this.fssJob = fssJob;
    }

    /**
     * is Intialized
     * @return the intialized
     */
    public boolean isIntialized() {
        return intialized;
    }

    /**
     * sets Intialized
     * @param intialized the intialized to set
     */
    public void setIntialized(boolean intialized) {
        this.intialized = intialized;
    }

    /**
     * getS InactivationCheckJob
     * @return the inactivationCheckJob
     */
    public JobDetailImpl getInactivationCheckJob() {
        return inactivationCheckJob;
    }

    /**
     * setS InactivationCheckJob
     * @param inactivationCheckJob the inactivationCheckJob to set
     */
    public void setInactivationCheckJob(JobDetailImpl inactivationCheckJob) {
        this.inactivationCheckJob = inactivationCheckJob;
    }

    /**
     * gets NdfUpdateJob
     * @return the ndfUpdateJob
     */
        public JobDetailImpl getNdfUpdateJob() {
        return ndfUpdateJob;
    }

    /**
     * sets  NdfUpdateJob
     * @param ndfUpdateJob the ndfUpdateJob to set
     */
    public void setNdfUpdateJob(JobDetailImpl ndfUpdateJob) {
        this.ndfUpdateJob = ndfUpdateJob;
    }

    /**
     * gets RxNormUpdateJob
     * @return the rxNormUpdateJob
     */
        public JobDetailImpl getRxNormUpdateJob() {
        return rxNormUpdateJob;
    }

    /**
     * sets  RxNormUpdateJob
     * @param rxNormUpdateJob the rxNormUpdateJob to set
     */
    public void setRxNormUpdateJob(JobDetailImpl rxNormUpdateJob) {
        this.rxNormUpdateJob = rxNormUpdateJob;
    }

    /**
     * getNdfSynchQueueDomainCapability for FdbSchedulerProcessCapabilityImpl.
     * @return the ndfSynchQueueDomainCapability
     */
    public NdfSynchQueueDomainCapability getNdfSynchQueueDomainCapability() {
        return ndfSynchQueueDomainCapability;
    }

    /**
     * setNdfSynchQueueDomainCapability for FdbSchedulerProcessCapabilityImpl.
     * @param ndfSynchQueueDomainCapability the ndfSynchQueueDomainCapability to set
     */
    public void setNdfSynchQueueDomainCapability(NdfSynchQueueDomainCapability ndfSynchQueueDomainCapability) {
        this.ndfSynchQueueDomainCapability = ndfSynchQueueDomainCapability;
    }

    /**
     * getManagedItemCapabilityFactory for FdbSchedulerProcessCapabilityImpl.
     * @return the managedItemCapabilityFactory
     */
    public ManagedItemCapabilityFactory getManagedItemCapabilityFactory() {
        return managedItemCapabilityFactory;
    }

    /**
     * setManagedItemCapabilityFactory
     * @param managedItemCapabilityFactory the managedItemCapabilityFactory to set
     */
    public void setManagedItemCapabilityFactory(ManagedItemCapabilityFactory managedItemCapabilityFactory) {
        this.managedItemCapabilityFactory = managedItemCapabilityFactory;
    }

    /**
     * getVistaFileSynchCapability
     * @return the vistaFileSynchCapability
     */
    public VistaFileSynchCapability getVistaFileSynchCapability() {
        return vistaFileSynchCapability;
    }

    /**
     * setVistaFileSynchCapability
     * @param vistaFileSynchCapability the vistaFileSynchCapability to set
     */
    public void setVistaFileSynchCapability(VistaFileSynchCapability vistaFileSynchCapability) {
        this.vistaFileSynchCapability = vistaFileSynchCapability;
    }

    /**
     * getManagedItemCapability
     * @return the managedItemCapability
     */
    public ManagedItemCapability getManagedItemCapability() {
        return managedItemCapability;
    }

    /**
     * setManagedItemCapability
     * @param managedItemCapability the managedItemCapability to set
     */
    public void setManagedItemCapability(ManagedItemCapability managedItemCapability) {
        this.managedItemCapability = managedItemCapability;
    }
}
