package gov.va.med.pharmacy.peps.service.common.capability;

import java.io.IOException;
import java.util.Date;
import java.util.List;

import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.service.common.utility.NdfUpdateProcessFile;

/**
 * NdfUpdateProcessCapability contains the logic for putting the NDFUpdate Data into an update file and updating references 
 * in PPS_N to tie that data to the update file.
 * 
 */
public interface NdfUpdateProcessCapability {

    /**
     * Process PMI Data.
     * 
     * @param ndfUpdateFile the ndf update file
     * @throws IOException 
     */
    void processPemData(NdfUpdateProcessFile ndfUpdateFile) throws IOException;
    
    /**
     * Process PMI Data.
     * 
     * @param ndfUpdateFile the ndf update file
     * @throws IOException Thrown if issues writing to the file.
     */
    void processWarningLabelData(NdfUpdateProcessFile ndfUpdateFile) throws IOException;
    
    /**
     * Process VA Product Data.
     * 
     * @param ndfUpdateFile the ndf update file
     * @throws IOException Thrown if issues writing to the file.

     */
    void processVaProcuctMapData(NdfUpdateProcessFile ndfUpdateFile) throws IOException;
    
    /**
     * Process 5000 series data.
     * @param startDTM 
     *
     * @param ndfUpdateFile the ndf update file
     * @throws IOException Thrown if issues writing to the file.
     */
    public void processMiscellaneousNdfDataItems(Date startDTM, NdfUpdateProcessFile ndfUpdateFile) throws IOException;
    
    /**
     * Write the NDF Data to the specified file in HL7 format. 
     * @param startDate Date of the earliest data to include in the file.
     * @param ndfUpdateFile the NDFUpdateProcessFile in which to put the data.
     * @throws IOException Thrown if issues writing to the file.
     */
    public void processNdfData(Date startDate, NdfUpdateProcessFile ndfUpdateFile) throws IOException;
    
    /**
     * Process default text data.
     *
     * @param ndfUpdateFile the ndf update file
     * @throws IOException Thrown if issues writing to the file.
     * @throws Exception Thrown if issues other than writing to the file.
     */
    public void processTextItems(NdfUpdateProcessFile ndfUpdateFile) throws IOException;
    
    /**
     * Update the ndf_update_file_fk field with seq number for this processing cycle.
     *
     * @param ndfUpdateFileVo the ndf update file
     */
    public void updateNdfUpdateFileFk(NdfUpdateFileVo ndfUpdateFileVo);

    /**
     * Remove the file associations with all other data.
     *
     * @param fileSeqNo the file seq no to remove
     */
    public void removeAllFileAssociations(Long fileSeqNo);    
    
    /**
     * Remove the ndf_update_file_fk field for provided seq number.
     *
     * @param fileSeqNo the file seq no to remove
     */
    public void removeNdfUpdateFileFk(Long fileSeqNo);
    
    /**
     * Updates the File reference on the Rematch data. 
     * @param startDate Date of the earliest data to include in the file
     * @param ndfUpdateFileVo the ndf update file to associate to the rematch data.
     */
    public void updateRematchNdfUpdateFileFk(Date startDate, NdfUpdateFileVo ndfUpdateFileVo);
    
    /**
     * Remove/Undo the File reference from the Rematch data - used in cases where the file errored or was rejected.  
     * @param fileSeqNo the sequence of the NDF Update file to disassociate with the rematch data  
     */
    public void removeRematchNdfUpdateFileFk(Long fileSeqNo);
    
    /**
     * Returns Update File message data for the preview reports.
     * @param startDTM Date of the earliest data to include in the file
     * @return List<String> List containing message data
     */
    public List<String> getPreviewUpdateFileReportData(Date startDTM);
}
