package gov.va.med.pharmacy.peps.service.common.capability;

import gov.va.med.pharmacy.peps.common.exception.ValidationException;
import gov.va.med.pharmacy.peps.common.exception.ValueObjectValidationException;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.common.vo.PaginatedList;
import gov.va.med.pharmacy.peps.common.vo.StatusVo;
import gov.va.med.pharmacy.peps.common.vo.UpdateSearchCriteria;
import gov.va.med.pharmacy.peps.common.vo.UserVo;

import java.util.Date;
import java.util.List;

/**
 * Contains the main logic to create the NDF Update File and send email notifications.
 */
public interface NdfUpdateCapability {

    /**
     * Main process of the NDF Update Job that creates the update file and associated metadata.
     *
     * @param user the user
     */    
    public void process (UserVo user);
    
    /**
     * Retrieve all NdfUpdateFileVo.
     * @return List<FileUpdateVo> updateFiles
     */
    List<NdfUpdateFileVo> retrieveFileUpdates();

    /**
     * Retrieve paged NdfUpdateFileVo.
     *
     * @param searchCriteria
     *            the search criteria
     * @return List<FileUpdateVo> updateFiles
     */
    PaginatedList<NdfUpdateFileVo> retrievePagedFileUpdates(UpdateSearchCriteria searchCriteria);

    /**
     * Retrieve NdfUpdateFileVo ready for approval/rejection.
     * @return List<NdfUpdateFileVo> pendingUpdateFiles
     */
    List<NdfUpdateFileVo> retrievePendingUpdate();

    /**
     * Update the given NdfUpdateFileVo in the database.
     * 
     * @param updatedFile
     *            ManagedItemVo to update
     * @param user
     *            UserVo updating item
     */
    void updateNdfUpdateFile(NdfUpdateFileVo updatedFile, UserVo user);

    /**
     * Update the given NdfUpdateFileVo in the database and send an e-mail
     * notification to the appropriate recipients.
     *
     * @param updateFile the update file
     */ 
    void ndfUpdateFileNotification(NdfUpdateFileVo updateFile);

    /**
     * Retrieve NdfUpdateFileVo by NDFUpdateFileId.
     * @param id NDFUpdateFileId 
     * @return NdfUpdateFileVo The found file row, or null if the id is null or no row found with that id.
     */
    NdfUpdateFileVo retrieveById(Long id);

    /**
     * Retrieve a list of FileUpdateVo with initiated status.  Realistically, there should only ever be one.
     *
     * @return List<FileUpdateVo> initiatedUpdateFiles
     */
    List<NdfUpdateFileVo> retrieveInitiatedUpdate();

    /**
     * Insert the given NdfUpdateFileVo in the database.
     *
     * @param insertFile the insert file
     * @param user            UserVo updating item
     * @return the ndf update file vo
     */
    NdfUpdateFileVo insertNdfUpdateFile(NdfUpdateFileVo insertFile, UserVo user);

    /**
     * Retrieve all the status values (EPL_STATUS).
     * @return List<StatusVo> all update file statuses
     */
    List<StatusVo> retrieveAllFileStatuses();  
    
    /**
     * Transmit update file to production - Moves the file to the production folder, sends an email notification, and stores 
     * the NdfUpdateFileVo with an updated status.  
     *
     * @param updateFile the update file
     * @param user the user
     * @throws ValidationException Thrown if File transfer is turned off in the National Settings, if the file transmission 
     *      failed, or some other error was encountered.
     */    
    public void transmitUpdateFileToProd(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException;
    
    /**
     * Transmit update file to test - Moves the file to the test folder, sends an email notification, and stores 
     * the NdfUpdateFileVo with an updated status.  
     *
     * @param updateFile the update file
     * @param user the user
     * @throws ValidationException Thrown if File transfer is turned off in the National Settings, if the file transmission 
     *      failed, or some other error was encountered.
     */    
    public void transmitUpdateFileToTest(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException;    

    /**
     * Transmit update file to reject - Renames and moves the file to the reject folder, sends an email notification, 
     * and stores the NdfUpdateFileVo with an updated status.  
     *
     * @param updateFile the update file
     * @param user the user
     * @throws ValidationException Thrown if File transfer is turned off in the National Settings, if the file transmission 
     *      failed, or some other error was encountered.
     */       
    public void transmitUpdateFileToReject(NdfUpdateFileVo updateFile, UserVo user) throws ValidationException;    
    
    /**
     * Retrieves the latest NdfUpdateFileVo, meaning the one with the max file id.
     *
     * @return the NdfUpdateFileVo with the highest file id
     */    
    NdfUpdateFileVo retrieveCurrentStatus();

    /**
     * Creates and stores an NdfUpdateFileVo with {@code StatusVo.INITIATED} status.
     *
     * @param user the user
     * @return NdfUpdateFileVo the saved NdfUpdateFileVo with status {@code StatusVo.INITIATED}
     */
    NdfUpdateFileVo createInitiatedFile(UserVo user);
    
    /**
     * Approve.  Updates the status and transmits the file to the production folder.
     *
     * @param toUpdate the to update
     * @param user the user
     * @throws ValidationException the validation exception
     */
    public void approve(NdfUpdateFileVo toUpdate, UserVo user) throws ValidationException;
    
    /**
     * Reject.  Updates the status, moves the file to the reject folder, and remove the associations between the File 
     * and the data.
     *
     * @param toUpdate the to update
     * @param user the user
     * @throws ValidationException the validation exception
     */
    public void reject(NdfUpdateFileVo toUpdate, UserVo user) throws ValidationException;

}
