package gov.va.med.pharmacy.peps.service.common.update.impl;

import gov.va.med.pharmacy.peps.common.vo.DrugClassVo;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.ManagedDataVo;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.RequestItemStatus;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.common.vo.VuidStatusHistoryVo;
import gov.va.med.pharmacy.peps.common.vo.diff.Difference;
import gov.va.med.pharmacy.peps.external.common.preencapsulation.utility.update.item.DrugClassConverter;
import gov.va.med.pharmacy.peps.external.common.utility.MumpsConverter;
import gov.va.med.pharmacy.peps.external.common.vo.outbound.common.ItemAction;
import gov.va.med.pharmacy.peps.service.common.update.NdfFileSyncProcessor;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;


/**
 * This class implements methods which populate the new NDF outgoing difference table using the existing vaDispenseUnitSyncRequest object
 * for vista file 50.605 definition and KID build output for adds and modifications of PPS VA drug class data.
 *
 * @author serverbutts
 * @version 1.0
 */
public class VADrugClassNdfFileSyncProcessor extends AbstractSimpleNdfFileSyncProcessor implements NdfFileSyncProcessor {

    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(VADrugClassNdfFileSyncProcessor.class);

    private static final String MASTER_VUID_FIELD_NUMBER = "99.98";
    private static final String VUID_FIELD_NUMBER = "99.99";
    private static final String EFFECTIVE_DATE_TIME_FIELD_NUMBER = "99.991,.01";
    private static final String EFFECTIVE_DATE_STATUS_FIELD_NUMBER = "99.991,.02";
    private static final String CLASSIFICATION_FIELD_NUMBER = "1";
    private static final String PARENT_CLASS_FIELD_NUMBER = "2";
    private static final String TYPE_FIELD_NUMBER = "3";
    
    /**
     * The field number for the VA Drug Class Description field.
     */
    public static final String DESCRIPTION_FIELD_NUMBER = "4";

    private static final String WHITE_SPACE = " ";

    public static final Set<FieldKey> FIELDS = Collections.unmodifiableSet(new LinkedHashSet<FieldKey>(Arrays
    .asList(new FieldKey[] {FieldKey.CODE, FieldKey.CLASSIFICATION, FieldKey.PARENT_DRUG_CLASS,
                            FieldKey.CLASSIFICATION_TYPE,FieldKey.DESCRIPTION, FieldKey.ITEM_STATUS})));

    public VADrugClassNdfFileSyncProcessor(Set<FieldKey> fields, String fileNumber, String fieldAddNumber) {
        super(fields, fileNumber, fieldAddNumber);
    }

    @Override
    public void processNew(ManagedDataVo managedData) {

    }

    @Override
    public void processModified(ManagedDataVo managedData) {

    }

    @Override
    public void processNew(ManagedItemVo managedItem, UserVo user) {
        // Inserts new record into EplNdfOutgoingDifferences table.

        String userName = null;
        String vUID = null;
        String masterEntryForVUID = null;

        DrugClassVo vaDrugClass = (DrugClassVo) managedItem;

        userName  = user.getUsername();

        vUID = StringUtils.isNotBlank(vaDrugClass.getVuid()) ? vaDrugClass.getVuid() : WHITE_SPACE;

        masterEntryForVUID = vaDrugClass.isMasterEntryForVuid() ? "1" : "0";

        if ((StringUtils.isNotEmpty(vaDrugClass.getValue())) && (RequestItemStatus.APPROVED.equals(vaDrugClass.getRequestItemStatus()) )) {

            //code
            insertNewElement(vaDrugClass.getDrugClassIen(), vaDrugClass.getCode(), userName);

            //classification - required
            insertNewChildElement(vaDrugClass.getDrugClassIen(), vaDrugClass.getClassification(), userName, CLASSIFICATION_FIELD_NUMBER);

            //parent drug class
            if (vaDrugClass.getParentDrugClass() != null) {
                insertNewChildElement(vaDrugClass.getDrugClassIen(), vaDrugClass.getParentDrugClass().getDrugClassIen(), userName, PARENT_CLASS_FIELD_NUMBER);
            }

            //type - required
            insertNewChildElement(vaDrugClass.getDrugClassIen(), String.valueOf(vaDrugClass.getClassificationType().getCode()),
                    userName, TYPE_FIELD_NUMBER);

            //description
            if(vaDrugClass.getDescription() != null) {
                insertNewChildElement(vaDrugClass.getDrugClassIen(), vaDrugClass.getDescription(), userName, DESCRIPTION_FIELD_NUMBER);
            }

            // Master entry for vuid
            //TODO check if this is correct
            insertNewChildElement(vaDrugClass.getDrugClassIen(), masterEntryForVUID, userName, MASTER_VUID_FIELD_NUMBER);

            //VUID entry
            insertNewChildElement(vaDrugClass.getDrugClassIen(), vUID, userName, VUID_FIELD_NUMBER);

            //Effective Date
            if ((null != vaDrugClass.getEffectiveDates()) && (vaDrugClass.getEffectiveDates().size()>0)) {

                int counter = 0;

                StringBuffer sBuf = new StringBuffer();

                for (VuidStatusHistoryVo effectiveDate : vaDrugClass.getEffectiveDates()) {

                    counter += 1;

                    String vistaIENChild = sBuf.append(vaDrugClass.getDrugClassIen()).append(",").append(counter).toString();

                    insertNewChildElement(vistaIENChild, MumpsConverter.convertDate(effectiveDate.getEffectiveDateTime()), userName, EFFECTIVE_DATE_TIME_FIELD_NUMBER);
                    //required
                    insertNewChildElement(vistaIENChild, effectiveDate.getItemStatus().isActive() ? "1" : "0", userName, EFFECTIVE_DATE_STATUS_FIELD_NUMBER);
                }
            } else {
                LOG.error("Drug Class Effective Date Cannot be Null. This field should be populated as it's a required Field.");
            }
        }
     }


    @Override
    public void processModified(ManagedItemVo managedItem, UserVo user, Collection<Difference> differences) {

        // Inserts either new record or updates older one in the EplNdfOutgoingDifferences table.

        Map<FieldKey, Difference> setDifference = toSetDifference(differences);

        boolean hasDifferences = DrugClassConverter.hasNewOrModifiedFields(FIELDS, setDifference, ItemAction.MODIFY);
        if (hasDifferences) {

            DrugClassVo vaDrugClass = (DrugClassVo) managedItem;

            if ( (StringUtils.isNotEmpty(vaDrugClass.getValue())) && (RequestItemStatus.APPROVED.equals(vaDrugClass.getRequestItemStatus()) )) {

                //parent drug class
                if (DrugClassConverter.fieldWasModified(FieldKey.PARENT_DRUG_CLASS,differences)){
                    if(vaDrugClass.getParentDrugClass() != null){
                        insertModifiedElement(vaDrugClass.getDrugClassIen(), vaDrugClass.getParentDrugClass().getDrugClassIen(),
                                              user.getUsername(), PARENT_CLASS_FIELD_NUMBER, null);
                    }else{
                        insertModifiedElement(vaDrugClass.getDrugClassIen(), WHITE_SPACE, user.getUsername(), PARENT_CLASS_FIELD_NUMBER, null);
                    }
                }

                //type - required
                if (DrugClassConverter.fieldWasModified(FieldKey.CLASSIFICATION_TYPE, differences)) {
                    insertModifiedElement(vaDrugClass.getDrugClassIen(), String.valueOf(vaDrugClass.getClassificationType().getCode()), user.getUsername(), TYPE_FIELD_NUMBER, null);
                }

                //description
                if(DrugClassConverter.fieldWasModified(FieldKey.DESCRIPTION, differences)){
                    if(vaDrugClass.getDescription() != null){
                        insertModifiedElement(vaDrugClass.getDrugClassIen(), vaDrugClass.getDescription(), user.getUsername(), DESCRIPTION_FIELD_NUMBER, null);
                    }else{
                        insertModifiedElement(vaDrugClass.getDrugClassIen(), WHITE_SPACE, user.getUsername(), DESCRIPTION_FIELD_NUMBER, null);
                    }
                }
            }
        }
    }
}
