package gov.va.med.pharmacy.peps.service.common.update.impl;

import gov.va.med.pharmacy.peps.common.utility.VistaFileNumber;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.ManagedDataVo;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.PackageSizeVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.common.vo.diff.Difference;
import gov.va.med.pharmacy.peps.domain.common.model.EplNdfOutgoingDifferences;
import gov.va.med.pharmacy.peps.external.common.preencapsulation.utility.update.item.PackageSizeConverter;
import gov.va.med.pharmacy.peps.external.common.utility.MumpsConverter;
import gov.va.med.pharmacy.peps.external.common.vo.outbound.common.ItemAction;
import gov.va.med.pharmacy.peps.service.common.update.NdfFileSyncProcessor;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.springframework.stereotype.Service;

/**
 * Used to save additions/modifications to Package Size objects so that the information will be sent out 
 * in the NDF Update File.
 * 
 * @author vhaisaduffg
 *
 */
@Service
public class PackageSizeNdfFileSyncProcessor extends AbstractSimpleNdfFileSyncProcessor implements NdfFileSyncProcessor {

//    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager
//            .getLogger(PackageSizeNdfFileSyncProcessor.class);
    /**
     *  The Inactivation Date Field Number
     */
    private static final String INACTIVATION_DATE_FIELD_NUMBER = "1";

    /**
     * Constructor for creating a new class.
     * 
     * @param fields the attributes in a Package Size object
     * @param fileNumber the VistA file number for a Package Size
     * @param fieldAddNumber the field number for the IEN
     */
    public PackageSizeNdfFileSyncProcessor(@SuppressWarnings("rawtypes") Set<FieldKey> fields, String fileNumber, 
            String fieldAddNumber) {
        super(fields, fileNumber, fieldAddNumber);
    }

    /**
     * This method is not to be used.  It is only here to satisfy the NdfFileSyncProcessor interface.
     * 
     * @param managedData the new Package Size data
     */
    @Override
    public final void processNew(ManagedDataVo managedData) {
        // Not used
    }

    /**
     * Saves a new Package Size object's information for inclusion in the next NDF Update File.
     * 
     * @param managedItem the Package Size information
     * @param user the user who inserted the Package Size information
     */
    @Override
    public final void processNew(ManagedItemVo managedItem, UserVo user) {
        PackageSizeVo packageSizeVo = (PackageSizeVo) managedItem;
        
        // There is currently no way to "Approve" a new Package Size.  So do not even check for Approval.
        EplNdfOutgoingDifferences diffObj = new EplNdfOutgoingDifferences();

        // IEN
        diffObj.setVistaIen(packageSizeVo.getNdfPackageSizeIen());
        diffObj.setNewValue(packageSizeVo.getPackageSize().toString()); 
        diffObj.setCreatedBy(user.getUsername());
        processInsert(diffObj); 
    }

    /**
     * This method is not to be used.  It is only here to satisfy the NdfFileSyncProcessor interface.
     * 
     * @param managedData the modified Package Size data
     */
    @Override
    public final void processModified(ManagedDataVo managedData) {
        // Not used
    }

    /**
     * This method would be used to update a Package Size.  But since PPS-N currently does not have a way
     * to update a Package Size, this method will never be called.
     * 
     * @param managedItem the modified Package Size data
     * @param user the user updating the information
     * @param differences the differences between the old a new Package Size information
     */
    @Override
    public final void processModified(ManagedItemVo managedItem, UserVo user, Collection<Difference> differences) {
        @SuppressWarnings("rawtypes") Map<FieldKey, Difference> setDifference =  super.toSetDifference(differences);
        boolean hasDifference = PackageSizeConverter.hasNewOrModifiedFields(getFields(), setDifference, ItemAction.MODIFY);

        PackageSizeVo packageSizeVo = (PackageSizeVo) managedItem;

        if (hasDifference) {
            EplNdfOutgoingDifferences diffObj = new EplNdfOutgoingDifferences();

            diffObj.setVistaIen(packageSizeVo.getNdfPackageSizeIen());
            diffObj.setVistaFieldNumber(INACTIVATION_DATE_FIELD_NUMBER);
            if(packageSizeVo.getInactivationDate() != null){
                diffObj.setNewValue(MumpsConverter.convertDate(packageSizeVo.getInactivationDate()));                 
            }  
            else{
                //passing an empty string because this is a non null field
                diffObj.setNewValue(" ");                  
            }
            diffObj.setCreatedBy(user.getUsername());
            processUpdate(diffObj);
        }
    }
}
