package gov.va.med.pharmacy.peps.service.common.update.impl;

import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.IngredientVo;
import gov.va.med.pharmacy.peps.common.vo.ManagedDataVo;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.RequestItemStatus;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.common.vo.VuidStatusHistoryVo;
import gov.va.med.pharmacy.peps.common.vo.diff.Difference;
import gov.va.med.pharmacy.peps.domain.common.model.EplNdfOutgoingDifferences;
import gov.va.med.pharmacy.peps.external.common.preencapsulation.utility.update.item.DrugIngredientConverter;
import gov.va.med.pharmacy.peps.external.common.utility.MumpsConverter;
import gov.va.med.pharmacy.peps.external.common.vo.outbound.common.ItemAction;
import gov.va.med.pharmacy.peps.service.common.update.NdfFileSyncProcessor;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;


/**
 * This class implements methods which populate the new NDF outgoing difference table using the existing vaDispenseUnitSyncRequest object
 * for vista file 50.416 definition and KID build output for adds and modifications of PPS VA drug Ingredients data.
 * 
 * @author serverbutts
 * @version 1.0
 */
public class DrugIngredientsNdfFileSyncProcessor extends AbstractSimpleNdfFileSyncProcessor implements NdfFileSyncProcessor {
   
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(DrugIngredientsNdfFileSyncProcessor.class);
    
    private static final String MASTER_VUID_FIELD_NUMBER = "99.98";
    private static final String VUID_FIELD_NUMBER = "99.99";
    private static final String CHILD_FIELD_NUMBER_01 = "99.991,.01";
    private static final String CHILD_FIELD_NUMBER_02 = "99.991,.02";
    private static final String PRIMARY_INGREDIENT = "2";
    private static final String INACTIVATION_DATE = "3";
    
    private static final String WHITE_SPACE = " ";
    
    public static final Set<FieldKey> FIELDS = Collections.unmodifiableSet(new LinkedHashSet<FieldKey>(Arrays
    .asList(new FieldKey[] { FieldKey.CODE, FieldKey.INACTIVATION_DATE, FieldKey.PRIMARY_INGREDIENT, 
                             FieldKey.MASTER_ENTRY_FOR_VUID, FieldKey.ITEM_STATUS })));

   
    public DrugIngredientsNdfFileSyncProcessor(Set<FieldKey> fields, String fileNumber, String fieldAddNumber) {
        super(fields, fileNumber, fieldAddNumber);
     }
  
    @Override
    public void processNew(ManagedDataVo managedData) {
       
    }
    
    @Override
    public void processModified(ManagedDataVo managedData) {
     
    }

    @Override
    public void processNew(ManagedItemVo managedItem, UserVo user) {
        // Inserts new record into EplNdfOutgoingDifferences table.
       
        String userName = null;
        String vUID = null;
        String masterEntryForVUID = null;
       
        IngredientVo drugIngredients = (IngredientVo) managedItem;
        
        userName  = user.getUsername();
          
        vUID = StringUtils.isNotBlank(drugIngredients.getVuid()) ? drugIngredients.getVuid() : WHITE_SPACE;
        masterEntryForVUID = drugIngredients.getMasterEntryForVuid() ? "1" : "0";
        
        if ((StringUtils.isNotEmpty(drugIngredients.getValue())) && (RequestItemStatus.APPROVED.equals(drugIngredients.getRequestItemStatus()) )) {
                           
            insertNewElement(drugIngredients.getNdfIngredientIen(), drugIngredients.getValue(), userName);
            
            //primary ingredient
            if(drugIngredients.getPrimaryIngredient() != null)
                insertNewChildElement(drugIngredients.getNdfIngredientIen(), drugIngredients.getPrimaryIngredient().getNdfIngredientIen(),
                                      userName, PRIMARY_INGREDIENT);
            
            // Master entry for vuid
            insertNewChildElement(drugIngredients.getNdfIngredientIen(), masterEntryForVUID, userName, MASTER_VUID_FIELD_NUMBER);
            
            //VUID entry
            insertNewChildElement(drugIngredients.getNdfIngredientIen(), vUID, userName, VUID_FIELD_NUMBER);
            
            //Effective Date
            if (null != drugIngredients.getEffectiveDates() && drugIngredients.getEffectiveDates().size()>0) {
                
                int counter = 0;
                
                StringBuffer sBuf = new StringBuffer();

                for (VuidStatusHistoryVo effectiveDate : drugIngredients.getEffectiveDates()) {
                    
                    counter += 1;
                    
                    String vistaIENChild = sBuf.append(drugIngredients.getNdfIngredientIen()).append(",").append(counter).toString();

                    insertNewChildElement(vistaIENChild, MumpsConverter.convertDate(effectiveDate.getEffectiveDateTime()), userName, CHILD_FIELD_NUMBER_01);
                    
                    insertNewChildElement(vistaIENChild, effectiveDate.getItemStatus().isActive() ? "1" : "0", userName, CHILD_FIELD_NUMBER_02);
                }
               } else {             
                LOG.error("Drug Ingredients Effective Date Cannot be Null. This field should be populated as it's a required Field.");
            }         
        }
    }

   
    @Override
    public void processModified(ManagedItemVo managedItem, UserVo user, Collection<Difference> differences) {

        // Inserts either new record or updates older one in the EplNdfOutgoingDifferences table.

        Map<FieldKey, Difference> setDifference = toSetDifference(differences);

        boolean hasDifferences = DrugIngredientConverter.hasNewOrModifiedFields(FIELDS, setDifference, ItemAction.MODIFY);
        if (hasDifferences) {                 

            IngredientVo drugIngredients = (IngredientVo) managedItem;

            if ((StringUtils.isNotEmpty(drugIngredients.getValue())) && (RequestItemStatus.APPROVED.equals(drugIngredients.getRequestItemStatus()) )) {

               //diff object uses item status when modification is for activation/inactivation  
                if(DrugIngredientConverter.fieldWasModified(FieldKey.ITEM_STATUS, differences)){
                    if(drugIngredients.getInactivationDate() != null){
                        insertModifiedElement(drugIngredients.getNdfIngredientIen(),
                            MumpsConverter.convertDate(drugIngredients.getInactivationDate()), user.getUsername(),INACTIVATION_DATE,null);
                    }else{
                        insertModifiedElement(drugIngredients.getNdfIngredientIen(),WHITE_SPACE, user.getUsername(),INACTIVATION_DATE,null);
                    }
                }
                
                if(DrugIngredientConverter.fieldWasModified(FieldKey.PRIMARY_INGREDIENT,differences)){
                    if(drugIngredients.getPrimaryIngredient() != null){
                        insertModifiedElement(drugIngredients.getNdfIngredientIen(),
                            drugIngredients.getPrimaryIngredient().getNdfIngredientIen(), user.getUsername(),PRIMARY_INGREDIENT,null);
                    }else{
                        insertModifiedElement(drugIngredients.getNdfIngredientIen(),WHITE_SPACE, user.getUsername(),PRIMARY_INGREDIENT,null);
                    }
                }
                
                if(DrugIngredientConverter.fieldWasModified(FieldKey.MASTER_ENTRY_FOR_VUID,differences)){
                    insertModifiedElement(drugIngredients.getNdfIngredientIen(),
                        drugIngredients.getMasterEntryForVuid() ? "1" : "0", user.getUsername(),MASTER_VUID_FIELD_NUMBER,null);
                }
            }
        }
    } 
}
