package gov.va.med.pharmacy.peps.service.common.update.impl;

import gov.va.med.pharmacy.peps.common.vo.DrugDrugInteractionVo;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.ItemStatus;
import gov.va.med.pharmacy.peps.common.vo.ManagedDataVo;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.RequestItemStatus;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.common.vo.diff.Difference;
import gov.va.med.pharmacy.peps.domain.common.model.EplNdfOutgoingDifferences;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.DrugDrugInteractionConverter;
import gov.va.med.pharmacy.peps.external.common.utility.AbstractConverter;
import gov.va.med.pharmacy.peps.external.common.utility.MumpsConverter;
import gov.va.med.pharmacy.peps.external.common.vo.outbound.common.ItemAction;
import gov.va.med.pharmacy.peps.service.common.update.NdfFileSyncProcessor;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;


/**
 * This class implements methods which populate the new NDF outgoing difference table using the existing DrugDrugInteractionVo object
 * for vista file 56 definition and KID build output for adds and modifications of PPS VA drug drug interaction data.
 * 
 * @author serverbutts
 * @version 1.0
 */
public class DrugDrugInteractionNdfFileSyncProcessor extends AbstractSimpleNdfFileSyncProcessor implements NdfFileSyncProcessor {
   
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(DrugDrugInteractionNdfFileSyncProcessor.class);    
     
    private static final String INGREDIENT1_FIELD_NUMBER = "1";
    private static final String INGREDIENT2_FIELD_NUMBER = "2";
    private static final String SEVERITY_FIELD_NUMBER = "3";
    private static final String NATIONALLY_ENTERED_FIELD_NUMBER = "4";
    private static final String TOTAL_INDEXES_ENTERED_FIELD_NUMBER = "5";
    private static final String LOCALLY_EDITED_FIELD_NUMBER = "6";
    private static final String INACTIVATION_DATE_FIELD_NUMBER = "7";
    private static final String WHITE_SPACE = " ";
    
   
    public static final Set<FieldKey> FIELDS = Collections.unmodifiableSet(new LinkedHashSet<FieldKey>(Arrays
    .asList(new FieldKey[] { FieldKey.VALUE,FieldKey.SEVERITY, FieldKey.INACTIVATION_DATE, FieldKey.ITEM_STATUS })));

   
    public DrugDrugInteractionNdfFileSyncProcessor(Set<FieldKey> fields, String fileNumber, String fieldAddNumber) {
        super(fields, fileNumber, fieldAddNumber);
        // TODO Auto-generated constructor stub
    }
  

      
    @Override
    public void processNew(ManagedDataVo managedData) {
        
       
    }
    
    @Override
    public void processModified(ManagedDataVo managedData) {
        
     
    }

   
    @Override
    public void processNew(ManagedItemVo managedItem, UserVo user) {
        // Inserts new record into EplNdfOutgoingDifferences table.

        // parse out the data and populate the EplNdfOutgoingDifferences obj for persistence.

        String userName = null;        

        try {
            
            DrugDrugInteractionVo drugInteraction = (DrugDrugInteractionVo) managedItem;

            userName  = StringUtils.isNotEmpty(user.getUsername())? user.getUsername():"360";              


            if ( (StringUtils.isNotEmpty(drugInteraction.getValue())) && (RequestItemStatus.APPROVED.equals(drugInteraction.getRequestItemStatus()) )) {

                EplNdfOutgoingDifferences differencesObj = new EplNdfOutgoingDifferences();

                differencesObj.setNewValue(drugInteraction.getValue());               

                // need to check what might go in the else clause.
                if(StringUtils.isNotEmpty(drugInteraction.getDrugDrugInteractionIen())){

                    differencesObj.setVistaIen(drugInteraction.getDrugDrugInteractionIen());
                } 
                else{
                    LOG.error("Drug interaction IEN can't be null.");
                }


                differencesObj.setCreatedBy(userName);               


                insertNewElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getValue(), userName);

                // Ingredient 1
                insertNewChildElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getIngredient1().getNdfIngredientIen(), userName, INGREDIENT1_FIELD_NUMBER);

                // Ingredient 2
                insertNewChildElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getIngredient2().getNdfIngredientIen(), userName, INGREDIENT2_FIELD_NUMBER);          

                // Severity 
                insertNewChildElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getDrugSeverity().getId(), userName, SEVERITY_FIELD_NUMBER);

                // naturally entered 
                insertNewChildElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getNationallyEntered(), userName, NATIONALLY_ENTERED_FIELD_NUMBER);

                // total indexed  - Current not needed
                //insertNewChildElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getTotalIndexes(), userName, TOTAL_INDEXES_ENTERED_FIELD_NUMBER);

                // total indexed - Current not needed
                //insertNewChildElement(drugInteraction.getDrugDrugInteractionIen(), drugInteraction.getLocallyEdited(), userName, LOCALLY_EDITED_FIELD_NUMBER);

            }
        } 
        catch (Exception e) {
            // catpure  the exception here.
            LOG.error("Error occured in processNew while inserting new drug drug interaction related information into epl_ndf_outgoing_differences table.");

        }


    }

   
    @Override
    public void processModified(ManagedItemVo managedItem, UserVo user, Collection<Difference> differences) {

        // Inserts either new record or updates older one in the EplNdfOutgoingDifferences table.

        String userName = null;    

        try {

            userName  = StringUtils.isNotEmpty(user.getUsername())? user.getUsername():"360";   

            Map<FieldKey, Difference> setDifference = toSetDifference(differences);
            
            if (ItemStatus.INACTIVE.equals(AbstractConverter.getOldValue(FieldKey.ITEM_STATUS, setDifference))
                && ItemStatus.ACTIVE.equals(AbstractConverter.getNewValue(FieldKey.ITEM_STATUS, setDifference))) { // reactivate

                setDifference.put(FieldKey.INACTIVATION_DATE, new Difference(FieldKey.INACTIVATION_DATE, new Date(), null));
            }

            boolean hasDifferences = DrugDrugInteractionConverter.hasNewOrModifiedFields(FIELDS, setDifference, ItemAction.MODIFY);


            if (hasDifferences) {                 

                DrugDrugInteractionVo drugInteraction = (DrugDrugInteractionVo) managedItem;

                if ( RequestItemStatus.APPROVED.equals(drugInteraction.getRequestItemStatus()) ) {


                    // check severity change.

                    if (DrugDrugInteractionConverter.fieldWasModified(FieldKey.SEVERITY, differences)){

                        LinkedHashMap<String, String> severityMap = super.getSeverityMap();

                        Difference diff = setDifference.get(FieldKey.SEVERITY);
                        
                        String newValue = severityMap.get(diff.getNewValueShortString());
                        
                        String oldValue =severityMap.get(diff.getOldValueShortString());

                        insertModifiedElement(drugInteraction.getDrugDrugInteractionIen(),newValue, userName,SEVERITY_FIELD_NUMBER,oldValue );
                    }

                    else if (DrugDrugInteractionConverter.fieldWasModified(FieldKey.ITEM_STATUS, differences)){


                        Difference diff = setDifference.get(FieldKey.INACTIVATION_DATE);

                        String priorInactivatedDate = convertPriorInactivationDate(diff);

                        String inactivatedDate = convertInactivationDate(diff, drugInteraction);   

                        insertModifiedElement(drugInteraction.getDrugDrugInteractionIen(), inactivatedDate, userName, INACTIVATION_DATE_FIELD_NUMBER, priorInactivatedDate);

                    }


                }

            }


        } 
        catch (Exception e) {
            // catpure  the exception here.
            LOG.error("Error occured in processModified, while saving drug drug interaction related information into epl_ndf_outgoing_differences table.");

        }

    }
    
    


    private String convertPriorInactivationDate(Difference diff){

        return diff != null ? MumpsConverter.convertDate((Date)diff.getOldValue()): null;

    }

    private String convertInactivationDate(Difference diff, DrugDrugInteractionVo drugInteractionVo){

        return diff == null ? MumpsConverter.convertDate(drugInteractionVo.getInactivationDate()) : WHITE_SPACE;

    }



    
    

}
