package gov.va.med.pharmacy.peps.domain.common.utility.impl;


import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.dao.EplProductDao;
import gov.va.med.pharmacy.peps.domain.common.model.EplProductDo;
import gov.va.med.pharmacy.peps.domain.common.utility.RxNormCapability;
import gov.va.med.pharmacy.peps.domain.common.utility.client.rxnorm.Properties;
import gov.va.med.pharmacy.peps.domain.common.utility.client.rxnorm.RxNormClient;
import gov.va.med.pharmacy.peps.domain.common.utility.client.rxnorm.RxNormData;

import java.util.Date;
import java.util.List;

import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;


/**
 * This class handles the process of keeping RxNorm data up to date on the EPL_PRODUCTS table.
 * It uses RxNormClient to execute two web service calls to NLM and the product DAO to populate 
 * the data into the appropriate columns.
 * 
 * @author K. Edgecomb
 *
 */
@Component
public class RxNormCapabilityImpl implements RxNormCapability {

    private RxNormClient rxNormClient;
    private EplProductDao eplProductDao;
    
    
    /* (non-Javadoc)
	 * @see gov.va.med.pharmacy.peps.domain.common.utility.impl.RxNormCapability#populateRxNormForProduct(gov.va.med.pharmacy.peps.common.vo.UserVo, java.lang.Long)
	 */
    @Override
	@Transactional
    public long populateRxNormForProduct(UserVo user, Long productId) {
    	Date runDate = new Date();
        EplProductDo product = eplProductDao.retrieve(productId);
        populateAndPersist(user, product, runDate);
        return 0;
    }

    
	/* (non-Javadoc)
	 * @see gov.va.med.pharmacy.peps.domain.common.utility.RxNormCapability#fetchRxNormUpdateProducts()
	 */
	@Transactional
    public List<EplProductDo> fetchRxNormUpdateProducts() {
        List<EplProductDo> products = eplProductDao.retrieveProductsNotExcludedFromRxNormUpdate();
        return products;
    }

    
    /* (non-Javadoc)
     * @see gov.va.med.pharmacy.peps.domain.common.utility.RxNormCapability#populateAndPersist(gov.va.med.pharmacy.peps.common.vo.UserVo, gov.va.med.pharmacy.peps.domain.common.model.EplProductDo, java.util.Date)
     */
    public void populateAndPersist(UserVo user, EplProductDo product, Date runDate) {

    	RxNormData rxNormData;
        
        // Call the NLM web service with VUID to get a CUID if available
        rxNormData = rxNormClient.getRxcuid(product.getVuid());
        
        // If the web service returns rxNorm CUID data
        if (rxNormData != null && rxNormData.getIdGroup() != null && rxNormData.getIdGroup().getRxnormId() != 0) {
        	
        	// Perform a second web service call using the CUID to get all rxnorm data
            rxNormData = rxNormClient.getRxNorm(rxNormData.getIdGroup().getRxnormId());
            
            Properties properties = rxNormData.getProperties();
            
            if (properties != null) {
            	product.setRxNormLastUpdateDate(runDate);
            	
            	// If the returned data is the same as the current product data, don't push to Vista
            	if (propertiesMatchProduct(properties, product)) {
            		// This is not supposed to update the database.
            		// Update to suppress vista push
            		// eplProductDao.update(product, user);
            	} else {
            		copyProperites(properties, product);
            		// Update to trigger vista push
            		eplProductDao.update(product, user);
            	}
            }
        }
    }

    
    /**
     * Copies the passed RxNorm properties into the appropriate instance variables on the passed product
     * 
     * @param properties
     * @param product
     */
    private void copyProperites(Properties properties, EplProductDo product) {
        product.setRxNormLanguage(properties.getLanguage());
        product.setRxNormName(properties.getName());
        product.setRxNormRxcui(properties.getRxcui());
        product.setRxNormSuppress(properties.getSuppress());
        product.setRxNormSyn(properties.getSynonym());
        product.setRxNormTty(properties.getTty());
        product.setRxNormUmlscui(properties.getUmlscui());
    }
    
    
    /**
     * Determines whether or not any of the RxNorm properties are actually different than those currently stored in the 
     * product from the EPL_PRODUCTS table.
     * 
     * @param properties
     * @param product
     * @return
     */
    private boolean propertiesMatchProduct(Properties properties, EplProductDo product) {
    	boolean match = true;
    	if (product.getRxNormRxcui() != null && !product.getRxNormRxcui().equals(properties.getRxcui())) {
    		match = false;
    	}
    	if (product.getRxNormRxcui() == null && properties.getRxcui() > 0) {
    		match = false;
    	}
    	if (product.getRxNormName() != null && !product.getRxNormName().equals(properties.getName())) {
    		match = false;
    	}
    	if (product.getRxNormName() == null && properties.getName() != null) {
    		match = false;
    	}
    	if (product.getRxNormSuppress() != null && !product.getRxNormSuppress().equals(properties.getSuppress())) {
    		match = false;
    	}
    	if (product.getRxNormSuppress() == null && properties.getSuppress() != null) {
    		match = false;
    	}
    	if (product.getRxNormTty() != null && !product.getRxNormTty().equals(properties.getTty())) {
    		match = false;
    	}
    	if (product.getRxNormTty() == null && properties.getTty() != null) {
    		match = false;
    	}
    	if (product.getRxNormUmlscui() != null && !product.getRxNormUmlscui().equals(properties.getUmlscui())) {
    		match = false;
    	}
    	if (product.getRxNormUmlscui() == null && properties.getUmlscui() != null) {
    		match = false;
    	}
    	return match;
    }


	/* (non-Javadoc)
	 * @see gov.va.med.pharmacy.peps.domain.common.utility.RxNormCapability#getRxNormClient()
	 */
	public RxNormClient getRxNormClient() {
		return rxNormClient;
	}
	/* (non-Javadoc)
	 * @see gov.va.med.pharmacy.peps.domain.common.utility.impl.RxNormCapability#setEplProductDao(gov.va.med.pharmacy.peps.domain.common.dao.EplProductDao)
	 */
	@Override
	public void setEplProductDao(EplProductDao eplProductDao) {
		this.eplProductDao = eplProductDao;
	}


	/* (non-Javadoc)
	 * @see gov.va.med.pharmacy.peps.domain.common.utility.RxNormCapability#getEplProductDao()
	 */
	public EplProductDao getEplProductDao() {
		return eplProductDao;
	}
	/* (non-Javadoc)
	 * @see gov.va.med.pharmacy.peps.domain.common.utility.impl.RxNormCapability#setRxNormClient(gov.va.med.pharmacy.peps.domain.common.utility.client.rxnorm.RxNormClient)
	 */
	@Override
	public void setRxNormClient(RxNormClient rxNormClient) {
		this.rxNormClient = rxNormClient;
	}

}
