package gov.va.med.pharmacy.peps.domain.common.utility.converter;

import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.RematchSuggestionVo;
import gov.va.med.pharmacy.peps.domain.common.model.DataObject;
import gov.va.med.pharmacy.peps.domain.common.model.EplRematchSuggestionDo;

/**
 * Convert to/from {@link RematchSuggestion} and {@link EplRematchSuggestionDo}.
 */
public class RematchSuggestionConverter extends Converter<RematchSuggestionVo, EplRematchSuggestionDo>{
    
    private NdfUpdateFileConverter ndfUpdateFileConverter;
    private MatchTypeConverter matchTypeConverter;
    
    /**
     * Fully copies data from the given RematchSuggestionVo into a {@link DataObject}.
     * 
     * @param data RematchSuggestionVo to convert
     * @return fully populated {@link DataObject}
     * 
     * @see gov.va.med.pharmacy.peps.domain.common.utility.converter.Converter#toDataObject(
     *      gov.va.med.pharmacy.peps.common.vo.ValueObject)
     */
    @Override
    protected EplRematchSuggestionDo toDataObject(RematchSuggestionVo data) {
        EplRematchSuggestionDo save = new EplRematchSuggestionDo();
        
        if(data.getEplRematchSuggestionId() != null ){
            save.setEplRematchSuggestionId(Long.valueOf(data.getEplRematchSuggestionId()));            
        }
        save.setNewIen(data.getNewIen());
        save.setOldIen(data.getOldIen()); 
        
        if (data.getMatchType() != null) {           
            save.setMatchType(matchTypeConverter.convert(data.getMatchType()));
        }
        
        if (data.getNdcUpdateFile() != null) { 
            save.setNdcUpdateFile(ndfUpdateFileConverter.convert(data.getNdcUpdateFile()));            
        }
    
        save.setMatchScope(data.getMatchScope());
        
    
        return save;
    }


    /**
     * Fully copies data from the given {@link DataObject} into a NdfUpdateFileVo.
     * <p>
     * Parent objects, if any, are minimally populated. Child objects, if any, are not populated at all. Any remaining
     * aggregated {@link ManagedItemVo} are minimally populated. If any of these item types are intended to be fully
     * populated, the appropriate {@link Converter#convert(DataObject)} should be called.
     * 
     * @param data {@link DataObject} to convert
     * @return fully populated NdfUpdateFileVo
     * 
     * @see gov.va.med.pharmacy.peps.domain.common.utility.converter.Converter#toValueObject(
     *      gov.va.med.pharmacy.peps.domain.common.model.DataObject)
     */
    @Override
    protected RematchSuggestionVo toValueObject(EplRematchSuggestionDo data) {
        RematchSuggestionVo vo = new RematchSuggestionVo();
        vo.setNewIen(data.getNewIen());
        vo.setOldIen(data.getOldIen());
        vo.setEplRematchSuggestionId(data.getEplRematchSuggestionId().toString());        
    
        if (data.getMatchType() != null) {
            vo.setMatchType(matchTypeConverter.convert(data.getMatchType()));            
        } 
        
        if (data.getNdcUpdateFile() != null) { 
            vo.setNdcUpdateFile(ndfUpdateFileConverter.convert(data.getNdcUpdateFile()));            
        }
        vo.setMatchScope(data.getMatchScope());
        return vo;
    }


    public NdfUpdateFileConverter getNdfUpdateFileConverter() {
        return ndfUpdateFileConverter;
    }

    
    public void setNdfUpdateFileConverter(NdfUpdateFileConverter ndfUpdateFileConverter) {
        this.ndfUpdateFileConverter = ndfUpdateFileConverter;
    }
    
    public MatchTypeConverter getMatchTypeConverter() {
        return matchTypeConverter;
    }

    
    public void setMatchTypeConverter(MatchTypeConverter matchTypeConverter) {
        this.matchTypeConverter = matchTypeConverter;
    }

}
