package gov.va.med.pharmacy.peps.domain.common.utility.converter;



import gov.va.med.pharmacy.peps.common.vo.DrugDrugInteractionVo;
import gov.va.med.pharmacy.peps.common.vo.DrugSeverityVo;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.ItemStatus;
import gov.va.med.pharmacy.peps.common.vo.PrimaryIngredientVo;
import gov.va.med.pharmacy.peps.common.vo.RequestItemStatus;
import gov.va.med.pharmacy.peps.common.vo.ValueObject;
import gov.va.med.pharmacy.peps.common.vo.diff.Difference;
import gov.va.med.pharmacy.peps.domain.common.dao.EplDdiSeverityDao;
import gov.va.med.pharmacy.peps.domain.common.dao.EplIngredientDao;
import gov.va.med.pharmacy.peps.domain.common.model.DataObject;
import gov.va.med.pharmacy.peps.domain.common.model.EplDdiSeverityDo;
import gov.va.med.pharmacy.peps.domain.common.model.EplDrugDrugInteractionDo;
import gov.va.med.pharmacy.peps.domain.common.model.EplIngredientDo;
import gov.va.med.pharmacy.peps.external.common.vo.outbound.common.ItemAction;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;


/**
 * @author serverbutts
 * Convert to/from {@link DrugDrugInteractionVo} and {@link EplDrugDrugInteractionDo}.
 */
public class DrugDrugInteractionConverter extends Converter<DrugDrugInteractionVo, EplDrugDrugInteractionDo> {

    private EplIngredientDao eplIngredientDao;

    private EplDdiSeverityDao eplDdiSeverityDao;

    private PrimaryIngredientConverter primaryIngredientConverter;

    private DrugSeverityConverter drugSeverityConverter;





    /**
     * Fully copies data from the given {@link ValueObject} into a {@link DataObject}.
     * 
     * @param data {@link ValueObject} to convert
     * @return fully populated {@link DataObject}
     * 
     */
    @Override
    public EplDrugDrugInteractionDo toDataObject(DrugDrugInteractionVo data) {

        EplDrugDrugInteractionDo drugInteraction = new EplDrugDrugInteractionDo();

        if(null!=data.getId()){
            drugInteraction.setEplDrugDrugInteractionId(new Long(data.getId()));
        }


        drugInteraction.setInteractionName(data.convertToInteractionName(data));    


        if (null!=data.getDrugDrugInteractionIen() ) {
            drugInteraction.setVistaIen(new Long(data.getDrugDrugInteractionIen()));
        }

        drugInteraction.setDrugIngredient1(findIngredient(data.getIngredient1().getValue()));
        drugInteraction.setDrugIngredient2(findIngredient(data.getIngredient2().getValue()));
        drugInteraction.setSeverity(findSeverity(data.getDrugSeverity().getValue()));

        drugInteraction.setCreatedBy(data.getCreatedBy());
        drugInteraction.setCreatedDtm(data.getCreatedDate());
        drugInteraction.setLastModifiedBy(data.getModifiedBy());
        drugInteraction.setLastModifiedDtm(data.getModifiedDate());

        drugInteraction.setInactivationDate(data.getInactivationDate());

        // some default values for now.
        drugInteraction.setNationallyEntered(true);
        drugInteraction.setLocallyEdited(false);
        drugInteraction.setTotalIndexes(new Long(0));


        return drugInteraction;
    }





    /**
     * Fully copies data from the given EplDrugDrugInteractionDo into a DrugDrugInteractionVo.     
     * 
     * @param data EplDrugDrugInteractionDo to convert
     * @return fully populated DrugDrugInteractionVo
     * 
     */   
    @Override
    public DrugDrugInteractionVo toValueObject(EplDrugDrugInteractionDo data) {

        DrugDrugInteractionVo drugInteraction = new DrugDrugInteractionVo();        

        drugInteraction.setId(data.getEplDrugDrugInteractionId().toString());

        PrimaryIngredientVo ingredient1 = primaryIngredientConverter.toValueObject(data.getDrugIngredient1());

        PrimaryIngredientVo ingredient2 = primaryIngredientConverter.toValueObject(data.getDrugIngredient2());

        DrugSeverityVo drugSeverityVo = drugSeverityConverter.toValueObject(data.getSeverity());        


        if (null!=data.getVistaIen()) {
            drugInteraction.setDrugDrugInteractionIen(data.getVistaIen().toString());
        }       


        drugInteraction.setValue(data.getInteractionName());
        drugInteraction.setDrugDrugInteractionDisplay(data.getInteractionName());

        drugInteraction.setItemStatus(data.getInactivationDate()!=null?ItemStatus.valueOf("INACTIVE"):ItemStatus.valueOf("ACTIVE"));
        drugInteraction.setRequestItemStatus(RequestItemStatus.valueOf("APPROVED"));

        drugInteraction.setIngredient1(ingredient1);
        drugInteraction.setIngredient2(ingredient2);
        drugInteraction.setDrugSeverity(drugSeverityVo);        


        drugInteraction.setCreatedBy(data.getCreatedBy());
        drugInteraction.setCreatedDate(data.getCreatedDtm());
        drugInteraction.setModifiedBy(data.getLastModifiedBy());
        drugInteraction.setModifiedDate(data.getLastModifiedDtm());        


        drugInteraction.setInactivationDate(data.getInactivationDate());


        drugInteraction.setNationallyEntered(String.valueOf(data.getNationallyEntered()));
        drugInteraction.setLocallyEdited(String.valueOf(data.getLocallyEdited()));
        drugInteraction.setTotalIndexes(data.getTotalIndexes().toString());

        return drugInteraction;
    }


    private EplIngredientDo findIngredient(String name) {
        return getEplIngredientDao().findIngredientByName(name);
    }

    private EplDdiSeverityDo findSeverity(String name) {
        return getEplDdiSeverityDao().findByName(name);

    }

    /**
     * @return the eplIngredientDao
     */
    public EplIngredientDao getEplIngredientDao() {
        return eplIngredientDao;
    }

    /**
     * @param eplIngredientDao the eplIngredientDao to set
     */
    @Autowired(required = true)
    public void setEplIngredientDao(EplIngredientDao eplIngredientDao) {
        this.eplIngredientDao = eplIngredientDao;
    }

    /**
     * @return the eplDdiSeverityDao
     */
    public EplDdiSeverityDao getEplDdiSeverityDao() {
        return eplDdiSeverityDao;
    }

    /**
     * @param eplDdiSeverityDao the eplDdiSeverityDao to set
     */
    @Autowired
    public void setEplDdiSeverityDao(EplDdiSeverityDao eplDdiSeverityDao) {
        this.eplDdiSeverityDao = eplDdiSeverityDao;
    }



    /**
     * @return the primaryIngredientConverter
     */
    public PrimaryIngredientConverter getPrimaryIngredientConverter() {
        return primaryIngredientConverter;
    }


    /**
     * @param primaryIngredientConverter the primaryIngredientConverter to set
     */
    @Autowired
    public void setPrimaryIngredientConverter(PrimaryIngredientConverter primaryIngredientConverter) {
        this.primaryIngredientConverter = primaryIngredientConverter;
    }


    /**
     * @return the drugSeverityConverter
     */
    public DrugSeverityConverter getDrugSeverityConverter() {
        return drugSeverityConverter;
    }



    /**
     * @param drugSeverityConverter the drugSeverityConverter to set
     */
    @Autowired
    public void setDrugSeverityConverter(DrugSeverityConverter drugSeverityConverter) {
        this.drugSeverityConverter = drugSeverityConverter;
    }

    /**
     * Check for new/modified fields in the difference set.
     * 
     * @param fields set of defined fields
     * @param differences difference set
     * @param itemAction add/modify/inactivate
     * @return true if has modified fields
     */
    public static boolean hasNewOrModifiedFields(Set<FieldKey> fields, Map<FieldKey, Difference> differences,
        ItemAction itemAction) {
        if (ItemAction.ADD.equals(itemAction)) {
            return true;
        } else { // check for modified VistA field
            for (FieldKey key : fields) {
                if (differences.containsKey(key)) {
                    return true;
                }
            }
        }

        return false;
    }


    /**
     * Check if specific field was modified in difference set.
     * 
     * @param field key
     * @param differences difference set
     * @return true if field was modified
     */
    public static boolean fieldWasModified(FieldKey key, Collection<Difference> differences) {

        for (Difference diff : differences) {
            if ( diff.getFieldKey().equals(key)) {
                return true;        // Get out as soon as we find the key
            }
        }

        return false;
    }








}
