package gov.va.med.pharmacy.peps.domain.common.capability.impl;


import gov.va.med.pharmacy.peps.common.exception.DuplicateItemException;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.PackageSizeVo;
import gov.va.med.pharmacy.peps.common.vo.PackageTypeVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.capability.PackageSizeDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.dao.DataAccessObject;
import gov.va.med.pharmacy.peps.domain.common.dao.EplPackageSizeDao;
import gov.va.med.pharmacy.peps.domain.common.model.EplPackageSizeDo;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.Converter;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.PackageSizeConverter;
import gov.va.med.pharmacy.peps.service.common.capability.NdfFileSyncCapability;
import gov.va.med.pharmacy.peps.service.common.update.NdfFileSyncProcessor;
import gov.va.med.pharmacy.peps.service.common.update.NdfFileSyncProcessorFactory;

import org.hibernate.Criteria;
import org.hibernate.criterion.Restrictions;


/**
 * Perform CRUD operations on package types
 */
public class PackageSizeDomainCapabilityImpl extends ManagedDataDomainCapabilityImpl<PackageSizeVo, EplPackageSizeDo>
implements PackageSizeDomainCapability {

    private EplPackageSizeDao eplPackageSizeDao;
    private PackageSizeConverter packageSizeConverter;
    private NdfFileSyncCapability ndfFileSyncCapability;

    /**
     * Insert the given {@link ManagedItemVo} with a duplicate check.
     * 
     * @param managedItem {@link ManagedItemVo}
     * @param user {@link UserVo} performing the action
     * @return {@link ManagedItemVo} inserted with new ID
     * @throws DuplicateItemException if item already exists by uniqueness fields
     */
    @Override
    public PackageSizeVo create(PackageSizeVo managedItem, UserVo user) throws DuplicateItemException {
        Long id = getSeqNumDomainCapability().generateIen("EPL_PACKAGE_SIZES",user);
        managedItem.setNdfPackageSizeIen(id.toString());
        
        EplPackageSizeDo packageSizeDo = packageSizeConverter.convert(managedItem);       
        EplPackageSizeDo returnedDo = eplPackageSizeDao.insert(packageSizeDo, user);
        PackageSizeVo returnedVo = packageSizeConverter.convert(returnedDo);
        NdfFileSyncProcessor processor = NdfFileSyncProcessorFactory.getProcessor(returnedVo);
        processor.setNdfFileSyncCapability(ndfFileSyncCapability);
        processor.processNew(returnedVo, user);
        
        return returnedVo;
    }
    
    /**
     * retrieveBySize
     * @param Double packageSize
     * @return PackageSizeVo
     */

    public PackageSizeVo retrieveByPackageSize(Double packageSize) {

        EplPackageSizeDo returnedDos = eplPackageSizeDao.retrieveByPackageSize(packageSize);

        return packageSizeConverter.convert(returnedDos);
    }
    
    
    /**
     * retrieveById
     * @param Long eplid
     * @return PackageSizeVo
     */

    public PackageSizeVo retrieveById(Long eplId) {

        EplPackageSizeDo returnedDo = eplPackageSizeDao.retrieveById(eplId);
       
        return packageSizeConverter.convert(returnedDo);
    }

    /**
     * Create Hibernate {@link Criteria} that will find an item by its uniqueness fields.
     * 
     * @param item {@link PackageTypeVo} for which to create uniqueness {@link Criteria}
     * @return {@link Criteria}
     */
    @Override
    protected Criteria createUniquenessCriteria(PackageSizeVo item) {
        Criteria criteria = getDataAccessObject().getCriteria();
        criteria.add(Restrictions.ilike(EplPackageSizeDo.PACKAGE_SIZE, item.getPackageSize()));

        return criteria;
    }

    /**
     * Get the {@link DataAccessObject} that this capability uses.
     * 
     * @return {@link DataAccessObject}
     */
    public DataAccessObject getDataAccessObject() {
        return eplPackageSizeDao;
    }

    /**
     * setEplPackageSizeDao
     * @param eplPackageSizeDao EplPackageSizeDao property
     */
    public void setEplPackageSizeDao(EplPackageSizeDao eplPackageSizeDao) {
        this.eplPackageSizeDao = eplPackageSizeDao;
    }

    /**
     * setPackageSizeConverter
     * @param packageSizeConverter packageSizeConverter property
     */
    public void setPackageSizeConverter(PackageSizeConverter packageSizeConverter) {
        this.packageSizeConverter = packageSizeConverter;
    }
    
    /**
     * setNdfFileSyncCapability
     * @param capability the NdfFileSyncCapability bean
     */
    public void setNdfFileSyncCapability(NdfFileSyncCapability capability) {
        this.ndfFileSyncCapability = capability;
    }
    
    /**
     * Return the {@link Converter} instance for this capability.
     * 
     * @return {@link Converter}
     */
    public PackageSizeConverter getConverter() {
        return packageSizeConverter;
    }
}
