package gov.va.med.pharmacy.peps.domain.common.capability.impl;

import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.NdfUpdateFileVo;
import gov.va.med.pharmacy.peps.common.vo.PaginatedList;
import gov.va.med.pharmacy.peps.common.vo.SortOrder;
import gov.va.med.pharmacy.peps.common.vo.UpdateSearchCriteria;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.capability.NdfUpdateFileMgtDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.dao.EplNdfUpdateFileDao;
import gov.va.med.pharmacy.peps.domain.common.model.EplNdfUpdateFileDo;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.NdfUpdateFileConverter;

import java.util.ArrayList;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

public class NdfUpdateFileMgtDomainCapabilityImpl
implements NdfUpdateFileMgtDomainCapability {

    private EplNdfUpdateFileDao eplNdfUpdateFileDao;

    private NdfUpdateFileConverter ndfUpdateFileConverter;

    /**
     * Create the list of values of the NdfUpdateFile.
     * 
     * @return NdfUpdateFileVo
     */
    public List<NdfUpdateFileVo> retrieveAll() {
        Criteria criteria = eplNdfUpdateFileDao.getCriteria();
        criteria.addOrder(Order.desc(EplNdfUpdateFileDo.CREATED_DATE_TIME));
        criteria.add(Restrictions.not(Restrictions.eq("eplStatus.statusId",
                Long.valueOf(5))));
        criteria.setMaxResults(10);
        List<EplNdfUpdateFileDo> values = criteria.list();
        List<NdfUpdateFileVo> returnedValues = ndfUpdateFileConverter
                .convert(values);

        return returnedValues;
    }


    @SuppressWarnings("unchecked")
    public PaginatedList<NdfUpdateFileVo> retrievePaged(UpdateSearchCriteria searchCriteria) {
        Criteria criteria = createCriteriaObject(searchCriteria);
        int fullSize = criteria.list().size();

        criteria.setMaxResults(searchCriteria.getPageSize());
        criteria.setFirstResult(searchCriteria.getStartRow());

        List<EplNdfUpdateFileDo> values = criteria.list();
        List<NdfUpdateFileVo> returnedValues = getNdfUpdateFileConverter().convert(
                values);
        return new PaginatedList<NdfUpdateFileVo>(returnedValues, fullSize,
                FieldKey.CREATED_DTM, SortOrder.DESCENDING,
                searchCriteria.getStartRow(), searchCriteria.getPageSize());

    }

    private Criteria createCriteriaObject(UpdateSearchCriteria searchCriteria) {
        Criteria criteria = eplNdfUpdateFileDao.getCriteria();
        criteria.addOrder(Order.desc(EplNdfUpdateFileDo.CREATED_DATE_TIME));
        // Status Id of 5 is the current file and will not show in the history
        // section
        criteria.add(Restrictions.not(Restrictions.eq("eplStatus.statusId",
                Long.valueOf(5))));
        if (searchCriteria.getBeginningDateRange() != null
                && searchCriteria.getEndingDateRange() != null) {
            criteria.add(Restrictions.between("statusModifiedDtm",
                    searchCriteria.getBeginningDateRange(),
                    searchCriteria.getEndingDateRange()));
        } else {
            if (searchCriteria.getBeginningDateRange() != null) {
                criteria.add(Restrictions.ge("statusModifiedDtm",
                        searchCriteria.getBeginningDateRange()));
            }
            if (searchCriteria.getEndingDateRange() != null) {
                criteria.add(Restrictions.le("statusModifiedDtm",
                        searchCriteria.getEndingDateRange()));
            }
        }

        if (searchCriteria.getStatusId() != null) {
            criteria.add(Restrictions.eq("eplStatus.statusId",
                    searchCriteria.getStatusId()));
        }

        return criteria;
    }

    /**
     * Create the list of values of the pending NdfUpdateFile.
     * 
     * @return NdfUpdateFileVo
     */
    public List<NdfUpdateFileVo> retrievePending() {
        Criteria criteria = eplNdfUpdateFileDao.getCriteria();
        criteria.addOrder(Order.desc(EplNdfUpdateFileDo.CREATED_DATE_TIME));
        criteria.add(Restrictions.eq("eplStatus.statusId", Long.valueOf(5)));
        List<EplNdfUpdateFileDo> values = criteria.list();
        List<NdfUpdateFileVo> returnedValues = ndfUpdateFileConverter
                .convert(values);

        return returnedValues;
    }

    /**
     * Retrieve a NdfUpdateFileVo by id.
     *
     * @param id Long the id to search by
     * @return NdfUpdateFileVo or null if the id is null or no NdfUpdateFileVo found
     */
    @Override
    public NdfUpdateFileVo retrieveById(Long id) {

        if (id == null) {
            return null;
        }

        EplNdfUpdateFileDo eplDo = eplNdfUpdateFileDao.retrieve(id);

        if (eplDo == null) {
            return null;
        }

        return ndfUpdateFileConverter.convert(eplDo);
    }

    /**
     * Retrieves a list of NdfUpdateFileVo with status that has not reached completion of National Install.
     * Realistically, due to other rules in the application, there should be max one record returned.  
     * 
     * @return NdfUpdateFileVo
     */
    public List<NdfUpdateFileVo> retrieveInitiated() {
        Criteria criteria = eplNdfUpdateFileDao.getCriteria();
        criteria.addOrder(Order.desc(EplNdfUpdateFileDo.CREATED_DATE_TIME));
        List<Long> statuses = new ArrayList<Long>();
        statuses.add(new Long("1"));
        statuses.add(new Long("2"));
        statuses.add(new Long("3"));
        statuses.add(new Long("4"));
        criteria.add(Restrictions.in("eplStatus.statusId", statuses));
        List<EplNdfUpdateFileDo> values = criteria.list();
        List<NdfUpdateFileVo> returnedValues = ndfUpdateFileConverter
                .convert(values);

        return returnedValues;
    }

    /**
     * Return only the last completed NdfUpdateFile.
     * 
     * @return NdfUpdateFileVo
     */
    public NdfUpdateFileVo retrieveLastTranmittedToProduction() {

        Query query = eplNdfUpdateFileDao.getCurrentSession().getNamedQuery(
                "retrieveLastTranmittedToProduction");
        EplNdfUpdateFileDo value = (EplNdfUpdateFileDo) query.uniqueResult();
        NdfUpdateFileVo returnedValue = ndfUpdateFileConverter.convert(value);

        return returnedValue;
    }

    /**
     * Return current status of the NdfUpdateFile
     * 
     * @return NdfUpdateFileVo
     */
    public NdfUpdateFileVo retrieveCurrentStatus() {

        Query query = eplNdfUpdateFileDao.getCurrentSession().getNamedQuery(
                "retrieveCurrentStatus");
        EplNdfUpdateFileDo value = (EplNdfUpdateFileDo) query.uniqueResult();
        NdfUpdateFileVo returnedValue = ndfUpdateFileConverter.convert(value);

        return returnedValue;
    }

    /**
     * Return the max ID for approved NdcUpdateFiles
     * 
     * @return NdfUpdateFileVo
     */
    public NdfUpdateFileVo retrieveNextSeqNumber() {
        Criteria criteria = eplNdfUpdateFileDao.getCriteria();
        criteria.add(Restrictions.eq("eplStatus.statusId", Long.valueOf(7)));
        criteria.setProjection(Projections
                .max(EplNdfUpdateFileDo.STATUS_EPL_ID));
        List<EplNdfUpdateFileDo> values = criteria.list();
        List<NdfUpdateFileVo> returnedValues = ndfUpdateFileConverter
                .convert(values);

        return returnedValues.get(0);
    }

    /**
     * Update the given NdfUpdateFileVo.
     * 
     * @param updatedFile
     *            ndf file
     * @param user
     *            {@link UserVo} performing the action
     * 
     */
    public void update(NdfUpdateFileVo updatedFile, UserVo user) {
        EplNdfUpdateFileDo eplNdfUpdateFileDo = ndfUpdateFileConverter
                .convert(updatedFile);
        eplNdfUpdateFileDao.update(eplNdfUpdateFileDo, user);
    }

    /**
     * Update the given NdfUpdateFileVo.
     * 
     * @param initiatedFile new NdfUpdateFileVo to insert
     * @param user {@link UserVo} performing the action
     * 
     */
    public NdfUpdateFileVo insert(NdfUpdateFileVo initiatedFile, UserVo user) {
        EplNdfUpdateFileDo eplNdfUpdateFileDo = ndfUpdateFileConverter
                .convert(initiatedFile);
        EplNdfUpdateFileDo value = eplNdfUpdateFileDao.insert(
                eplNdfUpdateFileDo, user);
        NdfUpdateFileVo returnedValue = ndfUpdateFileConverter.convert(value);

        return returnedValue;
    }

    /**
     * retrieve record in initiated status and return result.
     * 
     * @return NdfUpdateFileVo
     */
    public NdfUpdateFileVo checkInitiatedFile() {

        Query query = eplNdfUpdateFileDao.getCurrentSession().getNamedQuery(
                "checkInitiatedFile");

        EplNdfUpdateFileDo value = (EplNdfUpdateFileDo) query.uniqueResult();
        NdfUpdateFileVo returnedValue = ndfUpdateFileConverter.convert(value);

        return returnedValue;
    }

    /**
     * Gets the epl ndf update file dao.
     *
     * @return the epl ndf update file dao
     */
    public EplNdfUpdateFileDao getEplNdfUpdateFileDao() {
        return eplNdfUpdateFileDao;
    }

    /**
     * Sets the epl ndf update file dao.
     *
     * @param eplNdfUpdateFileDao the new epl ndf update file dao
     */
    public void setEplNdfUpdateFileDao(EplNdfUpdateFileDao eplNdfUpdateFileDao) {
        this.eplNdfUpdateFileDao = eplNdfUpdateFileDao;
    }

    /**
     * Gets the ndf update file converter.
     *
     * @return the ndf update file converter
     */
    public NdfUpdateFileConverter getNdfUpdateFileConverter() {
        return ndfUpdateFileConverter;
    }

    /**
     * Sets the ndf update file converter.
     *
     * @param ndfUpdateFileConverter the new ndf update file converter
     */
    public void setNdfUpdateFileConverter(
            NdfUpdateFileConverter ndfUpdateFileConverter) {
        this.ndfUpdateFileConverter = ndfUpdateFileConverter;
    }

}
