/**
 * Source file created in 2007 by Southwest Research Institute
 */


package gov.va.med.pharmacy.peps.domain.common.capability.impl;


import gov.va.med.pharmacy.peps.common.exception.DuplicateItemException;
import gov.va.med.pharmacy.peps.common.exception.ItemNotFoundException;
import gov.va.med.pharmacy.peps.common.exception.ValidationException;
import gov.va.med.pharmacy.peps.common.utility.PPSConstants;
import gov.va.med.pharmacy.peps.common.vo.DispenseUnitVo;
import gov.va.med.pharmacy.peps.common.vo.FieldKey;
import gov.va.med.pharmacy.peps.common.vo.ManagedItemVo;
import gov.va.med.pharmacy.peps.common.vo.UserVo;
import gov.va.med.pharmacy.peps.domain.common.capability.DispenseUnitDomainCapability;
import gov.va.med.pharmacy.peps.domain.common.dao.DataAccessObject;
import gov.va.med.pharmacy.peps.domain.common.dao.EplProductDao;
import gov.va.med.pharmacy.peps.domain.common.dao.EplVaDispenseUnitDao;
import gov.va.med.pharmacy.peps.domain.common.model.EplDrugUnitDo;
import gov.va.med.pharmacy.peps.domain.common.model.EplProductDo;
import gov.va.med.pharmacy.peps.domain.common.model.EplVaDispenseUnitDo;
import gov.va.med.pharmacy.peps.domain.common.utility.SchemaUtility;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.Converter;
import gov.va.med.pharmacy.peps.domain.common.utility.converter.DispenseUnitConverter;
import gov.va.med.pharmacy.peps.domain.common.utility.querybuilder.HqlBuilder;

import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;


/**
 * Perform CRUD operations on dosage forms
 */
public class DispenseUnitDomainCapabilityImpl extends ManagedDataDomainCapabilityImpl<DispenseUnitVo, EplVaDispenseUnitDo>
    implements DispenseUnitDomainCapability {

    private EplVaDispenseUnitDao eplVaDispenseUnitDao;
    private EplProductDao eplProductDao;
    private DispenseUnitConverter dispenseUnitConverter;

    /**
     * Insert the given {@link ManagedItemVo} with a duplicate check.
     * 
     * @param managedItem {@link ManagedItemVo}
     * @param user {@link UserVo} performing the action
     * @return {@link ManagedItemVo} inserted with new ID
     * @throws DuplicateItemException if item already exists by uniqueness fields
     */
    @Override
    public DispenseUnitVo create(DispenseUnitVo managedItem, UserVo user) throws DuplicateItemException {
        Long id = getSeqNumDomainCapability().generateIen("EPL_VA_DISPENSE_UNITS", user);
        managedItem.setDispenseUnitIen(id.toString());
        return super.create(managedItem, user);
    }
    
    /**
     * deleteItem
     * @param dispenseUnitVo dispenseUnitVo
     * @throws ValidationException ValidationException
     */
    @Override
    public void deleteItem(DispenseUnitVo dispenseUnitVo) throws ValidationException {
        List<EplVaDispenseUnitDo> dispenseUnitDo = getDataAccessObject().retrieve(EplDrugUnitDo.EPL_ID,
                Long.valueOf(dispenseUnitVo.getId()));

        if (dispenseUnitDo.size() == 1) {
            if (dispenseUnitDo.get(0).getNdfDispenseunitIen() == null) {
                try {
                    checkForActiveDependencies(dispenseUnitVo, null);
                } catch (Exception e) {
                    throw new ValidationException(ValidationException.CANNOT_DELETE,
                        dispenseUnitVo.getValue(), " because another item is using it.");
                }

                getDataAccessObject().delete(dispenseUnitDo.get(0));
            } else {
                throw new ValidationException(ValidationException.CANNOT_DELETE,
                    dispenseUnitVo.getValue(), " because it has already synched with NDF.");
            }
        }
    }

    /**
     * dispenseUnitVo cannot be in-activated if it has an active product.
     *  
     * @param dispenseUnitVo dispenseUnitVo
     * @param user {@link UserVo} performing the action
     * @throws ValidationException ValidationException
     */
    @Override
    public void checkForActiveDependencies(DispenseUnitVo dispenseUnitVo, UserVo user) throws ValidationException {

        StringBuffer query = new StringBuffer();
        query.append(HqlBuilder.create("SELECT  item FROM ").append(EplProductDo.class)
                .append(PPSConstants.ITEM));
        String whereClause = " WHERE DISPENSE_UNIT_ID_FK = " + dispenseUnitVo.getId() + " AND ITEM_STATUS LIKE 'ACTIVE'";
        query.append(whereClause);

        List<EplProductDo> products = eplProductDao.executeHqlQuery(query.toString());

        if (products == null || products.size() == 0) {
            return;
        }

        // throw a validation exception if the dispenseUnit is Inactive.
        throw new ValidationException(ValidationException.INACTIVATE_DISPENSEUNIT,
                                      products.size(), products.get(0).getVaProductName());

    }

    /**
     * Create Hibernate {@link Criteria} that will find an item by its uniqueness fields.
     * 
     * @param item {@link DispenseUnitVo} for which to create uniqueness {@link Criteria}
     * @return {@link Criteria}
     */
    @Override
    protected Criteria createUniquenessCriteria(DispenseUnitVo item) {
        Criteria criteria = getDataAccessObject().getCriteria();
        criteria.add(Restrictions.ilike(EplVaDispenseUnitDo.DISPENSE_UNIT_NAME, item.getValue()));

        return criteria;
    }

    /**
     * Get the {@link DataAccessObject} that this capability uses.
     * 
     * @return {@link DataAccessObject}
     */
    public DataAccessObject getDataAccessObject() {
        return eplVaDispenseUnitDao;
    }

    /**
     * Return the {@link Converter} instance for this capability.
     * 
     * @return {@link Converter}
     */
    public DispenseUnitConverter getConverter() {
        return dispenseUnitConverter;
    }

    /**
     * setEplVaDispenseUnitDao
     * @param eplVaDispenseUnitDao eplVaDispenseUnitDao property
     */
    public void setEplVaDispenseUnitDao(EplVaDispenseUnitDao eplVaDispenseUnitDao) {
        this.eplVaDispenseUnitDao = eplVaDispenseUnitDao;
    }

    /**
     * setDispenseUnitConverter
     * @param dispenseUnitConverter dispenseUnitConverter property
     */
    public void setDispenseUnitConverter(DispenseUnitConverter dispenseUnitConverter) {
        this.dispenseUnitConverter = dispenseUnitConverter;
    }

    /**
     * setEplProductDao
     * @param eplProductDao eplProductDao property
     */
    public void setEplProductDao(EplProductDao eplProductDao) {
        this.eplProductDao = eplProductDao;
    }

    /**
     * Delete associated domain from data base if IEN is not assigned and other objects are not dependent on this object.
     * 
     * @param dispenseUnitVo to delete
     * @param user requesting the delete
     * 
     * @throws ValidationException to validate the IEN and dependencies
     * 
     */
    @Override
    public void deletePending(DispenseUnitVo dispenseUnitVo, UserVo user) throws ValidationException {

        StringBuffer query = new StringBuffer();
        query.append(HqlBuilder.create("SELECT  item FROM ").append(EplProductDo.class)
                .append(PPSConstants.ITEM));
        String whereClause = " WHERE DISPENSE_UNIT_ID_FK = " + dispenseUnitVo.getId();
        query.append(whereClause);

        List<EplProductDo> products = eplProductDao.executeHqlQuery(query.toString());

        if (products == null || products.size() == 0) {
            return;
        }

        throw new ValidationException(ValidationException.INACTIVATE_DISPENSEUNIT,
                                      products.size(), products.get(0).getVaProductName());

    }
    
    
    @Override
    public long getRevisionNumber(String id) throws ItemNotFoundException {
        Criteria criteria = getDataAccessObject().getCriteria();
        criteria.setCacheable(true).setCacheRegion(PPSConstants.NATIONAL_DATA_CACHE);
        String idProperty = SchemaUtility.getPropertyName(getDataObjectClass(), FieldKey.ID);
        criteria.add(Restrictions.eq(idProperty, getDataAccessObject().convertId(id)));
        String propertyName = SchemaUtility.getPropertyName(getDataObjectClass(), FieldKey.REVISION_NUMBER);
        criteria.setProjection(Projections.property(propertyName));

        Long revisionNumber = (Long) criteria.uniqueResult();

        if (revisionNumber == null) {
            throw new ItemNotFoundException(ItemNotFoundException.ITEM_NOT_FOUND, id);
        }

        return revisionNumber.longValue();
    }


}
